##############################################################################
# Funções para análise de expressão diferencial
#------------------------------------------------------------------------------


#------------------------------------------------------------------------------
# -> Funções


diffExpr <- function(seriesMatrix, expGroup, ctrlGroup, doLog2, adjMethod = "BH")
{
  library("Biobase")
  library("limma")
  compMatrix <- seriesMatrix[,unique(c(expGroup, ctrlGroup))]
  if (doLog2)
  {
    compMatrix <- log2(compMatrix)
  }
  
  rowCount = nrow(compMatrix)
  plotDisplayVals <- data.frame(ID = rownames(compMatrix), exp = rep(NA, rowCount), ctrl = rep(NA, rowCount),
                                logFC = rep(NA, rowCount), nlogpval = rep(NA, rowCount), row.names = rownames(compMatrix))
  
  # Comparação normal para casos com valor-p
  if(ncol(compMatrix) > 2) {
  
    ##### Matriz com apenas experimento e controle:
    
    mdesign <- unique(c(expGroup, ctrlGroup))
    mdesign[mdesign %in% expGroup] <- 1
    mdesign[mdesign %in% ctrlGroup] <- 2
    
    
    # Faz o design dos grupos baseado na matriz de design
    designGrupos<-model.matrix(~ 0+factor(mdesign))
    # Nomes dos grupos
    colnames(designGrupos) <- c("g1","g2")
    # Nomes das amostras correspondente aos grupos
    rownames(designGrupos)<-unique(c(expGroup, ctrlGroup))
    # Refaço o design de grupos para permitir overlaps de amostras
    designGrupos[,1:2] <- 0
    designGrupos[rownames(designGrupos) %in% ctrlGroup,1] <- 1
    designGrupos[rownames(designGrupos) %in% expGroup,2] <- 1
    # Constrói um modelo linear a partir do design de grupos
    agrupaReplicatas<-lmFit(compMatrix, designGrupos)
    # Faz o contraste da matriz a partir do modelo linear
    matrizContraste<-makeContrasts(g2-g1, levels=designGrupos)
    # Agrupa o modelo em contraste
    agrupaContraste<-contrasts.fit(agrupaReplicatas, matrizContraste)
    # Obtêm os genes diferencialmente expressos
    difExpressas<-eBayes(agrupaContraste)
    
    results<-topTable(difExpressas,coef=1,number=1000000, p.value = 1, sort.by = "none", adjust.method = adjMethod)
    results<-cbind(ID = rownames(results), results)
    if (doLog2){
      plotDisplayVals$exp <- 2^(agrupaReplicatas$coefficients[,"g2"])
      plotDisplayVals$ctrl <- 2^(agrupaReplicatas$coefficients[,"g1"])
    } else {
      plotDisplayVals$exp <- agrupaReplicatas$coefficients[,"g2"]
      plotDisplayVals$ctrl <- agrupaReplicatas$coefficients[,"g1"]
    }
    plotDisplayVals$logFC <- results$logFC
    plotDisplayVals$nlogpval <- -log10(results$adj.P.Val)
  } else if (ncol(compMatrix) == 2) { # Caso seja uma comparação de duas amostras apenas:
    results <- data.frame(ID = rownames(compMatrix), logFC = rep(NA, rowCount), AveExpr = rep(NA, rowCount), row.names = rownames(compMatrix))
    plotDisplayVals$exp <- compMatrix[, 1]
    plotDisplayVals$ctrl <- compMatrix[, 2]
    for (i in 1:rowCount){
      results[i,"logFC"] <- log2(compMatrix[i, 1]) - log2(compMatrix[i, 2])
      results[i, "AveExpr"] <- (compMatrix[i, 1] + compMatrix[i, 2])/2
    }
  }
  if (adjMethod == 'none' && 'adj.P.Val' %in% names(results))
    results = results[, !names(results) %in% c('adj.P.Val')]
  resultList <- list()
  resultList[["displayvalues"]] <- plotDisplayVals # Slots: exp, ctrl, logFC, nlogpval
  resultList[["results"]] <- results # Matriz de resultados que conhecemos
  #resultList[["difexpressas"]] <- difExpressas # Deprecated
  
  return(resultList)
}

adjustPlatform <- function(seriesMatrix, platformMatrix)
{
  if (nrow(seriesMatrix) != nrow(platformMatrix))
  {
    newPlat = as.data.frame(matrix('', nrow = nrow(seriesMatrix), ncol = ncol(platformMatrix),
                                   dimnames = list(rownames(seriesMatrix), colnames(platformMatrix))))
    comRows = intersect(rownames(seriesMatrix), rownames(platformMatrix))
    newPlat = as.matrix(newPlat)
    newPlat[comRows,] = as.matrix(platformMatrix[comRows,])
    newPlat = as.data.frame(newPlat)
    return(newPlat)
  } else {
    return(platformMatrix)
  }
}


