#### Funcões para o ArrayQualityMetrics



# Bibliotecas
library("affy")
library("GEOquery")
library("arrayQualityMetrics")
library("setRNG")
library("genefilter")
library("lattice")
library("SVGAnnotation")
library("Hmisc")
library("vsn")
library("hwriter")
library("gridSVG")
library("Cairo")
library("latticeExtra")
library("grDevices")


# Começando pelas variáveis da função arrayQualityMetrics


force <- TRUE
intgroup <- character(0)
reporttitle <- "Results from qMetrics"
spatial <- TRUE

  
######################################
# Da Função prepdata:

prepdatax <- function (expressionset, intgroup, do.logtransform) 
{
  conversions = c(RGList = "NChannelSet")
  for (i in seq_along(conversions)) {
    if (is(expressionset, names(conversions)[i])) {
      expressionset = try(as(expressionset, conversions[i]))
      if (is(expressionset, "try-error")) {
        stop(sprintf("The argument 'expressionset' is of class '%s', and its automatic conversion into '%s' failed. Please try to convert it manually, or contact the creator of that object.\n", 
                     names(conversions)[i], conversions[i]))
      }
      else {
        break
      }
    }
  }
  x = environment(arrayQualityMetrics)$platformspecific(expressionset, intgroup, do.logtransform)
  if (!all(intgroup %in% colnames(x$pData))) 
    stop("all elements of 'intgroup' should match column names of 'pData(expressionset)'.")
  x = append(x, list(numArrays = ncol(x$M), intgroup = intgroup, 
                     do.logtransform = do.logtransform))
  x = append(x, environment(arrayQualityMetrics)$intgroupColors(x))
  return(x)
}



#######################################
# Da Função outliers:

outliersx <- function (exprs, method = c("KS", "sum", "upperquartile")) {
  s = switch(method, KS = {
    fx = ecdf(as.vector(exprs))
    suppressWarnings(apply(exprs, 2, function(v) ks.test(v, 
                                                         y = fx, alternative = "two.sided")$statistic))
  }, sum = {
    colSums(exprs, na.rm = TRUE)
  }, upperquartile = {
    apply(exprs, 2, quantile, na.rm = TRUE, probs = 0.75)
  }, stop(sprintf("Invalid method '%s'", method)))
  fo = boxplotOutliers(s)
  new("outlierDetection", statistic = s, threshold = fo$threshold, 
      which = fo$which)
}



######################################
# Da Função aqm.heatmap:

aqm.heatmapx <- function (x, rgbi = c(1,1,1),rgbf = c(1,0,0)) {
  colorRange = rgb(seq(rgbi[1], rgbf[1], l = 256), seq(rgbi[2], rgbf[2], l = 256), 
                   seq(rgbi[3], rgbf[3], l = 256)) # Aqui que vai as cores!
  m = dist2(x$M)
  out = outliersx(m, method = "sum")
  out@description = c("sum of distances to other arrays <i>S<sub>a</sub></i>", 
                      "data-driven")
  dend = as.dendrogram(hclust(as.dist(m), method = "single"))
  ord = order.dendrogram(dend)
  colnames(m) = rownames(m) = paste0(ifelse(seq_len(x$numArrays) %in% 
                                              out@which, "* ", ""), seq_len(x$numArrays))
  haveDend = (ncol(m) <= (environment(arrayQualityMetrics)$arrayQualityMetricsGlobalParameters)$maxNumberOfArraysForDrawingDendrogram)
  if (haveDend) {
    theLegend = list(right = list(fun = dendrogramGrob, args = list(x = dend, 
                                                                    side = "right")))
    fillOrd = seq_len(x$numArrays)
  }
  else {
    theLegend = NULL
    fillOrd = ord
  }
  maxNrColors = 0
  ng = length(x$intgroup)
  if (ng > 0) {
    palettes = c("Set1", "Set2", "Set3", "Accent", "Dark2", 
                 "Paired", "Pastel1", "Pastel2")
    stopifnot(all(palettes %in% rownames(brewer.pal.info)))
    palettes = rep(palettes, ceiling(ng/length(palettes)))
    key = rects = vector(mode = "list", length = ng)
    names(rects) = rep("rect", ng)
    for (i in seq_len(ng)) {
      colors = brewer.pal(brewer.pal.info[palettes[i], 
                                          "maxcolors"], palettes[i])
      fac = factor(x$pData[[x$intgroup[i]]])
      fac = maximumLevels(fac, n = length(colors))
      colors = colors[seq_len(nlevels(fac))]
      key[[i]] = list(rect = list(col = colors), text = list(levels(fac)))
      rects[[i]] = list(col = "transparent", fill = colors[as.integer(fac)[fillOrd]], 
                        type = "rectangle")
      if (length(colors) > maxNrColors) 
        maxNrColors = length(colors)
    }
    key = unlist(key, recursive = FALSE)
    key$rep = FALSE
    thekey = draw.key(key = key)
    if (haveDend) {
      theLegend$right$args = append(theLegend$right$args, 
                                    list(size.add = 1, add = rects))
    }
    else {
      lay = grid.layout(nrow = 1, ncol = ng, heights = unit(1, 
                                                            "null"), widths = unit(rep(1, length = ng), rep("lines", 
                                                                                                            ng)), respect = FALSE)
      g = frameGrob(layout = lay)
      dy = 1/x$numArrays
      y = seq_len(x$numArrays) * dy
      for (i in seq_len(ng)) {
        g = placeGrob(g, rectGrob(y = y, height = dy, 
                                  vjust = 1, gp = do.call(gpar, rects[[i]])), 
                      row = 1, col = i)
      }
      idem = function(x) x
      theLegend = list(right = list(fun = idem, args = list(x = g)))
    }
  }
  else {
    thekey = NULL
  }
  hfig = levelplot(m[ord, ord], scales = list(x = list(rot = 90)), 
                   legend = theLegend, colorkey = list(space = "left"), 
                   xlab = "", ylab = "", col.regions = colorRange, main = thekey)
  nout = length(out@which)
  legend = paste0("The figure <!-- FIG --> shows a false color heatmap of the distances between arrays. ", 
                  "The color scale is chosen to cover the range of distances encountered in the dataset. ", 
                  "Patterns in this plot can indicate clustering of the arrays either because of intended biological or ", 
                  "unintended experimental factors (batch effects). ", 
                  "The distance <i>d<sub>ab</sub></i> between two arrays <i>a</i> and <i>b</i> is computed as the mean absolute difference ", 
                  "(L<sub>1</sub>-distance) between the data of the arrays (using the data from all probes without filtering). ", 
                  "In formula, <i>d<sub>ab</sub></i> = mean | <i>M<sub>ai</sub> - M<sub>bi</sub></i> |, ", 
                  "where <i>M<sub>ai</sub></i> is the value of the <i>i</i>-th probe on the <i>a</i>-th array. ", 
                  "Outlier detection was performed by looking for arrays for which the sum of the distances to all other arrays, ", 
                  "<i>S<sub>a</sub></i> = &Sigma;<sub><i>b</i></sub> <i>d<sub>ab</sub></i> was exceptionally large. ", 
                  if (nout > 0) 
                    paste(if (nout > 1) 
                      paste(nout, "such arrays were detected, and they are")
                      else "One such array was detected, and it is", "marked by an asterisk, *.")
                  else "No such arrays were detected.")
  new("aqmReportModule", plot = hfig, section = "Between array comparison", 
      title = "Distances between arrays", id = "hm", legend = legend, 
      size = c(w = 5 + x$numArrays * 0.075, h = 3 + x$numArrays * 
                 0.075 + maxNrColors * 0.2), colors = x$arrayColors, 
      outliers = out)
}




######################################
# Da Função aqm.boxplot:


aqm.boxplotx <- function (x, subsample = 20000, outlierMethod = "KS")  {
  if (nrow(x$M) > subsample) {
    ss = sample(nrow(x$M), subsample)
    Mss = x$M[ss, , drop = FALSE]
  } else {
    ss = TRUE
    Mss = x$M
  }
  out = outliers(Mss, method = outlierMethod)
  out@description = c("Kolmogorov-Smirnov statistic <i>K<sub>a</sub></i>", 
                      "data-driven")
  sample_id = rep(seq_len(x$numArrays), each = nrow(Mss))
  if (x$nchannels == 2) {
    values = with(x, c(R[ss, ], G[ss, ], Mss))
    sample_id = rep(sample_id, times = 3)
    panels = factor(rep(1:3, each = nrow(Mss) * x$numArrays), 
                    levels = 1:3, labels = c("a. Red Channel", "b. Green Channel", 
                                             "c. Log2(Ratio)"))
    formula = sample_id ~ values | panels
    lay = c(3, 1)
    legPanels = c("Three panels are shown: left, red channel; middle, green channel; right, log<sub>2</sub>(ratio). ", 
                  "on the distribution of Log<sub>2</sub>(Ratio). ")
  }  else {
    values = as.numeric(Mss)
    formula = sample_id ~ values
    lay = c(1, 1)
    legPanels = c("", "")
  }
  xAsterisk = quantile(Mss, probs = 0.01, na.rm = TRUE)
  box = bwplot(formula, groups = sample_id, layout = lay, as.table = TRUE, 
               strip = function(..., bg) strip.default(..., bg = "#cce6ff"), 
               horizontal = TRUE, main = if (!is.null(x$key)) 
                 draw.key(key = x$key), pch = "|", col = "black", 
               do.out = FALSE, box.ratio = 2, xlab = "", ylab = "Array", 
               fill = x$arrayColors, panel = panel.superpose, scales = list(x = list(relation = "free"), 
                                                                            y = list(axs = "i")), ylim = c(x$numArrays + 0.7, 
                                                                                                           0.3), prepanel = function(x, y) {
                                                                                                             list(xlim = quantile(x, probs = c(0.01, 0.99), na.rm = TRUE))
                                                                                                           }, panel.groups = function(x, y, ...) {
                                                                                                             panel.bwplot(x, y, ...)
                                                                                                             if (packet.number() == lay[1]) {
                                                                                                               whArray = list(...)$group.value
                                                                                                               if (whArray %in% out@which) 
                                                                                                                 ltext(xAsterisk, whArray, "*", font = 2, cex = 2, 
                                                                                                                       adj = c(0.5, 0.75))
                                                                                                             }
                                                                                                           })
  legend = paste0("The figure <!-- FIG --> shows boxplots representing summaries of the signal intensity distributions of the arrays. ", 
                  legPanels[1], "Each box corresponds to one array. Typically, one expects the boxes to have similar positions and widths. If the ", 
                  "distribution of an array is very different from the others, this may indicate an experimental problem. ", 
                  "Outlier detection was performed ", legPanels[2], "by computing the Kolmogorov-Smirnov statistic <i>K<sub>a</sub></i> between each array's distribution ", 
                  "and the distribution of the pooled data.")
  new("aqmReportModule", plot = box, section = "Array intensity distributions", 
      title = "Boxplots", id = "box", legend = legend, outliers = out, 
      colors = x$arrayColors, size = c(w = 3 + 3 * lay[1], 
                                       h = 2.5 + x$numArrays * 0.1 + 1/x$numArrays + length(x$key$rect$col) * 
                                         0.2))  }






######################################
# Da Função aqm.pca:


aqm.pcax <- function (x) {
  pca = prcomp(t(na.omit(x$M)))
  pcafig = xyplot(PC2 ~ PC1, data = as.data.frame(pca$x), pch = 19, 
                  cex = 1, col = x$arrayColors, main = if (!is.null(x$key)) 
                    draw.key(key = x$key), aspect = "iso")
  legend = paste0("The figure <!-- FIG --> shows a scatterplot of the arrays along the first two principal components. ", 
                  "You can use this plot to explore if the arrays cluster, and whether this is according to an intended experimental factor", 
                  if (length(x$intgroup) == 0) 
                    " (you can indicate such a factor by color using the 'intgroup' argument)"
                  else "", ", or according to unintended causes such as batch effects. Move the mouse over the points to see the sample names.<BR>", 
                  "Principal component analysis is a dimension reduction and visualisation technique that is here used to project ", 
                  "the multivariate data vector of each array into a two-dimensional plot, such that the spatial arrangement of the ", 
                  "points in the plot reflects the overall data (dis)similarity between the arrays.")
  new("aqmReportModule", plot = pcafig, section = "Between array comparison", 
      title = "Principal Component Analysis", id = "pca", legend = legend, 
      size = c(w = 1, h = 1) * 4 + 0.2 * sqrt(x$numArrays) + 
        c(w = 0, h = 1) * length(x$key$rect$col) * 0.2, colors = x$arrayColors, 
      svg = new("svgParameters", numPlotObjects = x$numArrays, 
                getPlotObjNodes = getPlotPoints, gridObjId = "xyplot.points"))
}




######################################
# Da Função aqm.density:

aqm.densityx <- function (x) {
  lwd = lty = 1
  if (x$nchannels == 2) {
    den1 = environment(arrayQualityMetrics)$dens(x$R)
    den2 = environment(arrayQualityMetrics)$dens(x$G)
    den3 = environment(arrayQualityMetrics)$dens(x$M)
    ddf = rbind(den1, den2, den3)
    panels = factor(rep(1:3, times = c(nrow(den1), nrow(den2), 
                                       nrow(den3))), levels = 1:3, labels = c("a. Red Channel", 
                                                                              "b. Green Channel", "c. Log2(Ratio)"))
    formula = y ~ x | panels
    lay = c(3, 1)
    n = x$numArrays
    getReportObjIdFromPlotObjId = function(j) {
      stopifnot(!is.na(j), j > 0L)
      (j - 1L)%%n + 1L
    }
    svgPar = new("svgParameters", numPlotObjects = 3L * x$numArrays, 
                 getReportObjIdFromPlotObjId = getReportObjIdFromPlotObjId)
  }
  else {
    ddf = environment(arrayQualityMetrics)$dens(x$M)
    formula = y ~ x
    lay = c(1, 1)
    svgPar = new("svgParameters", numPlotObjects = x$numArrays)
  }
  den = xyplot(formula, ddf, groups = which, layout = lay, 
               type = "l", ylab = "Density", xlab = "", main = if (!is.null(x$key)) 
                 draw.key(key = x$key), strip = function(..., bg) strip.default(..., 
                                                                                bg = "#cce6ff"), scales = list(relation = "free"), 
               col = x$arrayColors, lwd = lwd, lty = lty)
  legend = "The figure <!-- FIG --> shows density estimates (smoothed histograms) of the data. Typically, the distributions of the arrays should have similar shapes and ranges. Arrays whose distributions are very different from the others should be considered for possible problems. Various features of the distributions can be indicative of quality related phenomena. For instance, high levels of background will shift an array's distribution to the right. Lack of signal diminishes its right right tail. A bulge at the upper end of the intensity range often indicates signal saturation."
  new("aqmReportModule", plot = den, section = "Array intensity distributions", 
      title = "Density plots", id = "dens", legend = legend, 
      size = c(w = 2 + 2 * lay[1], h = 3.5 * lay[2] + length(x$key$rect$col) * 
                 0.2), colors = x$arrayColors, svg = svgPar)
}



######################################
# Da Função aqm.meansd:

aqm.meansdx <- function (x) {
  msdplo = function() {
    meanSdPlot(x$M, ylab = "Standard deviation of the intensities", 
               xlab = "Rank(mean of intensities)")
  }
  legend = "The figure <!-- FIG --> shows a density plot of the standard deviation of the intensities across arrays on the <i>y</i>-axis versus the rank of their mean on the <i>x</i>-axis. The red dots, connected by lines, show the running median of the standard deviation. After normalisation and transformation to a logarithm(-like) scale, one typically expects the red line to be approximately horizontal, that is, show no substantial trend. In some cases, a hump on the right hand of the x-axis can be observed and is symptomatic of a saturation of the intensities."
  new("aqmReportModule", plot = msdplo, section = "Variance mean dependence", 
      title = "Standard deviation versus rank of the mean", 
      id = "msd", legend = legend, colors = x$arrayColors, 
      size = c(w = 6, h = 6))
}



######################################
# Da Função aqm.probesmap:

aqm.probesmapx <- function (x) {
  if (!("hasTarget" %in% colnames(x$fData))) 
    return(NULL)
  df = data.frame(hasTarget = rep(x$fData$hasTarget, ncol(x$A)), 
                  A = as.vector(x$A))
  den = densityplot(~A, data = df, groups = hasTarget, plot.points = FALSE, 
                    auto.key = list(lines = TRUE, points = FALSE, x = 1, 
                                    y = 0.99, corner = c(1, 1)), xlab = "Intensity", 
                    xlim = quantile(df$A, probs = c(0.01, 0.99)), adjust = 0.5)
  legend = "The figure <!-- FIG --> shows the intensity distributions (density estimates, pooled across all arrays) grouped by the value of the feature-level property <tt>hasTarget</tt>. This plot allows you to see whether the intensities of different types of features are systematically different. For instance, if <tt>hasTarget</tt> indicates whether or not a feature maps to a known gene, then you would expect the distribution of those features for which this property is <tt>TRUE</tt> to be shifted towards higher values."
  new("aqmReportModule", plot = den, section = "Feature stratification", 
      title = "Feature stratification", id = "feats", legend = legend, 
      colors = x$arrayColors, size = c(w = 5, h = 5))
}






######################################
# Da Função aqm.maplot

aqm.maplotx <- function (x, subsample = 20000, Dthresh = 0.15, maxNumArrays = 8, 
                         nrColumns = 4) {
  stopifnot(length(Dthresh) == 1, Dthresh > 0, length(subsample) == 
              1, subsample > 0)
  if (x$nchannels == 1) {
    stopifnot(identical(x$M, x$A))
    medArray = rowMedians(x$M, na.rm = TRUE)
    M = x$M - medArray
    A = (x$M + medArray)/2
  }
  else {
    M = x$M
    A = x$A
    stopifnot(identical(dim(M), dim(A)))
  }
  if (nrow(M) > subsample) {
    sel = sample(nrow(M), subsample)
    sM = M[sel, ]
    sA = A[sel, ]
  }
  else {
    sM = M
    sA = A
  }
  stat = sapply(seq_len(x$numArrays), function(j) {
    hoeffd(sA[, j], sM[, j])$D[1, 2]
  })
  attr(stat, "name") = "<i>D<sub>a</sub></i>"
  out = new("outlierDetection", statistic = stat, threshold = Dthresh, 
            which = which(stat > Dthresh), description = c(attr(stat, 
                                                                "name"), "fixed"))
  selected = environment(arrayQualityMetrics)$processMaxNumArrays(x$numArrays, maxNumArrays, 
                                 nrColumns, stat)
  xlim = quantile(A, probs = 1e-04 * c(1, -1) + c(0, 1), na.rm = TRUE)
  ylim = quantile(M, probs = 1e-04 * c(1, -1) + c(0, 1), na.rm = TRUE)
  panelNames = sprintf("array %d (D=%4.2f)", selected$j, stat[selected$j])
  i = seq_along(selected$j)
  df = data.frame(i = factor(i, levels = i), px = i, py = i)
  ma = xyplot(py ~ px | i, data = df, xlim = xlim, ylim = ylim, 
              xlab = "A", ylab = "M", panel = function(x, y, ...) panel.smoothScatter(x = A[, 
                                                                                            selected$j[x]], y = M[, selected$j[y]], raster = TRUE, 
                                                                                      nbin = 250, ...), as.table = TRUE, layout = selected$lay, 
              asp = "iso", strip = function(..., bg, factor.levels) strip.default(..., 
                                                                                  bg = "#cce6ff", factor.levels = panelNames))
  vv = if (length(out@which) == 1) 
    c("One array", "was", "")
  else c(paste(length(out@which), "arrays"), "were", "s")
  outliertext = sprintf("%s had <i>D<sub>a</sub></i>&gt;%g and %s marked as outlier%s. ", 
                        vv[1], Dthresh, vv[2], vv[3])
  legend = paste("The figure <!-- FIG --> shows MA plots. M and A are defined as:<br>", 
                 "M = log<sub>2</sub>(I<sub>1</sub>) - log<sub>2</sub>(I<sub>2</sub>)<br>", 
                 "A = 1/2 (log<sub>2</sub>(I<sub>1</sub>)+log<sub>2</sub>(I<sub>2</sub>)),<br>", 
                 if (x$nchannels == 1) 
                   paste0("where I<sub>1</sub> is the intensity of the array studied,", 
                          "and I<sub>2</sub> is the intensity of a \"pseudo\"-array that consists of the median across arrays.")
                 else "where I<sub>1</sub> and I<sub>2</sub> are the intensities of the two channels.", 
                 " Typically, we expect the mass of the distribution in an MA plot to be concentrated along the M = 0 axis, ", 
                 "and there should be no trend in M as a function of A. If there is a trend in the lower range of A, this often ", 
                 "indicates that the arrays have different background intensities; this may be addressed by background correction. ", 
                 "A trend in the upper range of A can indicate saturation of the measurements; in mild cases, this may be addressed ", 
                 "by non-linear normalisation (e.g. quantile normalisation).<br>", 
                 "Outlier detection was performed by computing Hoeffding's statistic <i>D<sub>a</sub></i> on the joint distribution of A and M for each array. ", 
                 selected$legOrder, "The value of <i>D<sub>a</sub></i> is shown in the panel headings. ", 
                 outliertext, "For more information on Hoeffing's <i>D</i>-statistic, please see the manual page of the function ", 
                 "<tt>hoeffd</tt> in the <tt>Hmisc</tt> package.")
  new("aqmReportModule", plot = ma, section = "Individual array quality", 
      title = "MA plots", id = "ma", legend = legend, outliers = out, 
      size = with(selected, c(w = 3 * lay[1], h = 2.5 * lay[2])), 
      colors = x$arrayColors)
}



######################################
# Da Função aqm.spatial

aqm.spatialx <- function (x, scale = "rank", channels = c("M", "R", "G"), maxNumArrays = 8, 
                          nrColumns = 4) {
  if (!((length(scale) == 1) && is.character(scale) && (scale %in% 
                                                        c("direct", "rank")))) 
    stop("'scale' must be 'direct' or 'rank'\n")
  channels = intersect(channels, names(x))
  if (is.numeric(x$sx) && is.numeric(x$sy)) {
    lapply(channels, spatialplot, x = x, scale = scale, maxNumArrays = maxNumArrays, 
           nrColumns = nrColumns)
  }
  else {
    NULL
  }
}


######################################
# Da Função reportModule
# Alterações feitas: Não produz o arquivo svg, tamanhos são diferentes

reportModulex <- function (p, module, currentIndex, arrayTable, outdir) 
{
  stopifnot(is(module, "aqmReportModule"))
  validObject(module, test = FALSE)
  svgwarn = FALSE
  stopifnot(!is.na(module@title))
  name = module@id
  stopifnot(!any(is.na(module@size)))
  module@size["h"] <- module@size["h"] * 1.25
  module@size["w"] <- module@size["w"] * 1.25
  h = module@size["h"]
  w = module@size["w"]
  stopifnot(length(currentIndex) == 1, is.numeric(currentIndex), 
            !is.na(currentIndex))
  dpi = environment(arrayQualityMetrics)$arrayQualityMetricsGlobalParameters$dpi
  if (is.na(module@svg@numPlotObjects)) {
    #nameimg = paste0(name, ".svg")
    nameimg = paste0(name, ".wmf")
    #nameimgsmall = paste0(name, "small", ".png")
    #nameimgbig = paste0(name, "big", ".png")
    # Exporta em WMF
    #win.metafile(filename = file.path(outdir, nameimg), height = h * 
          #dpi / 75, width = w * dpi / 75,family = "sans")
    #svg(filename = file.path(outdir, nameimg), height = h * 
          #dpi / 75, width = w * dpi / 75)
    #png(filename = file.path(outdir, nameimg), height = h * 
          #dpi, width = w * dpi)
    #png(filename = file.path(outdir, nameimgsmall), height = h * 0.5 * 
          #dpi, width = w * 0.5 * dpi)
    #png(filename = file.path(outdir, nameimgbig), height = h * 2 *
          #dpi, width = w * 2 * dpi)
    
    environment(arrayQualityMetrics)$makePlot(module)
    #tryCatch(grDevices::dev.off(), error = function(e) e, finally = writeLines("Usual grDevices bug."))
    img = hmakeTag("img", src = nameimg, border = 0, alt = nameimg, 
                   id = paste("Fig", name, sep = "ls:"))
  }
  else if (FALSE) {
    nameimg = paste0(name, ".svg")
    if (is(module@plot, "trellis")) {
      theName = file.path(outdir, nameimg)
      thePrefix = paste("Fig", name, sep = ":")
      eval(substitute(gridsvg(name = theName, width = w, 
                              height = h, res = dpi, prefix = thePrefix, usePaths = "none")))
      environment(arrayQualityMetrics)$makePlot(module)
      annRes = environment(arrayQualityMetrics)$annotateSvgGrid(annotationInfo = module@svg, 
                               name = name)
      tryCatch(grDevices::dev.off(), error = function(e) e, finally = writeLines("Usual grDevices bug."))
    }
    else if (FALSE) {
      svgtemp = paste0(tempfile(), ".svg")
      Cairo(file = svgtemp, type = "svg", height = h, width = w, 
            units = "in", dpi = dpi)
      environment(arrayQualityMetrics)$makePlot(module)
      tryCatch(grDevices::dev.off(), error = function(e) e, finally = writeLines("Usual grDevices bug."))
      annRes = environment(arrayQualityMetrics)$annotateSvgPlot(infile = svgtemp, outfile = nameimg, 
                               outdir = outdir, annotationInfo = module@svg, 
                               name = name)
    }
    if (!annRes$annotateOK & FALSE) 
      svgwarn = paste("Note: the figure is static - enhancement with interactive effects failed.", 
                      "This is either due to a version incompatibility of the 'SVGAnnotation' R package and your", 
                      "version of 'Cairo' or 'libcairo', or due to plot misformating. Please consult the Bioconductor forum, or", 
                      "contact the maintainer of 'arrayQualityMetrics' with a reproducible example to help fix this problem.")
    img = hwrite(c(paste(readLines(file.path(outdir, nameimg), 
                                   warn = FALSE), collapse = "\n"), annotationTable(arrayTable, 
                                                                                    name = name)))
  }
  # PDFs não são exportados mais
  #namepdf = paste0(name, ".pdf")
  # pdf(file = file.path(outdir, namepdf), height = h, width = w)
  namewmf = paste0(name, ".wmf")
  win.metafile(filename = file.path(outdir, namewmf), height = h * 
                 dpi / 75, width = w * dpi / 75,family = "sans")
  
  environment(arrayQualityMetrics)$makePlot(module)
  tryCatch(grDevices::dev.off(), error = function(e) e, finally = writeLines("Working in plots."))
  #hwrite("\n\n", page = p)
  #hwrite(environment(arrayQualityMetrics)$toggleStart(name, display = module@defaultdisplay, 
                     #text = sprintf("Figure %d: %s.", currentIndex, module@title)), 
  #       page = p)
  # hwrite(img, page = p)
  #hwrite("<br>\n", page = p)
  # Já disse que PDF não vai ser exportado!
  #hwrite(gsub("The figure <!-- FIG -->", paste0("<b>Figure ", 
                                                #currentIndex, "</b>", if (!is.na(namepdf)) 
                                                #  hwrite(" (PDF file)", link = namepdf)), module@legend, 
              # ignore.case = TRUE), page = p)
  #hwrite("<br><br><br>\n", page = p)
  #if (!identical(svgwarn, FALSE)) 
    #hwrite(svgwarn, page = p)
  #hwrite(environment(arrayQualityMetrics)$toggleEnd(), page = p)
  if (!identical(NA_character_, module@outliers@description)) {
    currentIndex = currentIndex + 1
    reportModulex(p, environment(arrayQualityMetrics)$aqm.outliers(module), currentIndex, arrayTable, 
                 outdir)
  }
  
  return(currentIndex + 1)
}






######################################
# Da Função aqm.writereport

aqm.writereportx <- function (modules, arrayTable, reporttitle, outdir) {
  numReportObjs = nrow(arrayTable)
  reportObjs = seq_len(numReportObjs)
  if (numReportObjs == 0) 
    stop("'arrayTable' must not be empty.")
  ids = sapply(modules, slot, "id")
  stopifnot(!any(is.na(ids)), !any(duplicated(modules)))
  svgdata = lapply(modules, slot, "svg")
  names(svgdata) = ids
  hassvg = !is.na(sapply(svgdata, slot, "numPlotObjects"))
  svgdata = svgdata[hassvg]
  wh = which(sapply(modules, function(x) length(x@outliers@statistic) > 
                      0))
  outlierMethodTitles = sapply(modules, slot, "title")[wh]
  outlierMethodLinks = paste0("<a href=\"#", ids[wh], "\">")
  outlierExplanations = paste0("The columns named *1, *2, ... indicate the calls from the different outlier detection methods:<OL>", 
                               paste(sprintf("<LI> outlier detection by %s%s</a></LI>", 
                                             outlierMethodLinks, outlierMethodTitles), collapse = ""), 
                               "</OL>The outlier detection criteria are explained below in the respective sections. Arrays that were called outliers ", 
                               "by at least one criterion are marked by checkbox selection in this table, and are ", 
                               "indicated by highlighted lines or points in some of the plots below. ", 
                               "By clicking the checkboxes in the table, or on the corresponding points/lines in the plots, you can modify the selection. ", 
                               "To reset the selection, reload the HTML page in your browser.", 
                               "<br><br>", "At the scope covered by this software, outlier detection is a poorly defined question, and there is no 'right' or 'wrong' answer. ", 
                               "These are hints which are intended to be followed up manually. If you want to automate outlier detection, you need to limit the scope ", 
                               "to a particular platform and experimental design, and then choose and calibrate the metrics used.")
  outliers = matrix(NA, nrow = numReportObjs, ncol = length(wh), 
                    dimnames = list(NULL, sprintf("%s*%d</a>", outlierMethodLinks, 
                                                  seq_along(wh))))
  for (j in seq(along = wh)) {
    o = modules[[wh[j]]]@outliers@which
    stopifnot(!any(is.na(o)), all((o >= 1) & (o <= numReportObjs)))
    outliers[, j] = reportObjs %in% o
  }
  rowchar = as.character(row.names(arrayTable))
  rownum = paste(reportObjs)
  left = if (!identical(rowchar, rownum)) 
    data.frame(array = rownum, sampleNames = rowchar, stringsAsFactors = FALSE)
  else data.frame(array = rownum, stringsAsFactors = FALSE)
  arrayTableBig = cbind(left, ifelse(outliers, "x", ""), arrayTable, 
                        stringsAsFactors = FALSE)
  arrayTableCompact = cbind(left, arrayTable, stringsAsFactors = FALSE)
  rownames(arrayTableBig) = rownames(arrayTableCompact) = NULL
  p = environment(arrayQualityMetrics)$makeTitle(reporttitle = reporttitle, outdir = outdir, 
                params = c(HIGHLIGHTINITIAL = environment(arrayQualityMetrics)$toJSON_fromchar(ifelse(apply(outliers, 
                                                                           1, any), "true", "false")), ARRAYMETADATA = environment(arrayQualityMetrics)$toJSON_frommatrix(arrayTableCompact), 
                           SVGOBJECTNAMES = environment(arrayQualityMetrics)$toJSON_fromvector(names(svgdata)), 
                           REPORTOBJSTYLES = paste0(".aqm", reportObjs, " { }", 
                                                    collapse = "\n")))
  environment(arrayQualityMetrics)$makeIndex(p = p, modules = modules)
  environment(arrayQualityMetrics)$reportTable(p = p, arrayTable = arrayTableBig, tableLegend = outlierExplanations)
  currentSectionName = "Something Else"
  currentIndex = currentSection = 1
  for (i in seq(along = modules)) {
    if (modules[[i]]@section != currentSectionName & FALSE) {
      environment(arrayQualityMetrics)$makeSection(p = p, sectionNumber = currentSection, 
                  module = modules[[i]])
      currentSection = currentSection + 1
    }
    currentIndex = reportModulex(p = p, module = modules[[i]], 
                                currentIndex = currentIndex, arrayTable = arrayTableCompact, 
                                outdir = outdir)
    currentSectionName = modules[[i]]@section
  }
  environment(arrayQualityMetrics)$makeEnding(p)
  invisible(list(modules = modules, arrayTable = arrayTableBig, 
                 reporttitle = reporttitle, outdir = outdir))
}













###################################################################################################
###################################################################################################
# FUNÇÕES AFFY-ESPECÍFICO (RAW)


######################################
# Da Função prepaffy:

prepaffyx <- function (expressionset, x) {
  x$dataPLM = fitPLM(expressionset)
  x$pm = pm(expressionset)
  x$mm = mm(expressionset)
  x$mmOK = !any(is.na(x$mm))
  return(x)
}





######################################
# Da Função aqm.rle
aqm.rlex<- function (x, outlierMethod = "KS") {
  x$M = RLE(x$dataPLM, type = "values")
  rv = aqm.boxplot(x, outlierMethod = outlierMethod)
  rv@title = "Relative Log Expression (RLE)"
  rv@section = "Affymetrix specific plots"
  rv@id = "rle"
  rv@legend = paste(figurePhrase(rv@title), "Arrays whose boxes are centered away from 0 and/or are more spread out are potentially problematic.", 
                    "Outlier detection was performed by computing the Kolmogorov-Smirnov statistic <i>R<sub>a</sub></i>", 
                    "between each array's RLE values and the pooled, overall distribution of RLE values.")
  rv@outliers@description = c("Kolmogorov-Smirnov statistic <i>R<sub>a</sub></i> of the RLE values", 
                              "data-driven")
  return(rv)
}


######################################
# Da Função aqm.nuse

aqm.nusex <- function (x, outlierMethod = "upperquartile") {
  x$M = NUSE(x$dataPLM, type = "values")
  rv = aqm.boxplot(x, outlierMethod = outlierMethod)
  rv@title = "Normalized Unscaled Standard Error (NUSE)"
  rv@section = "Affymetrix specific plots"
  rv@id = "nuse"
  rv@legend = paste(figurePhrase(rv@title), "For each array, the boxes should be centered around 1. An array were the values are elevated relative to the other arrays", 
                    "is typically of lower quality. Outlier detection was performed by computing the 75% quantile <i>N<sub>a</sub></i>", 
                    "of each array's NUSE values and looking for arrays with large <i>N<sub>a</sub></i>.")
  rv@outliers@description = c("<i>N<sub>a</sub></i>", "data-driven")
  return(rv)
}



######################################
# Da Função aqm.rnadeg

aqm.rnadegx <- function (expressionset, x) {
  rnaDeg = function() {
    plotAffyRNAdeg(AffyRNAdeg(expressionset, log.it = TRUE), 
                   lwd = 1, cols = x$arrayColors)
  }
  legend = paste(figurePhrase("RNA digestion"), "The shown values are computed from the preprocessed data (after background correction and quantile normalisation). Each array is represented by a single line; move the mouse over the lines to see their corresponding sample names. The plot can be used to identify array(s) that have a slope very different from the others. This could indicate that the RNA used for that array has been handled differently from what was done for the other arrays.")
  new("aqmReportModule", plot = rnaDeg, section = "Affymetrix specific plots", 
      title = "RNA digestion plot", id = "dig", legend = legend, 
      size = c(w = 5, h = 3.5), colors = x$arrayColors, svg = new("svgParameters", 
                                                                  numPlotObjects = x$numArrays))
}


######################################
# Da Função aqm.pmmm

aqm.pmmmx <- function (x) {
  if (!x$mmOK) 
    return(NULL)
  PM = density(as.matrix(log2(x$pm)))
  MM = density(as.matrix(log2(x$mm)))
  PMMM = function() {
    plot(MM, col = "grey", xlab = "log(Intensity)", main = "")
    lines(PM, col = "blue")
    legend("topright", c("PM", "MM"), lty = 1, lwd = 2, col = c("blue", 
                                                                "grey"), bty = "n")
  }
  legend = "Figure <!-- FIG --> shows the density distributions of the log<sub>2</sub> intensities grouped by the matching type of the probes. The blue line shows a density estimate (smoothed histogram) from intensities of perfect match probes (PM), the grey line, one from the mismatch probes (MM). We expect that MM probes have poorer hybridization than PM probes, and thus that the PM curve be to the right of the MM curve."
  new("aqmReportModule", plot = PMMM, section = "Affymetrix specific plots", 
      title = "Perfect matches and mismatches", id = "pmmm", 
      legend = legend, size = c(w = 6, h = 6), colors = x$arrayColors)
}



####################################################################################################
####################################################################################################
# Da Função arrayQualityMetrics:


arrayQualityMetricsx <- function (expressionset, outdir = reporttitle, force = FALSE, 
                                  do.logtransform = FALSE, intgroup = character(0), grouprep, 
                                  spatial = TRUE, reporttitle = paste("arrayQualityMetrics report for", 
                                                                      deparse(substitute(expressionset))), rgbi = c(1,1,1), rgbf = c(1,0,0), ...) 
{
  if (!missing(grouprep)) 
    .Deprecated(msg = paste("The argument 'grouprep' of the function 'arrayQualityMetrics'", 
                            "is deprecated and will be ignored. Use 'intgroup' instead."))
  for (v in c("outdir", "reporttitle")) if (!(is.character(get(v)) && 
                                              (length(get(v)) == 1))) 
    stop(sprintf("'%s' should be a character of length 1.", 
                 v))
  for (v in c("force", "do.logtransform", "spatial")) if (!(is.logical(get(v)) && 
                                                            (length(get(v)) == 1))) 
    stop(sprintf("'%s' should be a logical of length 1.", 
                 v))
  if (!is.character(intgroup)) 
    stop("'intgroup' should be a character.")
  environment(arrayQualityMetrics)$dircreation(outdir, force)
  m = list()
  old.seed = setRNG(kind = "default", seed = 28051968, normal.kind = "default")
  on.exit(setRNG(old.seed))
  x = prepdatax(expressionset, intgroup = intgroup, do.logtransform = do.logtransform)
  pdf(tempfile())
  m$heatmap = aqm.heatmapx(x,rgbi = rgbi, rgbf = rgbf)
  m$pca = aqm.pcax(x)
  m$boxplot = aqm.boxplotx(x)
  m$density = aqm.densityx(x)
  m$meansd = aqm.meansdx(x)
  m$probesmap = aqm.probesmapx(x)
  if (is(expressionset, "AffyBatch")) {
    x = prepaffyx(expressionset, x)
    m$rle = aqm.rlex(x)
    m$nuse = aqm.nusex(x)
    m$rnadeg = aqm.rnadegx(expressionset, x)
    m$pmmm = aqm.pmmmx(x)
  }
  m$maplot = aqm.maplotx(x)
  if (spatial) 
    m = append(m, aqm.spatialx(x))
  res = aqm.writereportx(modules = m, arrayTable = x$pData, 
                        reporttitle = reporttitle, outdir = outdir)
  tryCatch(grDevices::dev.off(), error = function(e) e, finally = writeLines("Usual grDevices bug."))
  invisible(res)
}



