######################################################
# Referência matrizes e utilidades
######################################################
# -> Funções que auxiliam a manipulação de matrizes
#-----------------------------------------------------

# Função toTempDir: Retorna a junção do diretório temporário com o arquivo
toTempDir <- function (filename) { return (paste0(tempDir, filename)) }

# Função printResult: Printa o conteúdo de um comando ou uma variável
printResult <- function(x)
{
  if (is.vector(x)) writeLines(paste(x, sep = "\t", collapse = "\t"))
  else if (is.matrix(x) || is.data.frame(x))
  {
    writeLines(apply(x, 1, "paste", sep="\t", collapse="\t"), sep="\n", useBytes = T)
    #writeLines(paste(apply(x, 1, "paste", sep="\t", collapse="\t"), sep="\a", collapse = "\a"))
  } else writeLines(x)
}

# Função headTail: Obtém os primeiros e últimos elementos de uma matriz
headTail <- function(mat, colname, maxRows = 10, decOrder = F)
{
  
  return(mat[unique(c(head(order(mat[,colname], decreasing = decOrder), n = ceiling(maxRows/2)),
                      head(order(mat[,colname], decreasing = !decOrder), n = floor(maxRows/2)))),])
}

# Função summarySingleResults
summarySingleResults <- function(resultsMatrix, platformMatrix = NULL, symbolColName = "", maxRows = 10)
{
  sumMatrix = headTail(resultsMatrix[,c("ID", "logFC")], "logFC", decOrder = T, maxRows = maxRows)
  if (is.null(platformMatrix) || nchar(symbolColName) == 0)
  {
    return(sumMatrix)
  }
  sumMatrix = cbind(platformMatrix[rownames(sumMatrix), symbolColName], sumMatrix)
  colnames(sumMatrix)[1] = paste0(symbolColName, "/ID")
  sumMatrix = as.matrix(sumMatrix)
  for(i in 1:nrow(sumMatrix))
  {
    if (nchar(as.character(sumMatrix[i,1])) == 0)
      sumMatrix[i,1] = sumMatrix[i,2]
  }
  return(sumMatrix[,-2])
}


# Função readParameters: Lê uma tabela de parâmetros
readParameters <- function(directory = tempDir, filename){
  mtable<-read.table(file = paste(directory, filename, sep = "", collapse = ""), header = FALSE,sep = "\t",row.names=1)
  return(mtable)
}

# Função getParameter: Obtém o(s) valor(es) correspondente(s) ao parâmetro de uma tabela
getParameter <- function(parTable, parameter, type="character"){
  res <- ""
  if(!(parameter %in% rownames(parTable))) return(res)
  res <- switch(type,
         "character" = as.vector(unlist(strsplit(as.character(parTable[parameter,]),"[|]"))),
         "logical" = as.logical(as.vector(unlist(strsplit(as.character(parTable[parameter,]),"[|]")))),
         "integer" = as.integer(as.vector(unlist(strsplit(as.character(parTable[parameter,]),"[|]")))),
         "numeric" = as.numeric(as.vector(unlist(strsplit(as.character(parTable[parameter,]),"[|]")))))
  return(res)
}

# Função giveStatus: Envia um sinal de status pro console R no C#:
giveStatus <- function(percent = -1, message = "?", engMsg = message){
  percent<-as.integer(percent)
  if(percent >= 0) {
    if(percent > 100) percent = 100
    writeLines(paste0(percentTag, as.character(percent)))
  }
  if(message != "?"){
    if (exists('isEnglish') && isEnglish) { writeLines(paste0(statusTag, engMsg)) } else { writeLines(paste0(statusTag, message)) }
  }
}

# Função returnPercentage: Compara um valor mínimo e máximo e devolve uma porcentagem dentro de uma parcimônia
returnPercentage <- function(value, totalValue, minimumPercentage = 0, maximumPercentage = 100){
  minimumPercentage <- max(minimumPercentage, 0)
  maximumPercentage <- min(maximumPercentage, 100)
  parcimony <- maximumPercentage - minimumPercentage
  return (((value * parcimony) / totalValue) + minimumPercentage)
}

# Função giveError: Imprime um sinal de erro usando a mesma lógica do giveSignal
giveError <- function(message = "Error!"){
  writeLines(paste0(errorTag, message))
  stop("ERRO!")
}


#### Função AddMergeColumn: Adiciona uma coluna/matriz em outra matriz com base em rownames iguais. É usada principalmente
# na ocasião da presença de uma coluna "Gene symbol".
# Dadas duas matrizes que contêm rownames em comum, adicionar a coluna de uma matriz para a outra matriz
# ajustando os valores aos rownames em comum e preenchendo os valores sem sobreposição com uma string ("")
AddMergeColumn<-function(msource, mtarget){
  newmatrix <- merge(as.matrix(mtarget), as.matrix(msource), by="row.names",all.x=TRUE) # Sobrepõe matrizes
  newmatrix <- newmatrix[,-1] # Retira a primeira coluna
  rownames(newmatrix) <- rownames(mtarget) # Coloca a primeira coluna como rownames
  colnames(newmatrix) <- c(colnames(mtarget),colnames(msource)) # Preenche nomes das colunas
  return(newmatrix)
}


# Função clearAll: Dado um vetor de objetos, remove cada um para ocupar espaço na memória (e manter organização)
clearAll<-function(vars) {
  for(fls in 1:length(vars)) {
    if(exists(vars[fls])) rm(list=vars[fls],envir = sys.frame(-1))
  }
}


# Lista todos os pacotes que estão com uma versão mais antiga da atualmente disponível
# Returna uma matriz com as colunas Version e Current, sendo os rownames os nomes dos pacotes em questão
packagesToBeUpdated <- function(namesOnly = F)
{
  library("Biobase")
  library("BiocInstaller")
  curr = .packages(T)
  #options(menu.graphics=FALSE)
  local({r <- getOption("repos"); r["CRAN"] <- "https://cran.us.r-project.org"; r["bioconductor"] <- "https://www.bioconductor.org"; options(repos=r)})
  avails = subset(x = available.packages(fields=c('Version'), repos = c(BiocInstaller::biocinstallRepos(), getOption("repos")[1]) ), select = c("Package", "Version"))
  avails = as.data.frame(avails[avails[,"Package"] %in% curr, ], stringAsFactors=T)
  avails$Current = sapply(avails$Package, package.version)
  avails$Status = apply(X=as.matrix(avails), MARGIN=1, FUN=function(x) compareVersion(x[2], x[3]))
  pendPkgs = avails[avails$Status > 0, c('Version', 'Current')]
  if (namesOnly)
  {
    return(rownames(pendPkgs))
  }
  return(pendPkgs)
}

# Remove todos os dados de sessão, incluindo matrizes e vetores
disposeSession <- function()
{
  objs = ls(envir = globalenv())
  rmObjs = c()
  for(obj in objs)
  {
    if (inherits(get(obj), c('matrix', 'data.frame', 'eSet', 'list', 'environment')) || (is.vector(get(obj)) && length(get(obj)) > 1) )
      rmObjs = c(rmObjs, obj)
  }
  if (length(rmObjs) > 0)
    remove(list = rmObjs, envir = globalenv())
  return(invisible(0))
}

# Dado uma string em formato base-64, decodifica e executa
callFromBase64 <- function(b64str)
{
  library('openssl')
  callInput = rawToChar(base64_decode(b64str))
  return(eval(expr = parse(text = callInput), envir = globalenv()))
}
