/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/
/////////////////////////////////////////////////////////////////////
///
///  PROJECT.....: CO-ME
///  RESPONSIBLE.: Anderson Maciel
///
///  AUTHOR......: Fabiana Benedetti
///  DATE........: September/10/2002
///  DESCRIPTION.: Class definition.
///
////////////////////////////////////////////////////////////////////

#include <algebra/comeface.h>
#include <algebra/comemesh.h>
#include <algebra/comematrix.h>
#include <bio/comebiostructure.h>
#include <math.h>
#include <bio/comemolecule.h>

//////////////////////////////////////////////////////////////////////
/// Construction/Destruction
//////////////////////////////////////////////////////////////////////
COME_Face :: COME_Face(){

	color[0] = 1;
	color[1] = .7;
	color[2] = .3;
	parent = NULL;
	nearestMolecule = NULL;
	collisionDetectable = true;
}

/*COME_Face :: COME_Face( const COME_Face& pFace ){

	normal = pFace.getNormal();
	normalGlobalPosition = pFace.getNormalGlobalPosition();
	nearestMolecule = pFace.getNearestMolecule();
	neighbourFaces = pFace.getNeighbourFaces();
	color[0] = pFace.getColor()[0];
	color[1] = pFace.getColor()[1];
	color[2] = pFace.getColor()[2];
	color[3] = pFace.getColor()[3];
	parent = pFace.getParent();
}*/

COME_Face :: COME_Face( vector<int> verticesN, COME_Mesh* parentN ){

	vertices = verticesN;
	parent	= parentN;
	updateNormal();
	nearestMolecule = NULL;
	color[0] = 1;
	color[1] = .7;
	color[2] = .3;	
	collisionDetectable = true;
}

COME_Face :: COME_Face( vector<int> verticesN ){

	vertices = verticesN;
	if( parent ){
		updateNormal();
	}
	nearestMolecule = NULL;
	color[0] = 1;
	color[1] = .7;
	color[2] = .3;	
	collisionDetectable = true;
}

COME_Face :: ~COME_Face(){

	
}

//////////////////////////////////////////////////////////////////////
/// Setting
//////////////////////////////////////////////////////////////////////
void		
COME_Face :: setVertex( vector<int> verticesN ) {
	
	vertices = verticesN;
}

void
COME_Face :: setNormal( COME_Vector3D normalN ){

	normal = normalN;
}

void
COME_Face::setNearestMolecule( COME_Molecule *nearest ){

	nearestMolecule = nearest;
}

void 
COME_Face::setColor(double* c){
	
	color[0] = c[0];
	color[1] = c[1];
	color[2] = c[2];
	color[3] = c[3];
}

void
COME_Face::setCollisionDetectable( bool yesno ){
	
	collisionDetectable = yesno;
}
	

//////////////////////////////////////////////////////////////////////
/// Getting
//////////////////////////////////////////////////////////////////////

int
COME_Face :: getIndexVertex( int iPosN ) const{

	return vertices[ iPosN ];
}

vector<int>
COME_Face:: getIndexVertices(){
	return vertices;
}
int	
COME_Face :: getNumberVertices() const{
	
	return vertices.size();
}

bool
COME_Face::getCollisionDetectable(){
	
	return collisionDetectable;
}

COME_Vertex3D	
COME_Face :: getVertex( int iPosN ) const{

	if( parent)
		return ((COME_Mesh*)parent)->getAVertex( vertices[ iPosN ] );
	printf( "COME_Face::getVertex(int): A parent mesh is not set for a face trying to return a vertex.\n" );
	exit(0);
	
}

COME_Vertex3D*
COME_Face :: getClosestVertexPt( COME_Point3D currentPoint ){

	double distance = INFINITE;
	int closest = 0;
	vector<COME_Vertex3D*> globalVertices = getVerticesGlobalPositionPt();
	for( int i = 0; i < globalVertices.size(); i++ ){
		
		if( distance > globalVertices[i]->getPoint3D().vpDistance( currentPoint ) ){
			distance = globalVertices[i]->getPoint3D().vpDistance( currentPoint );
			closest = i;
		}
	}
	return globalVertices[closest];
}

double
COME_Face :: geoSolidAngle( COME_Point3D point2Check ){

	double area = 0.0, ang, s, l1, l2;
	COME_Vector3D p1, p2, r1, a, b, n1, n2, plane;
	
	vector<COME_Vertex3D*> globalVertices = getVerticesGlobalPositionPt();
	plane = getNormalGlobalPositionPt()->getPoint3D();
	p2 = getVertexPt(2)->getPoint3D();
	p1 = getVertexPt(0)->getPoint3D();
	a = p2 - p1;
	
	for( int i = 0; i < vertices.size(); i++   ){
		
		r1 = p1 - point2Check;
		p2 = globalVertices[(i+1)%vertices.size()]->getPoint3D();
		b = p2 - p1;
		n1 = a * r1;
		n2 = r1 * b;
		
		l1 = n1.vpModule();
		l2 = n2.vpModule();
		s = n1.vpDotProduct(n2) / (l1 * l2 );
		ang = acos( max( -1.0, min( 1.0, s ) ) );
		s = ( b * a ).vpDotProduct( plane );
		area += s > 0.0 ? M_PI - ang : M_PI + ang;
		
		a = b * -1.0;
		p1 = p2;
	}
	
	area -= M_PI*(vertices.size()-2);
	
	return (plane.vpDotProduct(r1)> 0.0) ? -area : area;
}

COME_Vertex3D*	
COME_Face :: getVertexPt( int iPosN ) {

	if( parent)
		return ((COME_Mesh*)parent)->getAVertexPt( vertices[ iPosN ] );
	printf( "COME_Face::getVertexPt(int): A parent mesh is not set for a face trying to return a vertex.\n" );
	exit(0);
}


COME_Vertex3D	
COME_Face :: getVertexGlobalPosition( int iPosN ) const{
	return ((COME_Mesh*)parent)->getAVertexGlobalPosition( vertices[ iPosN ] );
}

COME_Vertex3D*	
COME_Face :: getVertexGlobalPositionPt( int iPosN ){
	return ((COME_Mesh*)parent)->getAVertexGlobalPositionPt( vertices[ iPosN ] );
}

COME_Vector3D
COME_Face :: getNormal() const{

	return normal;
}

COME_Vector3D
COME_Face :: getNormalGlobalPosition() const {

	return normalGlobalPosition;
}

COME_Vector3D*
COME_Face :: getNormalGlobalPositionPt(){

	return &normalGlobalPosition;
}

const double*
COME_Face::getColor() const {
	return color;
}

//////////////////////////////////////////////////////////////////////
/// This method calculates/recalculates the normal vector of this face.
//////////////////////////////////////////////////////////////////////
void
COME_Face::updateNormal(){

	COME_Vector3D v1; v1.setVector3D( ((COME_Mesh*)parent)->getAVertex(vertices[1]).getPoint3D() - ((COME_Mesh*)parent)->getAVertex(vertices[0]).getPoint3D() );
	COME_Vector3D v2; v2.setVector3D( ((COME_Mesh*)parent)->getAVertex(vertices[2]).getPoint3D() - ((COME_Mesh*)parent)->getAVertex(vertices[0]).getPoint3D() );
	COME_Vector3D v3 = v1 * v2;
	v3.vpNormalize();
	normal = COME_Vector3D( v3 );

}

void
COME_Face::updateNormalGlobalPosition(){
	
	vector<COME_Vertex3D*> globalVertices = getVerticesGlobalPositionPt();
	COME_Vector3D v1; v1.setVector3D(globalVertices[1]->getPoint3D() - globalVertices[0]->getPoint3D() );
	COME_Vector3D v2; v2.setVector3D(globalVertices[2]->getPoint3D() - globalVertices[0]->getPoint3D() );
	COME_Vector3D v3 = (v1 * v2);
	v3.vpNormalize();
	normalGlobalPosition = v3;
}


vector<COME_Vertex3D>	
COME_Face :: getVertices() const{

	vector<COME_Vertex3D> verticesN;
	for ( int i = 0; i < vertices.size(); i++ ){
		
		verticesN.push_back( getVertex( i ));		
	
	}

	return verticesN;
}


vector<COME_Vertex3D>	
COME_Face :: getVerticesGlobalPosition() const{

	vector<COME_Vertex3D> verticesN;
	for ( int i = 0; i < vertices.size(); i++ ){
		
		verticesN.push_back( getVertexGlobalPosition( i ));		
	
	}

	return verticesN;
}

vector<COME_Vertex3D*>	
COME_Face :: getVerticesGlobalPositionPt(){

	vector<COME_Vertex3D*> verticesN;
	for ( int i = 0; i < vertices.size(); i++ ){
		
		verticesN.push_back( getVertexGlobalPositionPt( i ));		
	
	}

	return verticesN;
}

COME_Molecule*
COME_Face::getNearestMolecule() const {

	return nearestMolecule;
}

//////////////////////////////////////////////////////////////////////
/// This method calculates and returns the Hooke's constant to the region
///	of this face according to its proximity to the nearest inner molecule and the
///	 and the Hooke's constant of such molecule.
//////////////////////////////////////////////////////////////////////
double
COME_Face::getElasticity(){

	return ( nearestMolecule->getMaterial()->getYoungsModulus() );
}


//////////////////////////////////////////////////////////////////////
/// This method calculates the center of this face by averaging its vertices positions.
//////////////////////////////////////////////////////////////////////
COME_Point3D
COME_Face::getCenter(){

	COME_Vector3D center;
	vector<COME_Vertex3D> verticesFace  = getVertices();

	for ( int i = 0; i < verticesFace.size(); i++ ){
		
		center.setX( center.getX() + verticesFace[ i ].getX() );
		center.setY( center.getY() + verticesFace[ i ].getY() );
		center.setZ( center.getZ() + verticesFace[ i ].getZ() );
	}

	center /= 3.0;

	return center.getPoint3D();

}

COME_Vertex3D
COME_Face::getCenterGlobalPosition(){

	COME_Vertex3D center;
	vector<COME_Vertex3D> verticesFace  = getVerticesGlobalPosition();

	for ( int i = 0; i < verticesFace.size(); i++ ){
		
		center.x = center.x + verticesFace[ i ].x;
		center.y = center.y + verticesFace[ i ].y;
		center.z = center.z + verticesFace[ i ].z;
	}

	center.x /= 3.0;
	center.y /= 3.0;
	center.z /= 3.0;

	return center;

}

COME_Vertex3D*
COME_Face::getCenterGlobalPositionPt(){

	COME_Vertex3D *center = new COME_Vertex3D();
	vector<COME_Vertex3D*> verticesFace  = getVerticesGlobalPositionPt();

	for ( int i = 0; i < verticesFace.size(); i++ ){
		
		center->x = center->x + verticesFace[ i ]->x;
		center->y = center->y + verticesFace[ i ]->y;
		center->z = center->z + verticesFace[ i ]->z;
	}

	center->x /= 3.0;
	center->y /= 3.0;
	center->z /= 3.0;

	return center;

}

//////////////////////////////////////////////////////////////////////
/// This method calculates the area of this face.
/// IT'S JUST A RAW ESTIMATION, THE AREA IS NOT CORRECTLY CALCULATED
//////////////////////////////////////////////////////////////////////
double
COME_Face :: getArea(){

	vector<COME_Vertex3D> verticesFace  = getVertices();
	double area = ( ( COME_Vector3D( ( verticesFace[1] - verticesFace[0] ) + ( verticesFace[2] - verticesFace[0] ) ).vpModule() / 2.0 ) * ( COME_Vector3D( verticesFace[2] - verticesFace[1] ).vpModule() ) ) / 2.0;
	return area;

}

//////////////////////////////////////////////////////////////////////
/// This method calculates the area of the triangle defined by the three given points..
//////////////////////////////////////////////////////////////////////
double
COME_Face :: triangleArea( COME_Point3D A, COME_Point3D B, COME_Point3D C ){

	COME_Vector3D bNormal = C - A; bNormal.vpNormalize();
	COME_Vector3D cNormal = B - A; cNormal.vpNormalize();
	
	double area = 0.5* COME_Vector3D( C - A ).vpModule() * COME_Vector3D( B - A ).vpModule() * sin( acos( bNormal.vpDotProduct( cNormal ) ) );
	//printf( "area: %f\n", area );
	return area;
}

//////////////////////////////////////////////////////////////////////
/// This method blends the stress of the vertices of this face.
//////////////////////////////////////////////////////////////////////
double
COME_Face :: getStress(){

	vector<COME_Vertex3D> verticesFace  = getVertices();
	double stress = ( verticesFace[0].getStress() + verticesFace[1].getStress() + verticesFace[2].getStress() ) / 3.0;
	return stress;

}

//////////////////////////////////////////////////////////////////////
/// Gives the distance between the given point and a triangular face.
/// Distance is calculated considering the closest point on the triangle.
/// The closest point may be on a vertex, on an edge or inside the triangle.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: distancePointFace( COME_Point3D pPos ){

	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexPt( 0 )->x, getVertexPt( 0 )->y, getVertexPt( 0 )->z );
	double distPointPlan = fabs( normal.vpDotProduct( p_a ) );

	COME_Point3D pPosPr = pPos - ( normal * distPointPlan );
	
	if( !isInside2D( pPosPr ) ){
		
		COME_Point3D pA, pB, pC;
		pA = COME_Vector3D( getVertexPt( 0 )->x, getVertexPt( 0 )->y, getVertexPt( 0 )->z );
		pB = COME_Vector3D( getVertexPt( 1 )->x, getVertexPt( 1 )->y, getVertexPt( 1 )->z );
		pC = COME_Vector3D( getVertexPt( 2 )->x, getVertexPt( 2 )->y, getVertexPt( 2 )->z );
		
		COME_Vector3D a_b = pB  - pA;
		a_b.vpNormalize();
		COME_Vector3D b_c = pC  - pB;
		b_c.vpNormalize();
		COME_Vector3D c_a = pA  - pC;
		c_a.vpNormalize();
		
		/// Project on lines formed by the edges
		COME_Point3D pOnEdgeAB = pA + ( a_b * ( (COME_Vector3D( pPosPr - pA )).vpDotProduct(a_b) )  );
		COME_Point3D pOnEdgeBC = pB + ( b_c * ( (COME_Vector3D( pPosPr - pB )).vpDotProduct(b_c) )  );
		COME_Point3D pOnEdgeCA = pC + ( c_a * ( (COME_Vector3D( pPosPr - pC )).vpDotProduct(c_a) )  );
		
		/// Project on vertices if out of edges
		if( ( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() + (COME_Vector3D(pB-pOnEdgeAB)).vpModule() ) > (COME_Vector3D(pA-pB)).vpModule() ){
			if( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() < (COME_Vector3D(pB-pOnEdgeAB)).vpModule() )
				pOnEdgeAB = pA;
			else pOnEdgeAB = pB;
		}
		if( ( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() + (COME_Vector3D(pC-pOnEdgeBC)).vpModule() ) > (COME_Vector3D(pB-pC)).vpModule() ){
			if( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() < (COME_Vector3D(pC-pOnEdgeBC)).vpModule() )
				pOnEdgeBC = pB;
			else pOnEdgeBC = pC;
		}
		if( ( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() + (COME_Vector3D(pA-pOnEdgeCA)).vpModule() ) > (COME_Vector3D(pC-pA)).vpModule() ){
			if( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() < (COME_Vector3D(pA-pOnEdgeCA)).vpModule() )
				pOnEdgeCA = pC;
			else pOnEdgeCA = pA;
		}
		
		double distProj = pPos.vpDistance( pOnEdgeAB );
		if( distProj > pPos.vpDistance( pOnEdgeBC ) ){
			distProj = pPos.vpDistance( pOnEdgeBC );
			if( distProj > pPos.vpDistance( pOnEdgeCA ) ){
				distProj = pPos.vpDistance( pOnEdgeCA );
			}
		}
		return distProj;
	}
	
	return distPointPlan;
}


//////////////////////////////////////////////////////////////////////
///  Assign the two points given by reference with the ones composing the edge closest to the given point
//////////////////////////////////////////////////////////////////////
void
COME_Face :: getClosestEdge( COME_Point3D &pA, COME_Point3D &pB, COME_Point3D p ){

	pA = COME_Point3D( getVertexPt( 0 )->x, getVertexPt( 0 )->y, getVertexPt( 0 )->z );
	pB = COME_Point3D( getVertexPt( 1 )->x, getVertexPt( 1 )->y, getVertexPt( 1 )->z );
	COME_Point3D pC = COME_Point3D( getVertexPt( 2 )->x, getVertexPt( 2 )->y, getVertexPt( 2 )->z );
	
	if( p.vpDistance( pA ) < p.vpDistance( pB ) ){
		if( p.vpDistance( pB ) < p.vpDistance( pC ) ){
			return;
		} else {
			pB = pC;
			return;
		}
	} else if ( p.vpDistance( pB ) < p.vpDistance( pC ) ){
		if( p.vpDistance( pC ) < p.vpDistance( pA ) ){
			pA = pC;
			return;
		} else {
			return;
		}
	}	
	
}

//////////////////////////////////////////////////////////////////////
///  Returns true if the given point (that's in the same plane of the face) is inside the face.
//////////////////////////////////////////////////////////////////////
bool
COME_Face :: isInside2D( COME_Point3D pPos ){

	//find pA
	COME_Point3D pA = COME_Point3D( getVertexPt( 0 )->x, getVertexPt( 0 )->y, getVertexPt( 0 )->z );
	COME_Point3D pB = COME_Point3D( getVertexPt( 1 )->x, getVertexPt( 1 )->y, getVertexPt( 1 )->z );
	COME_Point3D pC = COME_Point3D( getVertexPt( 2 )->x, getVertexPt( 2 )->y, getVertexPt( 2 )->z );
	
	// build vectors
	COME_Vector3D c_b = pC  - pB;
	COME_Vector3D p_b = pPos  - pB;
	COME_Vector3D a_b = pA  - pB;
	COME_Vector3D c_a = pC  - pA;
	COME_Vector3D p_a = pPos  - pA;
	COME_Vector3D b_a = pB  - pA;
	
	if( ( c_b * p_b ).vpDotProduct( c_b * a_b ) < 0 ){
		return false;
	} else if( ( c_a * p_a ).vpDotProduct( c_a * b_a ) < 0 ){
		return false;
	} else if( ( b_a * p_a ).vpDotProduct( b_a * c_a ) < 0 ){
		return false;
	} 
	return true;
}

//////////////////////////////////////////////////////////////////////
/// Gives the distance between the given point and a triangular face.
/// Distance is calculated considering the closest point on the triangle.
/// The closest point may be on a vertex, on an edge or inside the triangle.
/// Use global coordinates for the face.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: distancePointFaceGlobal( COME_Point3D pPos ){

	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double distPointPlan = normalGlobalPosition.vpDotProduct( p_a );

	COME_Point3D pPosPr = pPos - ( normalGlobalPosition * distPointPlan );
	
	distPointPlan = fabs( distPointPlan );
	
	if( !isInside2DGlobal( pPosPr ) ){
		
		COME_Point3D pA, pB, pC;
		pA = COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
		pB = COME_Vector3D( getVertexGlobalPositionPt( 1 )->x, getVertexGlobalPositionPt( 1 )->y, getVertexGlobalPositionPt( 1 )->z );
		pC = COME_Vector3D( getVertexGlobalPositionPt( 2 )->x, getVertexGlobalPositionPt( 2 )->y, getVertexGlobalPositionPt( 2 )->z );
		
		COME_Vector3D a_b = pB  - pA;
		a_b.vpNormalize();
		COME_Vector3D b_c = pC  - pB;
		b_c.vpNormalize();
		COME_Vector3D c_a = pA  - pC;
		c_a.vpNormalize();
		
		/// Project on lines formed by the edges
		COME_Point3D pOnEdgeAB = pA + ( a_b * ( (COME_Vector3D( pPosPr - pA )).vpDotProduct(a_b) )  );
		COME_Point3D pOnEdgeBC = pB + ( b_c * ( (COME_Vector3D( pPosPr - pB )).vpDotProduct(b_c) )  );
		COME_Point3D pOnEdgeCA = pC + ( c_a * ( (COME_Vector3D( pPosPr - pC )).vpDotProduct(c_a) )  );
		
		/// Project on vertices if out of edges
		double EPSILON = 0.0000000001;
		if( ( ( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() )
		||  ( ( (COME_Vector3D(pB-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() ) ){
			if( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() < (COME_Vector3D(pB-pOnEdgeAB)).vpModule() )
				pOnEdgeAB = pA;
			else pOnEdgeAB = pB;
		}
		if( ( ( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() )
		||  ( ( (COME_Vector3D(pC-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() ) ){
			if( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() < (COME_Vector3D(pC-pOnEdgeBC)).vpModule() )
				pOnEdgeBC = pB;
			else pOnEdgeBC = pC;
		}
		if( ( ( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() )
		||  ( ( (COME_Vector3D(pA-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() ) ){
			if( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() < (COME_Vector3D(pA-pOnEdgeCA)).vpModule() )
				pOnEdgeCA = pC;
			else pOnEdgeCA = pA;
		}
	
		double distProj = pPos.vpDistance( pOnEdgeAB );
		if( distProj > pPos.vpDistance( pOnEdgeBC ) ){
			distProj = pPos.vpDistance( pOnEdgeBC );
		}
		if( distProj > pPos.vpDistance( pOnEdgeCA ) ){
			distProj = pPos.vpDistance( pOnEdgeCA );
		}
		return distProj;
	}
	return distPointPlan;
}

//////////////////////////////////////////////////////////////////////
/// Gives the distance between the given point and a triangular face.
/// Distance is calculated considering the closest point on the triangle.
/// The closest point may be on a vertex, on an edge or inside the triangle.
/// Use global coordinates for the face.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: signedDistancePointFaceGlobal( COME_Point3D pPos ){

	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double distPointPlan = normalGlobalPosition.vpDotProduct( p_a );
	double signal = 1.0;
	if( distPointPlan < 0 ){
		signal = -1;
	}

	COME_Point3D pPosPr = pPos - ( normalGlobalPosition * distPointPlan );
	
	distPointPlan = fabs( distPointPlan );
	
	if( !isInside2DGlobal( pPosPr ) ){
		
		COME_Point3D pA, pB, pC;
		pA = COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
		pB = COME_Vector3D( getVertexGlobalPositionPt( 1 )->x, getVertexGlobalPositionPt( 1 )->y, getVertexGlobalPositionPt( 1 )->z );
		pC = COME_Vector3D( getVertexGlobalPositionPt( 2 )->x, getVertexGlobalPositionPt( 2 )->y, getVertexGlobalPositionPt( 2 )->z );
		
		COME_Vector3D a_b = pB  - pA;
		a_b.vpNormalize();
		COME_Vector3D b_c = pC  - pB;
		b_c.vpNormalize();
		COME_Vector3D c_a = pA  - pC;
		c_a.vpNormalize();
		
		/// Project on lines formed by the edges
		COME_Point3D pOnEdgeAB = pA + ( a_b * ( (COME_Vector3D( pPosPr - pA )).vpDotProduct(a_b) )  );
		COME_Point3D pOnEdgeBC = pB + ( b_c * ( (COME_Vector3D( pPosPr - pB )).vpDotProduct(b_c) )  );
		COME_Point3D pOnEdgeCA = pC + ( c_a * ( (COME_Vector3D( pPosPr - pC )).vpDotProduct(c_a) )  );
		
		/// Project on vertices if out of edges
		double EPSILON = 0.0000000001;
		if( ( ( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() )
		||  ( ( (COME_Vector3D(pB-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() ) ){
			if( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() < (COME_Vector3D(pB-pOnEdgeAB)).vpModule() )
				pOnEdgeAB = pA;
			else pOnEdgeAB = pB;
		}
		if( ( ( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() )
		||  ( ( (COME_Vector3D(pC-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() ) ){
			if( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() < (COME_Vector3D(pC-pOnEdgeBC)).vpModule() )
				pOnEdgeBC = pB;
			else pOnEdgeBC = pC;
		}
		if( ( ( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() )
		||  ( ( (COME_Vector3D(pA-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() ) ){
			if( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() < (COME_Vector3D(pA-pOnEdgeCA)).vpModule() )
				pOnEdgeCA = pC;
			else pOnEdgeCA = pA;
		}
	
		double distProj = pPos.vpDistance( pOnEdgeAB );
		if( distProj > pPos.vpDistance( pOnEdgeBC ) ){
			distProj = pPos.vpDistance( pOnEdgeBC );
		}
		if( distProj > pPos.vpDistance( pOnEdgeCA ) ){
			distProj = pPos.vpDistance( pOnEdgeCA );
		}
		return distProj * signal;
	}
	return distPointPlan * signal;
}


//////////////////////////////////////////////////////////////////////
/// Assign the two points given by reference with the ones composing the edge closest to the given point
/// Use global coordinates for the face.
//////////////////////////////////////////////////////////////////////
/*void
COME_Face :: getClosestEdgeGlobal( COME_Point3D &pA, COME_Point3D &pB, COME_Point3D p ){

	pA = COME_Point3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	pB = COME_Point3D( getVertexGlobalPositionPt( 1 )->x, getVertexGlobalPositionPt( 1 )->y, getVertexGlobalPositionPt( 1 )->z );
	COME_Point3D pC = COME_Point3D( getVertexGlobalPositionPt( 2 )->x, getVertexGlobalPositionPt( 2 )->y, getVertexGlobalPositionPt( 2 )->z );
	
	if( p.vpDistance( pA ) < p.vpDistance( pB ) ){
		if( p.vpDistance( pB ) < p.vpDistance( pC ) ){
			return;
		} else {
			pB = pC;
			return;
		}
	} else if ( p.vpDistance( pB ) < p.vpDistance( pC ) ){
		if( p.vpDistance( pC ) < p.vpDistance( pA ) ){
			pA = pC;
			return;
		} else {
			return;
		}
	}	
	
}*/

//////////////////////////////////////////////////////////////////////
/// Returns true if the given point (that's in the same plane of the face) is inside the face.
/// Use global coordinates for the face.
//////////////////////////////////////////////////////////////////////
bool
COME_Face :: isInside2DGlobal( COME_Point3D pPos ){

	//find pA
	COME_Point3D pA = COME_Point3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	COME_Point3D pB = COME_Point3D( getVertexGlobalPositionPt( 1 )->x, getVertexGlobalPositionPt( 1 )->y, getVertexGlobalPositionPt( 1 )->z );
	COME_Point3D pC = COME_Point3D( getVertexGlobalPositionPt( 2 )->x, getVertexGlobalPositionPt( 2 )->y, getVertexGlobalPositionPt( 2 )->z );

	// build vectors
	COME_Vector3D c_b = pC  - pB;
	COME_Vector3D p_b = pPos  - pB;
	COME_Vector3D a_b = pA  - pB;
	COME_Vector3D c_a = pC  - pA;
	COME_Vector3D p_a = pPos  - pA;
	COME_Vector3D b_a = pB  - pA;
	
	if( ( c_b * p_b ).vpDotProduct( c_b * a_b ) < 0 ){
		return false;
	} else if( ( c_a * p_a ).vpDotProduct( c_a * b_a ) < 0 ){
		return false;
	} else if( ( b_a * p_a ).vpDotProduct( b_a * c_a ) < 0 ){
		return false;
	} 
	return true;
}

//////////////////////////////////////////////////////////////////////
/// Returns the distance from the plane formed by the face to the given point.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: distancePointPlaneGlobalPosition( COME_Point3D pPos ){

	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double dist = normalGlobalPosition.vpDotProduct( p_a );

	return fabs(dist);

}

//////////////////////////////////////////////////////////////////////
/// Returns the distance from the plane formed by the face to the given vertex.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: distanceVertexPlaneGlobalPosition( COME_Vertex3D vertex ){

	COME_Vector3D p_a = COME_Vector3D( vertex.x, vertex.y, vertex.z ) - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double dist = normalGlobalPosition.vpDotProduct( p_a );

	return fabs(dist);

}

// Returns the vector having the given vertex as origin and its projection
// on the face as end point.
/*COME_Vector3D
COME_Face :: vectorVertexFaceGlobalPosition( COME_Vertex3D vertex ){

	//COME_Vector3D direction = (vertex - getCenterGlobalPosition());	
	//return (normalGlobalPosition.vpNormalize().vpDotProduct(direction));
	
	vector<COME_Vertex3D*> verticesFace = getVerticesGlobalPositionPt(); 

	COME_Vector3D unitAB;
	unitAB.setVector3D( *(verticesFace[1]) - *(verticesFace[0]) );
	unitAB.vpNormalize();

	COME_Vector3D AP;
	AP.setVector3D( vertex - *(verticesFace[0]) );

	COME_Vector3D AB;
	AB.setVector3D( *(verticesFace[1]) - *(verticesFace[0]) );

	COME_Vector3D w1 =  unitAB * AP.vpModule() * AP.vpDotProduct( AB );	             

	COME_Vector3D unitAC;
	unitAC.setVector3D( *(verticesFace[2]) - *(verticesFace[0]) );
	unitAC.vpNormalize();
	
	COME_Vector3D AC;
	AC.setVector3D( *(verticesFace[2]) - *(verticesFace[0]) );
		             
	COME_Vector3D w2 =  unitAC * AP.vpModule() * AP.vpDotProduct( AC );	

	COME_Point3D pointPlane = w1 + w2 + vertex;

	
	return COME_Vector3D( pointPlane - vertex );

}*/

//////////////////////////////////////////////////////////////////////
/// Returns the distance of penetration of the given face to the object
/// whom this face belongs.
/// If they don't penetrate returns zero.
//////////////////////////////////////////////////////////////////////
double				
COME_Face :: distanceFaceFace( COME_Face face ){

	vector<double> vDistance;
	double distance;

	vector<COME_Vertex3D> realVertices = getVertices();
	for ( int i = 0; i < realVertices.size(); i++ ){
	
		if ( face.isInside( realVertices[ i ] )) {
		
			vDistance.push_back( face.distancePointFace( realVertices[ i ] ));
		
		}	
	}

	if ( vDistance.size() != 0 ){
	
		distance = 0;
		for ( int j = 0; j < vDistance.size(); j++ ){

			if ( vDistance[ j ] > distance ){

				distance = vDistance[ j ];	
			}
		} 
	}
	else {
		distance = 0;
	}
	return distance;
}

double
COME_Face :: distanceFaceFaceGlobalPosition( COME_Face* faceN ){

	vector<double> vDistance;
	double distance;
 	
	vector<COME_Vertex3D*> vertices = getVerticesGlobalPositionPt();
	
	//Distance is negative if vertex inside the mesh of the face!!
	for ( int i = 0; i < vertices.size(); i++ ){
		
		vDistance.push_back( faceN->distanceVertexPlaneGlobalPosition( *(vertices[ i ]) ));
	}
	
	distance = 0;
	for ( int j = 0; j < vDistance.size(); j++ ){
		
		if ( fabs( vDistance[ j ] ) > fabs( distance ) ){
			distance = vDistance[ j ];
			}
	} 
	
	//printf ("DIST=%f\n", distance);
	return distance;
}


//////////////////////////////////////////////////////////////////////
/// Check if the given point is inside the mesh to which this face belongs.
/// Assume that if it is in the exterior side of the face (use normal to check) and
/// if it is in the exterior side of the neighboring faces it is OUTSIDE otherwise its INSIDE.
/// Use global coordinates for the face.
//////////////////////////////////////////////////////////////////////
bool
COME_Face :: isInsideGlobalPosition( COME_Point3D currentPoint ){
	
	COME_Vector3D direction;
	
	direction.setVector3D( currentPoint - getCenterGlobalPosition());
	direction.vpNormalize();
	
	COME_Vector3D globalNormal =  getNormalGlobalPosition();
		
	if( globalNormal.vpDotProduct( direction ) >= 0 ) return false;
	
	vector <int> neighbourFaces = getNeighbourFaces();
	for( int j = 0; j < neighbourFaces.size(); j++ ){
	
		direction.setVector3D( currentPoint - ((COME_Mesh*)(parent))->getAFace(neighbourFaces[j]).getCenterGlobalPosition());
		direction.vpNormalize();
		
		globalNormal =  ((COME_Mesh*)(parent))->getAFace(neighbourFaces[j]).getNormalGlobalPosition();
		
		if( globalNormal.vpDotProduct( direction ) >= 0 ) return false;
	}
	
	return true;
}

//////////////////////////////////////////////////////////////////////
/// Check if the given point is inside the mesh to which this face belongs.
/// Assume that this face is the closest to the point within this mesh.
/// If it is in the exterior side of the face (use normal to check) return FALSE.
/// If it is in the interior side of the face, check if its projection onto the face plane is effectivelly on the face or not.
/// Return FALSE if not and TRUE only if the point is in the interior of the normal and its projection in ON the face.
//////////////////////////////////////////////////////////////////////
bool
COME_Face::isInsideGlobalGOOD( COME_Point3D pPos ){

	COME_Vector3D direction;
	direction.setVector3D( pPos - getVertexGlobalPositionPt( 0 )->getVertexAsPoint3D() );
	direction.vpNormalize();
	if( normalGlobalPosition.vpDotProduct( direction ) >= 0 ) return false;
	
	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double distPointPlan = normalGlobalPosition.vpDotProduct( p_a );

	COME_Point3D pPosPr = pPos - ( normalGlobalPosition * distPointPlan );
	
	if( !isInside2DGlobal( pPosPr ) ) return false;

	return true;
}


//////////////////////////////////////////////////////////////////////
/// Check if the given point is inside the mesh to which this face belongs.
/// Assume that if it is in the exterior side of the face (use normal to check)  it is OUTSIDE otherwise its INSIDE.
/// Use local coordinates for the face.
//////////////////////////////////////////////////////////////////////
bool
COME_Face :: isInside( COME_Point3D currentPoint ){

	COME_Vector3D direction;
	vector<COME_Vertex3D> vert = getVertices();
	direction.setVector3D( currentPoint - vert[ 0 ] );

	direction.vpNormalize();
	if( getNormal().vpDotProduct( direction ) < 0 ){

		return true; //inside
	} else {
		
		return false;
	}

}


//////////////////////////////////////////////////////////////////////
/// Return true if the index v belongs to this face and false otherwise.
//////////////////////////////////////////////////////////////////////
bool
COME_Face::containsVertexIndex( int v ){

	for( int i = 0; i < vertices.size(); i++ ){

		if( vertices[i] == v ){

			return true;
		}
	}
	return false;
}

//////////////////////////////////////////////////////////////////////
/// Replace index v1 by index v2 in this face.
/// This method is used to change the vertex of one triangle by the vertex
/// of another. It is used, for example, in the edge-collapse algorithm.
//////////////////////////////////////////////////////////////////////
void
COME_Face::replaceIndex( int v1, int v2 ){

	for( int i = 0; i < vertices.size(); i++ ){

		if( vertices[i] == v1 ){

			vertices[i] = v2;
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// This method uses the last displacement vector of the molecule associated
/// to this face to displace all vertices of this face.
//////////////////////////////////////////////////////////////////////
void
COME_Face::update(){

	vector<COME_Vertex3D> vert = getVertices();
	COME_Vector3D displacement = nearestMolecule->getLastDisplacement();
	for ( int i = 0; i < vert.size(); i++ ){

		vert[i] += displacement;
		((COME_Mesh*)parent)->setAVertex( vert[ i ], vertices[ i ] );
	}

}

void
COME_Face:: setNeighbourFaces( vector<int> faces){
	neighbourFaces = faces;
}

vector <int>
COME_Face:: getNeighbourFaces() const {
	return neighbourFaces;
}

void
COME_Face:: addNeighbour( int faceIndex){
	neighbourFaces.push_back(faceIndex);
}


bool
COME_Face::isCompletelyInside(COME_Face* face2){
	
	vector<COME_Vertex3D*> vertices = face2->getVerticesGlobalPositionPt();
	int in=0;
	for (int i = 0; i < vertices.size(); i++ ){
		//cout << "Vertex to point (x,y,z) = " << vertices[i].getPoint3D().x << "," << vertices[i].getPoint3D().y << "," << vertices[i].getPoint3D().z << "\n";
		if (!isInsideGlobalPosition(vertices[i]->getVertexAsPoint3D())){
			return false;
		}		
	}
	return true;
}


/// ray starts from origin and is given by the vector
/// algorithm supposes the origin is outside the face.
/// return (0, 0, 0) if no intersection found
bool
COME_Face::intersectionRayTriangle( COME_Vector3D vRay, COME_Point3D &intersection ){

	COME_Vector3D p_a = COME_Point3D() - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double distPointPlan = normalGlobalPosition.vpDotProduct( p_a );

	vRay.vpNormalize();
	
	double cosinus = vRay.vpDotProduct( -normalGlobalPosition );
	if( cosinus <= 0.0 ){
		intersection = COME_Point3D();
		return false;
	}
	
	double distOPlanRay = distPointPlan / cosinus;
	
	intersection = vRay * distOPlanRay;
	
	if( !isInside2DGlobal( intersection ) ){
		intersection = COME_Point3D();
		return false;
	}
	
	return true;
}


COME_Point3D
COME_Face::getClosestPointOnFaceToPoint( COME_Point3D pPos ){

	COME_Vector3D p_a = pPos - COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
	double distPointPlan = normalGlobalPosition.vpDotProduct( p_a );

	COME_Point3D pPosPr = pPos - ( normalGlobalPosition * distPointPlan );
	
	if( !isInside2DGlobal( pPosPr ) ){
		
		COME_Point3D pA, pB, pC;
		pA = COME_Vector3D( getVertexGlobalPositionPt( 0 )->x, getVertexGlobalPositionPt( 0 )->y, getVertexGlobalPositionPt( 0 )->z );
		pB = COME_Vector3D( getVertexGlobalPositionPt( 1 )->x, getVertexGlobalPositionPt( 1 )->y, getVertexGlobalPositionPt( 1 )->z );
		pC = COME_Vector3D( getVertexGlobalPositionPt( 2 )->x, getVertexGlobalPositionPt( 2 )->y, getVertexGlobalPositionPt( 2 )->z );
		
		COME_Vector3D a_b = pB  - pA;
		a_b.vpNormalize();
		COME_Vector3D b_c = pC  - pB;
		b_c.vpNormalize();
		COME_Vector3D c_a = pA  - pC;
		c_a.vpNormalize();
		
		/// Project on lines formed by the edges
		COME_Point3D pOnEdgeAB = pA + ( a_b * ( (COME_Vector3D( pPosPr - pA )).vpDotProduct(a_b) )  );
		COME_Point3D pOnEdgeBC = pB + ( b_c * ( (COME_Vector3D( pPosPr - pB )).vpDotProduct(b_c) )  );
		COME_Point3D pOnEdgeCA = pC + ( c_a * ( (COME_Vector3D( pPosPr - pC )).vpDotProduct(c_a) )  );
		
		/// Project on vertices if out of edges
		double EPSILON = 0.0000000001;
		if( ( ( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() )
		||  ( ( (COME_Vector3D(pB-pOnEdgeAB)).vpModule() + EPSILON ) > (COME_Vector3D(pA-pB)).vpModule() ) ){
			if( ( COME_Vector3D(pA-pOnEdgeAB)).vpModule() < (COME_Vector3D(pB-pOnEdgeAB)).vpModule() )
				pOnEdgeAB = pA;
			else pOnEdgeAB = pB;
		}
		if( ( ( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() )
		||  ( ( (COME_Vector3D(pC-pOnEdgeBC)).vpModule() + EPSILON ) > (COME_Vector3D(pB-pC)).vpModule() ) ){
			if( ( COME_Vector3D(pB-pOnEdgeBC)).vpModule() < (COME_Vector3D(pC-pOnEdgeBC)).vpModule() )
				pOnEdgeBC = pB;
			else pOnEdgeBC = pC;
		}
		if( ( ( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() )
		||  ( ( (COME_Vector3D(pA-pOnEdgeCA)).vpModule() + EPSILON ) > (COME_Vector3D(pC-pA)).vpModule() ) ){
			if( ( COME_Vector3D(pC-pOnEdgeCA)).vpModule() < (COME_Vector3D(pA-pOnEdgeCA)).vpModule() )
				pOnEdgeCA = pC;
			else pOnEdgeCA = pA;
		}
		
		double distProj = pPos.vpDistance( pOnEdgeAB );
		COME_Point3D pPosOnEdge = pOnEdgeAB;
		if( distProj > pPos.vpDistance( pOnEdgeBC ) ){
			distProj = pPos.vpDistance( pOnEdgeBC );
			pPosOnEdge = pOnEdgeBC;
		}
		if( distProj > pPos.vpDistance( pOnEdgeCA ) ){
			distProj = pPos.vpDistance( pOnEdgeCA );
			pPosOnEdge = pOnEdgeCA;
		}
		return pPosOnEdge;
	}
	
	return pPosPr;
}

COME_Face *
COME_Face::getNeighborClosestTo( COME_Vertex3D *vPos ){

	COME_Face * fClosest = this;
	double distMin = this->distancePointFaceGlobal(  vPos->getVertexAsPoint3D() );
	
	for( int iNF = 0; iNF < neighbourFaces.size(); iNF++ ){
	
		if( ((COME_Mesh*)(parent))->getAFacePt( neighbourFaces[iNF] )->distancePointFaceGlobal( vPos->getVertexAsPoint3D() ) < distMin ){
			distMin = ((COME_Mesh*)(parent))->getAFacePt( neighbourFaces[iNF] )->distancePointFaceGlobal( vPos->getVertexAsPoint3D() );
			fClosest = ((COME_Mesh*)(parent))->getAFacePt( neighbourFaces[iNF] );
		}
		//printf( "%p dist Neighbor: %f min: %f index neighbor:%d \n", this, ((COME_Mesh*)(parent))->getAFacePt( neighbourFaces[iNF] )->distancePointFaceGlobal( vPos->getVertexAsPoint3D() ), distMin, iNF );
	}
	return fClosest;
}

COME_Vector3D
COME_Face::getBlendedVelocity( COME_Point3D pPos ){

	COME_Vector3D velocBlended;
	
	COME_Vertex3D *A =  getVertexGlobalPositionPt( 0 );
	COME_Vertex3D *B =  getVertexGlobalPositionPt( 1 );
	COME_Vertex3D *C =  getVertexGlobalPositionPt( 2 );
	
	double m1 = COME_Face::triangleArea( B->getVertexAsPoint3D(), C->getVertexAsPoint3D(), pPos );
	double m2 = COME_Face::triangleArea( C->getVertexAsPoint3D(), A->getVertexAsPoint3D(), pPos );
	double m3 = COME_Face::triangleArea( A->getVertexAsPoint3D(), B->getVertexAsPoint3D(), pPos );
	
	double sumMasses = m1 + m2 + m3;
	
	double w1 = m1 / sumMasses;
	double w2 = m2 / sumMasses;
	double w3 = m3 / sumMasses;
	
	velocBlended = ( A->getBlendedVelocity() * w1) + ( B->getBlendedVelocity() * w2) + ( C->getBlendedVelocity() * w3);
	
	return velocBlended;
}

	/* // old and imprecise version
	vector<COME_Vertex3D*> verticesFace = getVerticesGlobalPositionPt(); 

	COME_Vector3D unitAB;
	unitAB.setVector3D( *(verticesFace[1]) - *(verticesFace[0]) );
	unitAB.vpNormalize();
	
	COME_Vector3D unitAC;
	unitAC.setVector3D( *(verticesFace[2]) - *(verticesFace[0]) );
	unitAC.vpNormalize();

	COME_Vector3D AP;
	AP.setVector3D( pPos - *(verticesFace[0]) );

	COME_Vector3D w1 =  unitAB * AP.vpDotProduct( unitAB );
	COME_Vector3D w2 =  (normalGlobalPosition * w1) * AP.vpDotProduct( unitAC );

	COME_Point3D pointPlane = *(verticesFace[0]) + w1 + w2;
	
	return pointPlane;*/
