/***************************************************************************
 *   Copyright (C) 2001 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
///
///  PROJECT.....: CO-ME
///  RESPONSIBLE.: 
///
///  AUTHOR......: Anderson Maciel
//   DATE........: June/05/2001
///  DESCRIPTION.: Class declaration.
///
////////////////////////////////////////////////////////////////////

#include <algebra/comevertex3d.h>
#include <algebra/comematrix.h>
#include <bio/comemolecule.h>
#include <algebra/comemesh.h>
#include <math.h>


///////////////////////////////////////////////////////////////////
// Description: Class "COME_Vertex3D" constructor without parameter.
// Parameters.: -
// Return.....: -

COME_Vertex3D::COME_Vertex3D( void ){
	x = 0;
    y = 0;
    z = 0;
	color[0] = .9;
	color[1] = .7;
	color[2] = .3;
	collide = true;
	ownerFace = 0;
	//collisionCount = 0;
	//velocityCount = 0;
}

///////////////////////////////////////////////////////////////////
// Description: Class "COME_Vertex3D" constructor with parameters.
// Parameters.: double xi (initial x value), 
//				double yi (initial y value),
//				double zi (initial z value).
// Return.....: -

COME_Vertex3D::COME_Vertex3D( double xi, double yi, double zi) {
	x = xi;
    y = yi;
    z = zi;
	color[0] = .9;
	color[1] = .7;
	color[2] = .3;
	collide = true;	
	ownerFace = 0;
	//collisionCount = 0;
	//velocityCount = 0;
}

COME_Vertex3D::COME_Vertex3D( double xi, double yi, double zi, COME_Vector3D n) {
	x = xi;
    y = yi;
    z = zi;
	normal = n;
	color[0] = .9;
	color[1] = .7;
	color[2] = .3;
	collide = true;	
	ownerFace = 0;
	//collisionCount = 0;
	//velocityCount = 0;
}



///////////////////////////////////////////////////////////////////
// Description: Class "COME_Vertex3D" constructor with parameters.
// Parameters.: COME_Vertex3D (object).
// Return.....: -

COME_Vertex3D::COME_Vertex3D(const COME_Vertex3D &v) {
	x = v.x ;
    y = v.y ;
    z = v.z ;
	normal = v.normal;
	normalGlobalPosition = v.normalGlobalPosition;
	texCoord = v.texCoord;

	color[0] = v.color[0];
	color[1] = v.color[1];
	color[2] = v.color[2];
	color[3] = v.color[3];
	
	ownerFace = v.ownerFace;
	anchors = v.anchors;
	localPositions = v.localPositions;
	
	pPlane = v.pPlane;
	collide = v.collide;
	collide = true;
	//collisionCount = v.collisionCount;
	//velocityCount = v.velocityCount;
}

///////////////////////////////////////////////////////////////////
// Description: Class "COME_Vertex3D" constructor with parameters.
// Parameters.: COME_Vertex3D (object).
// Return.....: -

COME_Vertex3D::COME_Vertex3D( COME_Vertex3D *v ) {
	x = v->x ;
    y = v->y ;
    z = v->z ;
	normal = v->normal;
	normalGlobalPosition = v->normalGlobalPosition;

	color[0] = v->color[0];
	color[1] = v->color[1];
	color[2] = v->color[2];
	color[3] = v->color[3];
	
	ownerFace = v->ownerFace;
	anchors = v->anchors;
	localPositions = v->localPositions;
	
	pPlane = v->pPlane;
	collide = v->collide;
	collide = true;
	//collisionCount = v->collisionCount;
	//velocityCount = v->velocityCount;
}


///////////////////////////////////////////////////////////////////
// Description: Operator=
// Parameters.: COME_Vertex3D (object).
// Return.....: COME_Vertex3D (object).

COME_Vertex3D& COME_Vertex3D :: operator= ( COME_Vertex3D& v ) {

	if ( this == &v ) return ( *this ) ;
	
	x = v.x ;
	y = v.y ;
	z = v.z ;
	anchors=v.anchors;
	normal=v.normal;
	normalGlobalPosition=v.normalGlobalPosition;
	localPositions = v.localPositions;
	neighbourFaces= v.neighbourFaces;
	ownerFace = v.ownerFace;
	color[0] = v.color[0];
	color[1] = v.color[1];
	color[2] = v.color[2];
	color[3] = v.color[3];
	
	pPlane = v.pPlane;
	collide = collide;
	//collisionCount = v.collisionCount;
	//velocityCount = v.velocityCount;
	
	return ( *this ) ;
}


///////////////////////////////////////////////////////////////////
// Description: Operator=
// Parameters.: COME_Vector3D (object).
// Return.....: COME_Vertex3D (object).

COME_Vertex3D& COME_Vertex3D :: operator= ( const COME_Vector3D& v ) {
      if ( this == (COME_Vertex3D *)&v ) return ( *this ) ;
      x = v.x ;
      y = v.y ;
      z = v.z ;
      return ( *this ) ;
}

///////////////////////////////////////////////////////////////////
// Description: Operator+=
// Parameters.: COME_Vector3D (object).
// Return.....: COME_Vertex3D (object).

COME_Vertex3D& COME_Vertex3D :: operator+= ( const COME_Vector3D& v ) {
      if ( this == (COME_Vertex3D *)&v ) return ( *this ) ;
      x += v.x ;
      y += v.y ;
      z += v.z ;
      return ( *this ) ;
}


void
COME_Vertex3D::setNormal( COME_Vector3D n ){

	normal = n;
}

void
COME_Vertex3D::setCollisionDisp( COME_Vector3D c ){

	collisionDisp = c;
	/*if( ( c.x == 0.0 ) && ( c.y == 0.0 ) && ( c.z == 0.0 ) )
		collisionCount = 0;
	else
		collisionCount = 1;*/
}

void
COME_Vertex3D::setVelocityDisp( COME_Vector3D c ){

	velocityDisp = c;
	/*if( ( c.x == 0.0 ) && ( c.y == 0.0 ) && ( c.z == 0.0 ) )
		velocityCount = 0;
	else
		velocityCount = 1;*/
}

void
COME_Vertex3D::addCollisionDisp( COME_Vector3D c ){

	//collisionDisp += c;
	//collisionCount++;
	
	if( c.vpModule() > collisionDisp.vpModule() ) collisionDisp = c;
}

void
COME_Vertex3D::addVelocityDisp( COME_Vector3D c ){

	//velocityDisp += c;
	//velocityCount++;
	
	if( c.vpModule() > velocityDisp.vpModule() ) velocityDisp = c;
}

double*
COME_Vertex3D::getColor( void ){
	return color;
}

void 
COME_Vertex3D::setColor (double *c){
	color[0] = c[0];
	color[1] = c[1];
	color[2] = c[2];
	color[3] = c[3];
}
void
COME_Vertex3D::setNormalGlobalPosition( COME_Vector3D n ){

	normalGlobalPosition = n;
}

void
COME_Vertex3D::setOwnerFace( int index ){

	ownerFace = index;
}

void
COME_Vertex3D::setTexCoord( COME_Point2D newUV ){

	texCoord = newUV;
}

void
COME_Vertex3D::setCollide( bool yesno ){
	
	collide = yesno;
}

void
COME_Vertex3D::addAnchor( COME_Molecule* newAnchor ){

	anchors.push_back( newAnchor );
	//newAnchor->addBuoy( this );
}

void
COME_Vertex3D::addLocalPosition( COME_Point3D newPosition ){

	localPositions.push_back( newPosition );
}

bool
COME_Vertex3D::getCollide(){
	
	return collide;
}

COME_Vector3D
COME_Vertex3D::getNormal( void ){

	return normal;
}

COME_Vector3D
COME_Vertex3D::getCollisionDisp( void ){

	return collisionDisp;
}

COME_Vector3D
COME_Vertex3D::getVelocityDisp( void ){

	return velocityDisp;
}

COME_Vector3D
COME_Vertex3D::getCollisionDispAvg( void ){

	//if( collisionCount == 0 ) return COME_Vector3D();
	//printf( "dispCollision: %f %f %f count %d \n", collisionDisp.x, collisionDisp.y, collisionDisp.z, collisionCount );
	return collisionDisp;// / (double)collisionCount;
}

COME_Vector3D
COME_Vertex3D::getBlendedVelocity( void ){

	COME_Vector3D blendedVelocity;
	double sumDists = 0.0;

	int i;
	for( i = 0; i < localPositions.size(); i++ ){

		sumDists += pow( localPositions[i].vpDistance( COME_Point3D() ), 2.0 );
	}
	
	for( int i = 0; i < localPositions.size(); i++ ){

		double wi = ( 1.0 - ( ( pow( localPositions[i].vpDistance( COME_Point3D() ), 2.0 ) ) / sumDists ) ) / (double)(localPositions.size()-1);
		blendedVelocity = blendedVelocity + ( anchors[i]->getVelocity() * wi );
	}

	return blendedVelocity;
}

COME_Vector3D
COME_Vertex3D::getVelocityDispAvg( void ){

	//if( velocityCount == 0 ) return COME_Vector3D();
	//printf( "dispVelocity: %f %f %f count %d \n", velocityDisp.x, velocityDisp.y, velocityDisp.z, velocityCount );
	return velocityDisp; // / (double)velocityCount;
}

COME_Vector3D*
COME_Vertex3D::getNormalGlobalPositionPt( void ){

	return &normalGlobalPosition;
}

COME_Vector3D
COME_Vertex3D::getNormalGlobalPosition( void ){

	return normalGlobalPosition;
}

int
COME_Vertex3D::getOwnerFace( void ){

	return ownerFace;
}

vector<COME_Molecule*>&
COME_Vertex3D::getAnchors( void ){

	return anchors;
}

vector<COME_Molecule*>*
COME_Vertex3D::getAnchorsPt( void ){

	return &anchors;
}

vector<COME_Point3D>&
COME_Vertex3D::getLocalPositions( void ){

	return localPositions;
}

COME_Point2D
COME_Vertex3D::getTexCoord( void ){

	return texCoord;
}

bool
COME_Vertex3D::hasFixedAnchor( void ){

	for( int i = 0; i < anchors.size(); i++ ){

		if( anchors[i]->isFixed() )
			return true;
	}

	return false;
}

double
COME_Vertex3D::getStress( void ){

	double localStress = 0.0;
	for( int i = 0; i < anchors.size(); i++ ){

		localStress += anchors[i]->getStress();
	}

	return ( localStress / ((double) anchors.size() ) );
}

double
COME_Vertex3D::getStrain( void ){

	double localStrain = 0.0;
	for( int i = 0; i < anchors.size(); i++ ){

		localStrain += anchors[i]->getStrain();
	}

	return ( localStrain / ((double) anchors.size() ) );
}

COME_Vector3D
COME_Vertex3D::getColorMaterial( void ){

	if( anchors.size() ){

		return anchors[0]->getMaterial()->getColor();
	}

	return COME_Vector3D(1, 1, 1);
}

///////////////////////////////////////////////////////////////////
/// Blends all local points to compose the current position of this vertex.
///////////////////////////////////////////////////////////////////
COME_Point3D
COME_Vertex3D::getBlendedPosition( void ){

	COME_Point3D blendedPoint;
	double w = 0;

	for( int i = 0; i < localPositions.size(); i++ ){

		double wi = ( 1.0 / localPositions[i].vpDistance( COME_Point3D() ) );
		blendedPoint = blendedPoint + ( ( (*anchors[i]->getLocalFrame()) * localPositions[i] ) * wi );
		w += wi;
	}

	return ( blendedPoint * (1.0/w) );
}

void
COME_Vertex3D:: addNeighbour(int faceIndex, COME_Mesh *theMesh ){
	
	neighbourFaces.push_back(faceIndex);
	
	COME_Face *theFace = theMesh->getAFacePt( faceIndex );
	for( int iF = 0; iF < theFace->getNumberVertices(); iF ++ ){
	
		COME_Vertex3D* theVertex = theFace->getVertexGlobalPositionPt(iF);
		if(  theVertex != this ){
			neighborVerts.push_back( theVertex );
		}
	}
}

vector<int>
COME_Vertex3D:: getNeighbourFaces(){
	return neighbourFaces;
}

vector<COME_Vertex3D *>
COME_Vertex3D:: getNeighborVerts(){
	return neighborVerts;
}

COME_Point3D
COME_Vertex3D::getVertexAsPoint3D(){

	return COME_Point3D( x, y, z );
}

bool
COME_Vertex3D::anchorsCollide(){

	for( int i = 0; i < anchors.size(); i++ ){
	
		if( anchors[i]->isCollide() )
			return true;
	}
	return false;
	
}

COME_Vertex3D *
COME_Vertex3D:: getNeighborClosestTo( COME_Vertex3D *vPos ){

	double distMin = this->vpDistance( vPos->getVertexAsPoint3D() );
	COME_Vertex3D * vClosest = this;
	
	for( int iNV = 0; iNV < neighborVerts.size(); iNV++ ){
	
		if( neighborVerts[iNV]->vpDistance( vPos->getVertexAsPoint3D() ) < distMin ){
			distMin = neighborVerts[iNV]->vpDistance( vPos->getVertexAsPoint3D() );
			vClosest = neighborVerts[iNV];
		}
	}
	return vClosest;
}
