/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

///////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  FILE........: comebiostructure.h
//  DESCRIPTION.: 
//				   
//
//  AUTHOR......: Anderson Maciel
//  DATE........: May/27/2002
//  DESCRIPTION.: Class declaration.
//
///////////////////////////////////////////////////////////////////

#ifndef __COME_BIOSTRUCTURE_H	
#define __COME_BIOSTRUCTURE_H	

#include	<general/come.h>
#include	<algebra/comemesh.h>
#include	<algebra/comematrix.h>
#include	<bio/comemoleculestissue.h>


#ifndef	__COME_SCENARIO_H
class COME_Scenario;
#endif

typedef struct HashElement {
	
	COME_Face	*face;
	double		distance;
} COME_HashElement;

#define	elem(i,j)	((i*hashSize)+j)

///////////////////////////////////////////////////////////////////
/// An abstract class to represent one organ (biological structure)
///	of a patient.
/// It must be implemented by adding specific features and behavior
/// for each organ in the subclass.
///////////////////////////////////////////////////////////////////

class COME_BioStructure : public COME {

protected:
	int			system;
	double			volume;
	double			collisionRadius; /// Radius of the enclosing sphere defining collision volume
	COME_Mesh		*surface;
	bool			fixed;
	int				type;
	double			globalHooke;
	COME_Matrix		localInstanceMatrix;
	COME_Matrix		globalInstanceMatrix;
	

public:

	void	setSystem( int systemN );
	void	setSurface( COME_Mesh *surfaceN );
	void	setFixed( bool flag );
	void	setLIM( const COME_Matrix &limN );
	void	setGIM( const COME_Matrix &gimN );
	void	setGlobalHooke( double hookN );	
	void	setCollisionRadius( double rad );
	void	setType( int nType );
	
	int		getSystem() const;
	COME_Mesh*	getSurface();
	COME_Mesh*	getPointerToSurface();
	double		getGlobalHooke() const;
	double		getCollisionRadius() const;
	double		getTopStress();
	void 		scale( double factorx, double factory, double factorz );
	void 		translate( double px, double py, double pz );
	void 		translate( const COME_Vector3D &disp );
	void 		rotate( double rx, double ry, double rz );
	void		scaleNominals( double factor );

	bool			isFixed() const;
	double			getMass() const;
	const COME_Matrix&	getLIM() const;
	const COME_Matrix&	getGIM() const;

	virtual	bool	update( double timestep, double simClock ) = 0;
	virtual void	updateSkin() = 0;
	virtual void	updateSurface( bool regenerate ) = 0;
	//virtual void	initializeSkinning() = 0;

	virtual void	getEnvelop( COME_Point3D& mins, COME_Point3D& maxs ) = 0;

	virtual COME_MoleculesTissue	*getTissue() = 0;
	virtual void			respondCollision() = 0;
	
	void				initializeHashStructure();
	void				initializeHashStructureRayCasting();
	COME_HashElement	getHashElement( COME_Point2D p );
	int				fillHashHoles();
	
	int						hashSize;
	COME_HashElement			*hashTable;
	vector<COME_BioStructure*>	internals;
	vector<string>				internalsDesc;
	
	COME_Vector3D			sUnit;
	COME_Point2D				sSpherical;

	void	globalToLocal( COME_Matrix M, COME_Matrix MP );

};

#endif // __COME_BIOSTRUCTURE_H
