/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
///
///  PROJECT.....: CO-ME
///  RESPONSIBLE.: 
///
///  AUTHOR......: Anderson Maciel
///  DATE........: July/30/2002
///  DESCRIPTION.: Class definition.
///
////////////////////////////////////////////////////////////////////

#include <bio/comemolecule.h>
#include <bio/comemoleculescartilage.h>
#include <general/comescenario.h>
#include <math.h>
#include <algebra/quaternion.h>

//////////////////////////////////////////////////////////////////////
/// Default constructor.
//////////////////////////////////////////////////////////////////////

COME_Molecule::COME_Molecule()
{

	connectionList = new list<COME_MoleculeLink*>;
	parent = NULL;
	localFrame = NULL;
	fixed = false;
	liquidSharingArray = NULL;
	intersecClamp = false;
	collide = false;
	updated = true;
	stress = 0.0;
}

//////////////////////////////////////////////////////////////////////
/// Complete constructor.
//////////////////////////////////////////////////////////////////////

COME_Molecule::COME_Molecule( COME_Point3D positionN, double radiusN, double frictionConstN,
							  COME_Material *materialN, COME_Vector3D velocityN, COME_BioStructure *parentN ){

	nominalRadius = radius = radiusN;
	frictionConst = frictionConstN;
	parent = parentN;
	fixed = false;
	onSurface = false;
	localFrame = NULL;
	initPosition = positionN;

	setPosition( positionN );
	setVelocity( velocityN );
	
	if( materialN ){
		//setVolume( 4 * M_PI * pow( radius, 3 ) / 3 );
		setMass( getVolume() * materialN->getDensity() );
		setMaterial( materialN );
		liquidFraction = materialN->getLiquidFraction();
		permeability = materialN->getPermeability();
	} else {
		setMass( 0.0 );
		setMaterial( NULL );
		liquidFraction = 0.0;
		permeability = 0.0;
	}

	connectionList = new list<COME_MoleculeLink*>;

	liquidSharingArray = NULL;
	intersecClamp = false;
	collide = false;
	updated = true;
	stress = 0.0;
}

COME_Molecule::COME_Molecule( COME_Point3D &positionN, double &radiusN, double &frictionConstN, COME_Material *materialN, COME_Vector3D &velocityN, COME_BioStructure *parentN ){
				   
	//COME_Molecule( positionN, radiusN, frictionConstN, materialN, velocityN, parentN );
	nominalRadius = radius = radiusN;
	frictionConst = frictionConstN;
	parent = parentN;
	fixed = false;
	onSurface = false;
	localFrame = NULL;
	initPosition = positionN;

	setPosition( positionN );
	setVelocity( velocityN );
	
	if( materialN ){
		//setVolume( 4 * M_PI * pow( radius, 3 ) / 3 );
		setMass( getVolume() * materialN->getDensity() );
		setMaterial( materialN );
		liquidFraction = materialN->getLiquidFraction();
		permeability = materialN->getPermeability();
	} else {
		setMass( 0.0 );
		setMaterial( NULL );
		liquidFraction = 0.0;
		permeability = 0.0;
	}

	connectionList = new list<COME_MoleculeLink*>;

	liquidSharingArray = NULL;
	intersecClamp = false;
	collide = false;
	updated = true;
	stress = 0.0;
}


COME_Molecule::~COME_Molecule()
{
	if( connectionList ){
	
		list<COME_MoleculeLink*>::iterator iterThis;
		for (iterThis = connectionList->begin(); !( iterThis == connectionList->end() ); iterThis++){
		
			if( (*iterThis) ){
				list<COME_MoleculeLink*>::iterator iterOther;
				for (iterOther = (*iterThis)->getOtherElement(this)->getConnectionList()->begin(); !( iterOther == (*iterThis)->getOtherElement(this)->getConnectionList()->end() ); iterOther++){
					if( (*iterThis) == (*iterOther) ){

						(*iterThis)->getOtherElement(this)->getConnectionList()->erase( iterOther++ );
					}
				}
				delete (*iterThis);
			}
		}
		delete connectionList;
	}
}

//////////////////////////////////////////////////////////////////////
/// Setting
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::setRadius( double radiusN ){

	radius = radiusN;
}


void
COME_Molecule::setFrictionConst( double frictionConstN ){

	frictionConst = frictionConstN;
}

void
COME_Molecule::setConnectionList( list<COME_MoleculeLink*> *listN ){

	connectionList = listN;
}

void
COME_Molecule::setExternalForces( vector<COME_TimeForce*> listN ){

	externalForces = listN;
}



void
COME_Molecule::setOnSurface( bool flag ){

	onSurface = flag;
}

void
COME_Molecule::setCollisionForce( COME_Force forceN ){

	collisionForce = forceN;
}

void
COME_Molecule::setIfGreaterCollisionForce( COME_Force forceN ){

	if( collisionForce.getIntensity() < forceN.getIntensity() )
		collisionForce = forceN;
}
	

void
COME_Molecule::setIntersectClamp( int type ){

	intersecClamp = type;
}

void
COME_Molecule::setCollide( bool yesno ){

	collide = yesno;
}

void
COME_Molecule::setUpdated( bool yesno ){

	updated = yesno;
}


//////////////////////////////////////////////////////////////////////
/// Getting
//////////////////////////////////////////////////////////////////////
double
COME_Molecule::getRadius() const{

	return radius;
}

double
COME_Molecule::getFrictionConst() const{

	return frictionConst;
}

list<COME_MoleculeLink*>*
COME_Molecule::getConnectionList() const{

	return connectionList;
}

vector<COME_TimeForce*>
COME_Molecule::getExternalForces() const{

	return externalForces;
}

vector<COME_Vertex3D*>*
COME_Molecule::getBuois() {

	return &buois;
}

vector<int>*
COME_Molecule::getBuoisIndices() {

	return &buoisIndices;
}

bool
COME_Molecule::isOnSurface() const{

	return onSurface;
}

bool
COME_Molecule::isCollide() const{

	return collide;
}

bool
COME_Molecule::isUpdated() const{

	return updated;
}

bool
COME_Molecule::intersectClamp() const{

	if( intersecClamp == FREE )
		return false;
	return true;
}

bool
COME_Molecule::intersectClampMobile() const{

	if( intersecClamp == CLAMPMOBILE )
		return true;
	return false;
}

bool
COME_Molecule::intersectClampFix() const{

	if( intersecClamp == CLAMPFIX )
		return true;
	return false;
}

COME_Force
COME_Molecule::getCollisionForce() const{

	return collisionForce;
}

COME_Vector3D
COME_Molecule::getLastDisplacement() const{

	return lastDisplacement;
}

vector<COME_Molecule*>&
COME_Molecule::getOrthogonalNeighbors(){

	return orthogonalNeighbors;
}

COME_Matrix*
COME_Molecule::getLocalFrame(){

	return localFrame;
}

double
COME_Molecule::getVolume() const {

	return 4.0 * M_PI * pow( radius, 3 ) / 3.0;
}

double
COME_Molecule::getPressureArea() const {

	return M_PI * radius * radius;
}

double
COME_Molecule::getLiquidLevel() const{

	double solidVolume = ( ( 4.0 * M_PI * pow( nominalRadius, 3) ) / 3.0 ) * (1 - liquidFraction );

	if( liquidFraction == 0.0 ) return 0.0;

	return ( getVolume() - solidVolume ) / ( ( 4.0 * M_PI * pow( nominalRadius, 3) ) / 3.0 * liquidFraction );
}

double
COME_Molecule::getLiquidFraction() const{

	return liquidFraction;

}
double
COME_Molecule::getNominalRadius() const{

	return nominalRadius;
}

double
COME_Molecule::getPermeability() const{

	return permeability;
}


//////////////////////////////////////////////////////////////////////
/// This method adds linkN to the end of this molecule list of connections.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::addLink( COME_MoleculeLink *linkN ){

	connectionList->push_back( linkN );
}

//////////////////////////////////////////////////////////////////////
/// This method adds linkN to the end of this molecule list of connections.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::addBuoy( COME_Vertex3D *buoyN, int index ){

	buois.push_back( buoyN );
	buoisIndices.push_back( index );
}


//////////////////////////////////////////////////////////////////////
/// This method adds .
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::addOrthogonalNeighbor( COME_Molecule *newNeighbor ){

	orthogonalNeighbors.push_back( newNeighbor );
}

//////////////////////////////////////////////////////////////////////
/// This method adds forceN to the end of this molecule list of external forces.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::addExternalForce( COME_TimeForce *forceN ){

	externalForces.push_back( forceN );
}

//////////////////////////////////////////////////////////////////////
/// This method clears list of external forces.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::clearExternalForces(){

	for( int i = 0; i < externalForces.size(); i++ ){
		delete externalForces[i];
	}
	externalForces.clear();
}

/// Accumulate the given force with other contact's forces
void
COME_Molecule::addCollisionForce( COME_Force forceN ){

	collisionForce = collisionForce + forceN;
}

//////////////////////////////////////////////////////////////////////
/// This method creates a connection between this molecule and moleculeN.
/// A reference to the connection created will be added to the lists of
/// connections of both molecules (this and moleculeN). All connection
/// parameters will be calculated from the ones of molecules.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::createLink( COME_Molecule *moleculeN ){

	// Create connection
	COME_MoleculeLink *newLink = new COME_MoleculeLink( this, moleculeN );
	
	// Set connection attributes
	newLink->setDampingConst( (getMaterial()->getDamping() + moleculeN->getMaterial()->getDamping())/2.0 );
	newLink->setFrictionConst( (getFrictionConst() + moleculeN->getFrictionConst())/2.0 );
	
	// Calculate nominal distance
	newLink->setNominalDist( getPosition().vpDistance( moleculeN->getPosition() ) );

	// Calculate Hooke's constant
	double area1 = pow(getRadius()*2, 2);
	double area2 = pow( moleculeN->getRadius()*2, 2);
	double hookeConstant = ( ( ( getMaterial()->getYoungsModulus()* area1 ) + ( moleculeN->getMaterial()->getYoungsModulus()* area2 ) ) / 2.0 ) / newLink->getNominalDist();
	newLink->setHookeConst( hookeConstant );

	printf( "Link's nominal dist: %f hooke's: %f \n", newLink->getNominalDist(), hookeConstant );
	
	// Add connection reference to each of its molecules
	addLink( newLink );
	moleculeN->addLink( newLink );
}

//////////////////////////////////////////////////////////////////////
/// This method creates a connection between this molecule and moleculeN.
/// A reference to the connection created will be added to the lists of
/// connections of both molecules (this and moleculeN). All connection
/// parameters will be calculated from global Hooke of the organ given.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::createLink( COME_Molecule *moleculeN, double globalHooke ){

	// Create connection
	COME_MoleculeLink *newLink = new COME_MoleculeLink( this, moleculeN );
	
	// Set connection attributes
	newLink->setDampingConst( (getMaterial()->getDamping() + moleculeN->getMaterial()->getDamping())/2.0 );
	newLink->setFrictionConst( (getFrictionConst() + moleculeN->getFrictionConst())/2.0 );
	
	// Calculate nominal distance
	newLink->setNominalDist( getPosition().vpDistance( moleculeN->getPosition() ) );

	// Calculate Hooke's constant
	COME_Vector3D aniso = material->getAnisotropyVector();
	COME_Vector3D kVec = position - moleculeN->getPosition();
	kVec.vpNormalize();
	double kThisLink = ( ( fabs(kVec.getX()) * aniso.getX() ) + ( fabs(kVec.getY()) * aniso.getY() ) + ( fabs(kVec.getZ()) * aniso.getZ() ) )* globalHooke;
	newLink->setHookeConst( kThisLink );
	
	// Add connection reference to each of its molecules
	addLink( newLink );
	moleculeN->addLink( newLink );
}

//////////////////////////////////////////////////////////////////////
/// This method remove the connection between this molecule and moleculeN.
/// The references to the connection in the lists of connections of both
/// molecules will be removed and the object connection will be destroied.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::removeLink( COME_Molecule *moleculeN ){

	list<COME_MoleculeLink*>::iterator iterThis;
	for (iterThis = connectionList->begin(); !( iterThis == connectionList->end() ); iterThis++){
		if( ( (*iterThis)->getElement(0) == moleculeN ) || ( (*iterThis)->getElement(1) == moleculeN ) ){
			break;
		}
	}
	list<COME_MoleculeLink*>::iterator iterNew;
	for (iterNew = moleculeN->getConnectionList()->begin(); !( iterNew == moleculeN->getConnectionList()->end() ); iterNew++){
		if( ( (*iterNew)->getElement(0) == this ) || ( (*iterNew)->getElement(1) == this ) ){
			break;
		}
	}
	delete (*iterThis);
	moleculeN->getConnectionList()->erase( iterNew );
	connectionList->erase( iterThis );
}

//////////////////////////////////////////////////////////////////////
/// This method calculates the resulting force being applied to this 
/// molecule at this moment. It is obtained by adding local forces,
/// global forces and connection forces.
//////////////////////////////////////////////////////////////////////
COME_Force
COME_Molecule::calculateForce( double currTime, COME_Point3D *positionsIN, COME_Vector3D *velocitiesIN, int currIndex ){

	double stressForce = 0.0;

	// Calculate local force (produced by the current amount of movement of the molecule) 
	COME_Vector3D velocNormalized;
	if( velocitiesIN[ currIndex ].vpModule() == 0 ){
		velocNormalized = COME_Vector3D( 0,0,0 );
	} else {
		velocNormalized = velocitiesIN[ currIndex ] / velocitiesIN[ currIndex ].vpModule();
	}
	fLocal = COME_Force( (velocNormalized)*(pow(velocitiesIN[ currIndex ].vpModule(), 2) )*(-M_PI)*(radius*radius)*material->getMediumDensity() );
	
	/// Calculates gravity force.
	COME_Vector3D accG = ((COME_Scenario*)(parent->getParent()->getParent()))->getGravity();
	fGravity = accG * mass;

	/// Calculates external force.
	fExternal = externalForceTo( currTime );

	// Calculate connections' force (produced by all connections existent with this molecule)
	// It is the addition of friction, damping and original forces of the connections.
	double strainTotal = 0.0;
	fOriginal = fDamping = fFriction = COME_Force();
	list<COME_MoleculeLink*>::iterator iterO;
	for (iterO = connectionList->begin(); iterO != connectionList->end(); iterO++ ){
		
		COME_Molecule *temp;
		if( (*iterO)->getElement(0) == this ){
			temp = (*iterO)->getElement(1);
		} else {
			temp = (*iterO)->getElement(0);
		}

		// Find index of molecule temp to get respectives position and velocity 
		int i = temp->getIndex();

		COME_Vector3D tempVelocity = velocitiesIN[ i ]; 
		COME_Point3D tempPosition = positionsIN[ i ];


		double dist = positionsIN[ currIndex ].vpDistance( tempPosition ) ;

		strainTotal = ( (*iterO)->getNominalDist() - dist ) / (*iterO)->getNominalDist();

		COME_Vector3D parallelVelocity;
		if( dist == 0 ){

			//printf( "DISTANCE: %f \n", dist );
		} else {

			//printf( "HOOKE: %f", (*iterO)->getHookeConst() );
			// Calculate original elasticity force
			COME_Force currentOriginal = COME_Force( ( ( positionsIN[ currIndex ] - tempPosition ) * (1/dist) ) * ( -(*iterO)->getHookeConst()*( dist - (*iterO)->getNominalDist() ) ) );
			fOriginal = COME_Force( fOriginal + ( currentOriginal /*/ 3.75*/ ) );
			stressForce += currentOriginal.getIntensity();
			
			// Calculate internal damping force
			COME_Vector3D P; P.setVector3D( positionsIN[ currIndex ] - tempPosition );
			COME_Vector3D V = COME_Vector3D( velocitiesIN[ currIndex ] - tempVelocity );
			parallelVelocity = COME_Vector3D( P * ( V.vpDotProduct( P ) / (dist * dist) ) );

			COME_Force currentDamping = COME_Force( parallelVelocity * ( -(*iterO)->getDampingConst() ) );
			fDamping = COME_Force( fDamping + currentDamping );
		}

		// Calculate friction force
		COME_Force fNormal = COME_Force( fOriginal + fDamping );
		COME_Vector3D orthogonalVelocity = COME_Vector3D( ( velocitiesIN[ currIndex ] - tempVelocity ) - parallelVelocity );
		COME_Vector3D miNormal = COME_Vector3D( fNormal.getForceVector() );
		miNormal = miNormal * frictionConst;
		COME_Vector3D orthogVelocityNormalized;
		if( orthogonalVelocity.vpModule() == 0 ){
			orthogVelocityNormalized = COME_Vector3D( 0,0,0 );
		} else {
			orthogVelocityNormalized = orthogonalVelocity / orthogonalVelocity.vpModule();
		}
		COME_Force currentFriction = COME_Force( orthogVelocityNormalized * ( -miNormal.vpModule() ) );
		fFriction = COME_Force( fFriction + currentFriction );
		stressForce += currentFriction.getIntensity();

	}

	/// Totalize connection's forces
	COME_Force fConnection = COME_Force( fOriginal + fDamping + fFriction );

	/// Calculate external friction
	COME_Vector3D direction = COME_Vector3D( velocitiesIN[ currIndex ] );
	double coeff = 0.2;
	COME_Force fExtFriction = direction.vpNormalize() * ( collisionForce.getIntensity() * -coeff );
	stressForce += fExtFriction.getIntensity();
	
	/// Totalize forces
	COME_Force resultant = COME_Force( fLocal + fConnection + fGravity + fExternal + collisionForce + fExtFriction );
	stressForce += fGravity.getIntensity() + fExternal.getIntensity();

	/// Calculate stress and strain
	stress = stressForce / ( M_PI * radius * radius );
	strain = strainTotal / connectionList->size();

	// Reset collision force
	fCollision = collisionForce; //keep thid force to chart plot
	collisionForce = COME_Force();

	return resultant;
}

//////////////////////////////////////////////////////////////////////
/// This method derivs the two sets of variables positions and velocities
/// into the two new sets velocities and accelerations. 
/// It calls the protected method calculateForce() to compute the final
/// force being applied on this molecule and apply Newton's laws to calculate
/// the resultant acceleration and the final velocity.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::derivs( double currTime, double timestep, COME_Point3D *positionsIN, COME_Vector3D *velocitiesIN, COME_Vector3D *velocitiesOUT, COME_Vector3D *accelerationsOUT, int currIndex ){
	
	if( !isFixed() ){
		updateLinks();
		#ifndef GPU_FORCES
			COME_Force force = calculateForce( currTime, positionsIN, velocitiesIN, currIndex );
		#else
			COME_Force force = ((COME_MoleculesTissue*)parent)->getGPUForce(currIndex);
		#endif
		acceleration = accelerationsOUT[ currIndex ] = COME_Vector3D( force.getForceVector() / mass );
		velocitiesOUT[ currIndex ] = COME_Vector3D( velocitiesIN[ currIndex ] + ( accelerationsOUT[ currIndex ] * timestep * 0.5 ) );
		// divide increment by 2 above to consider avg velocity
	}
}


//////////////////////////////////////////////////////////////////////
/// This method tests the distances of this molecule to the ones it is connected to. 
/// It calls the protected method removeLink() to remove the connections with 
/// molecules that are far than the maximum allowed distance before fracture away from this.
///	It returns the number of connections removed.
//////////////////////////////////////////////////////////////////////
int
COME_Molecule::updateLinks(){

	int removed = 0;
/*	list<COME_MoleculeLink*>::iterator iterO;
	for (iterO = connectionList->begin(); iterO != connectionList->end(); iterO++){
		
		COME_Molecule *temp;
		if( (*iterO)->getElement(0) == this ){
			temp = (*iterO)->getElement(1);
		} else {
			temp = (*iterO)->getElement(0);
		}
		double dist = position.vpDistance( temp->getPosition() ) ;
		if( dist > (*iterO)->getFractureDist() ){
			iterO++;
			removeLink( temp );
			removed++;
		}
		
	}
	if( removed ){
		COME_Point3D pos = getPosition();
		printf( "remotion - molecule: %f %f %f - %d connections.\n", pos.getX(), pos.getY(), pos.getZ(), removed );
	}
*/	return removed;
}

//////////////////////////////////////////////////////////////////////
/// This method linearly interpolates the vector of time-force of this molecule
/// to calculate the force to be applied in the current time.
//////////////////////////////////////////////////////////////////////
COME_Force
COME_Molecule::externalForceTo( double currTime ){

	if( externalForces.size() == 0 ){
		
		return COME_Force();

	} else {
		double iT = externalForces[0]->getTime();
		double fT = externalForces[externalForces.size()-1]->getTime();
		COME_Vector3D fF, iF;

		for( int i = 1; i < externalForces.size(); i++ ){

			if( currTime < externalForces[i]->getTime() ){
				fT = externalForces[i]->getTime();
				fF = externalForces[i]->getForceVector();
				iT = externalForces[i-1]->getTime();
				iF = externalForces[i-1]->getForceVector();
				break;
			}
		}

		COME_Force currForce = COME_Force( iF + ( ( fF - iF )*( ( currTime - iT ) / ( fT - iT ) ) ) );

		return currForce;
	}
}

//////////////////////////////////////////////////////////////////////
/// This overloaded method does more than just set a new value to position.
/// It uses old and new positions to calculate a displacement vector that
/// is stored in the lastDisplacement attribute.
/// All that with the goal of updating mesh faces associated to this 
/// molecule by the event of a deformation.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::setPosition( COME_Point3D &positionN ) {

	// When a molecule is just created its position is (0,0,0), so the first set
	// will cause an incorrect large displacement.
	// This test avoid creation of a displacement when this molecule is brand new.
	if( position.vpDistance( COME_Point3D( 0, 0, 0 ) ) != 0.0 ){
		lastDisplacement.setVector3D( positionN - position );
	}
	position = COME_Point3D( positionN );
}

//////////////////////////////////////////////////////////////////////
/// This method restaures the position of this molecule to the original
/// position used to construct it or the last checked position.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::resetPosition() {

	position = initPosition;
}

//////////////////////////////////////////////////////////////////////
/// This method sets the initPosition as being the current, establishing
/// a checkpoint for the position.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::checkPosition() {

	initPosition = position;
}


//////////////////////////////////////////////////////////////////////
/// JUST FOR TESTING IN jOINT mODELER
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::setINITPosition( COME_Point3D &newPos ){ //TEMP for testing RESPONSE

	initPosition = newPos;
}

COME_Point3D
COME_Molecule::getINITPositionGlobal(){ //TEMP for testing RESPONSE

	COME_Matrix localGIM = ((COME_BioStructure*)parent)->getGIM();
	if( description == "origin" )
		localGIM = COME_Matrix();
	//printf( "CHECK: Line commented because of getGIM() link error on discretizer \n" );
	if( !isFixed() )
		return initPosition;
	return localGIM * initPosition;
}
	
//////////////////////////////////////////////////////////////////////
/// This method sets for this molecule the neighbour molecules and
/// used it to build a "Molecule's Reference Frame".
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::initializeOrthogonalNeighbors(){

	//Test if by the way this molecule is not connected to 3 others
	if( connectionList->size() < 3 ){
		printf( "Surface molecule found with less than 3 neighbors.\n" );
		exit(666);
		//return;
	}
	
	// Test if orthogonalNeighbor is already initialized
	// (it happens when a molecule is anchor of more than one vertex.)
	if( orthogonalNeighbors.size() == 0 ){

		//Choose the first neighbor (randomly) from the connection List
		list<COME_MoleculeLink*>::iterator iterO = connectionList->begin();
		COME_Molecule *temp;
		if( (*iterO)->getElement(0) == this ){
			temp = (*iterO)->getElement(1);
		} else {
			temp = (*iterO)->getElement(0);
		}
		orthogonalNeighbors.push_back( temp );

		COME_Vector3D vI( orthogonalNeighbors[0]->getGlobalPosition() - this->getGlobalPosition() ); vI.vpNormalize();
		COME_Vector3D vJ;
		COME_Vector3D vK;
		COME_Vector3D normal;


		//Find the second neighbour by choosing the neighbour witch forms 
		//the closest angle to 90 with the first neighbour
		double localAngle = M_PI;
		COME_Molecule* localMolecule;
		for (iterO = connectionList->begin(), iterO++; iterO != connectionList->end(); iterO++ ){
			
			if( (*iterO)->getElement(0) == this ){
				temp = (*iterO)->getElement(1);
			} else {
				temp = (*iterO)->getElement(0);
			}

			vJ =  temp->getGlobalPosition() - this->getGlobalPosition() ; vJ.vpNormalize();
			if( fabs( vI.getAngle( vJ ) - ( M_PI / 2.0 ) ) < localAngle ){
			
				localMolecule = temp;
				localAngle = fabs (vI.getAngle( vJ ) - ( M_PI / 2.0 ));
			}
		}
		orthogonalNeighbors.push_back( localMolecule );
		
		
		//Find the third neighbour by choosing the neighbour witch forms 
		//the closest angle to 90 with the first and second neighbour
		vJ = orthogonalNeighbors[1]->getGlobalPosition() - this->getGlobalPosition(); vJ.vpNormalize();
		normal = vI.vpCrossProduct(vJ); normal.vpNormalize();
		localAngle = M_PI;
		double AngleFromNormal;
		for (iterO = connectionList->begin(), iterO++; iterO != connectionList->end(); iterO++ ){
			
			if( (*iterO)->getElement(0) == this ){
				temp = (*iterO)->getElement(1);
			} else {
				temp = (*iterO)->getElement(0);
			}

			vK= temp->getGlobalPosition() - this->getGlobalPosition(); vK.vpNormalize();
			
			AngleFromNormal = normal.getAngle(vK);
			if (AngleFromNormal > (M_PI / 2.0)) AngleFromNormal = M_PI-AngleFromNormal; // it never happens anyway
			if (fabs(AngleFromNormal) < localAngle){
				localMolecule = temp;
				localAngle = fabs(AngleFromNormal);
			}
		}
		//cout << "Angle of third coordinate Axis = " << localAngle*180/M_PI << " \n";  
		orthogonalNeighbors.push_back( localMolecule );
		
		//create the original frame
		makeOriginalFrame();
		// Create matrix to represent this Local Frame
		makeLocalFrame();
	}
	
}

void
COME_Molecule::makeOriginalFrame(){

	if( orthogonalNeighbors.size() > 0 ){

		if( localFrame == NULL ){
			localFrame = new COME_Matrix();
		}
		
		COME_Vector3D vI( orthogonalNeighbors[0]->getGlobalPosition() - this->getGlobalPosition() ); vI.vpNormalize();
		COME_Vector3D vJ( orthogonalNeighbors[1]->getGlobalPosition() - this->getGlobalPosition() ); vJ.vpNormalize();
		COME_Vector3D vK( orthogonalNeighbors[2]->getGlobalPosition() - this->getGlobalPosition() ); vK.vpNormalize();

		/// for debuging
		
		//COME_Vector3D vI( 1, 0, 0 );
		//COME_Vector3D vJ( 0, 1, 0 );
		//COME_Vector3D vK( 0, 0, 1 );
/*
		printf( "I: %f %f %f O: %f %f %f \n", orthogonalNeighbors[0]->getPosition().x, orthogonalNeighbors[0]->getPosition().y, orthogonalNeighbors[0]->getPosition().z, this->getPosition().x, this->getPosition().y, this->getPosition().z );
		printf( "J: %f %f %f O: %f %f %f \n", orthogonalNeighbors[1]->getPosition().x, orthogonalNeighbors[1]->getPosition().y, orthogonalNeighbors[1]->getPosition().z, this->getPosition().x, this->getPosition().y, this->getPosition().z );
		printf( "K: %f %f %f O: %f %f %f \n", orthogonalNeighbors[2]->getPosition().x, orthogonalNeighbors[2]->getPosition().y, orthogonalNeighbors[2]->getPosition().z, this->getPosition().x, this->getPosition().y, this->getPosition().z );
*/


/*
		COME_Vector3D vI( orthogonalNeighbors[0]->getGlobalPosition() - this->getGlobalPosition() );
		COME_Vector3D vJ( orthogonalNeighbors[1]->getGlobalPosition() - this->getGlobalPosition() );
		COME_Vector3D vK( orthogonalNeighbors[2]->getGlobalPosition() - this->getGlobalPosition() );
		*/
		
		originalLocalFrame.push_back(vI);
		originalLocalFrame.push_back(vJ);
		originalLocalFrame.push_back(vK);
	}
}

//////////////////////////////////////////////////////////////////////
/// Build or rebuild the matrix representing the local frame of this molecule.
/// It's made only if orthogonalNeighbors exist.
//////////////////////////////////////////////////////////////////////
void
COME_Molecule::makeLocalFrame(){
	
	if( orthogonalNeighbors.size() > 0 ){

		if( localFrame == NULL ){

			localFrame = new COME_Matrix();
		}

		
		COME_Vector3D vI( orthogonalNeighbors[0]->getGlobalPosition() - this->getGlobalPosition() ); vI.vpNormalize();
		COME_Vector3D vJ( orthogonalNeighbors[1]->getGlobalPosition() - this->getGlobalPosition() ); vJ.vpNormalize();
		COME_Vector3D vK( orthogonalNeighbors[2]->getGlobalPosition() - this->getGlobalPosition() ); vK.vpNormalize();
		
		//printf( "I: %f %f %f O: %f %f %f \n", orthogonalNeighbors[0]->getGlobalPosition().x, orthogonalNeighbors[0]->getGlobalPosition().y, orthogonalNeighbors[0]->getGlobalPosition().z, vI.x, vI.y, vI.z );
		//printf( "J: %f %f %f O: %f %f %f \n", orthogonalNeighbors[1]->getGlobalPosition().x, orthogonalNeighbors[1]->getGlobalPosition().y, orthogonalNeighbors[1]->getGlobalPosition().z, vJ.x, vJ.y, vJ.z );
		//printf( "K: %f %f %f O: %f %f %f \n", orthogonalNeighbors[2]->getGlobalPosition().x, orthogonalNeighbors[2]->getGlobalPosition().y, orthogonalNeighbors[2]->getGlobalPosition().z, vK.x, vK.y, vK.z );

		/// for debuging
		/*COME_Vector3D vI( 1, 0, 0 );
		COME_Vector3D vJ( 0, 1, 0 );
		COME_Vector3D vK( 0, 0, 1 );
		*/
		
		COME_Vector3D x(1,0,0);	
		COME_Vector3D y(1,0,0);	
		
		Quaternion rotation(0,0,0,1);		
				
		double angle = vI.getAngle(originalLocalFrame[0]);
		x = originalLocalFrame[0];
		y = vI;
		double angle2 = vJ.getAngle(originalLocalFrame[1]);
		double angle3 = vK.getAngle(originalLocalFrame[2]);
		
		if (angle2 > angle) {
			x = originalLocalFrame[1];
			y= vJ;
			angle = angle2;
		}
		if (angle3 > angle){
			x = originalLocalFrame[2];
			y = vK;
		}
				
		rotation.makeFromVecs(x, y);
		double RotationMatrix[3][3];
		rotation.ToMatrix(RotationMatrix);
				
		localFrame->setValueAt( 0, 0, RotationMatrix[0][0]);
		localFrame->setValueAt( 1, 0, RotationMatrix[1][0]);
		localFrame->setValueAt( 2, 0, RotationMatrix[2][0]);
		localFrame->setValueAt( 3, 0, this->getPosition().getX() );
		
		localFrame->setValueAt( 0, 1, RotationMatrix[0][1] );
		localFrame->setValueAt( 1, 1, RotationMatrix[1][1] );
		localFrame->setValueAt( 2, 1, RotationMatrix[2][1] );
		localFrame->setValueAt( 3, 1, this->getPosition().getY() );

		localFrame->setValueAt( 0, 2, RotationMatrix[0][2]);
		localFrame->setValueAt( 1, 2, RotationMatrix[1][2]);
		localFrame->setValueAt( 2, 2, RotationMatrix[2][2]);
		localFrame->setValueAt( 3, 2, this->getPosition().getZ() );

		localFrame->setValueAt( 0, 3, 0.0 );
		localFrame->setValueAt( 1, 3, 0.0 );
		localFrame->setValueAt( 2, 3, 0.0 );
		localFrame->setValueAt( 3, 3, 1.0 );	
	}
}
