/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  AUTHOR......: Anderson Maciel
//  DATE........: July/30/2002
//  DESCRIPTION.: Class declaration.
//
//  AUTHOR......: Anderson Maciel
//  DATE........: August/23/2002
//  DESCRIPTION.: Remove links when molecules are too distant.
//
////////////////////////////////////////////////////////////////////

#ifndef __COME_MOLECULE_H	
#define __COME_MOLECULE_H	

#ifndef __COME_BIOSTRUCTURE_H
class COME_BioStructure;
#endif

#include <algebra/comesubspaceelement.h>
#include <bio/comemoleculelink.h>
#include <physics/comeforce.h>
#include <physics/cometimeforce.h>



////////////////////////////////////////////////////////////////////
/// This class represents a generic molecular subdivision of an object.
/// Each molecule replaces a set of molecules of the real object and
/// it is, in fact, a spherical volume with a constant radius 
/// that approaches and diverges of its neighbours without regard
/// to geometrical penetration.
////////////////////////////////////////////////////////////////////
class COME_Molecule : public COME_SubSpaceElement  {

protected:
	double				radius;
	double				nominalRadius;
	double				frictionConst;
	list<COME_MoleculeLink*>	*connectionList;
	vector<COME_Molecule*>	orthogonalNeighbors;
	vector<COME_Vertex3D*>	buois;
	vector<int>				buoisIndices;
	COME_Matrix*			localFrame;
	vector<COME_Vector3D>	originalLocalFrame;
	COME_Point3D			initPosition;

	bool					onSurface;
	bool					collide;
	bool					updated;
	vector<COME_TimeForce*>	externalForces;
	COME_Force			collisionForce;
	COME_Vector3D		lastDisplacement;
	double				liquidFraction;
	double				permeability;
	
	int					intersecClamp;

	COME_Force	calculateForce( double currTime, COME_Point3D *positionsIN, COME_Vector3D *velocitiesIN, int currIndex );
	void		removeLink( COME_Molecule *moleculeN );

public:
	COME_Molecule();
	COME_Molecule( COME_Point3D positionN, double radiusN, double frictionConstN,
				   COME_Material *materialN, COME_Vector3D velocityN, COME_BioStructure *parentN );
	COME_Molecule( COME_Point3D &positionN, double &radiusN, double &frictionConstN,
				   COME_Material *materialN, COME_Vector3D &velocityN, COME_BioStructure *parentN );

	~COME_Molecule();

	void	setRadius( double radiusN );
	void	setFrictionConst( double frictionConstN );
	void	setConnectionList( list<COME_MoleculeLink*> *listN );
	void	setExternalForces( vector<COME_TimeForce*> listN );
	void	setOnSurface( bool flag );
	void	setCollisionForce( COME_Force forceN );
	void	setIfGreaterCollisionForce( COME_Force forceN );
	void	setPosition( COME_Point3D &positionN );
	void	setIntersectClamp( int type );
	void	setCollide( bool yesno );
	void	setUpdated( bool yesno );
	
	void	addCollisionForce( COME_Force forceN );

	double				getRadius() const;
	double				getFrictionConst() const;
	list<COME_MoleculeLink*>	*getConnectionList() const;
	vector<COME_Molecule*>		&getOrthogonalNeighbors();
	vector<COME_TimeForce*>		getExternalForces() const;
	vector<COME_Vertex3D*>		*getBuois();
	vector<int>					*getBuoisIndices();
	bool				isOnSurface() const;
	COME_Force			getCollisionForce() const;
	COME_Vector3D			getLastDisplacement() const;
	COME_Matrix*			getLocalFrame();
	double				getVolume() const;
	double				getLiquidLevel() const;
	double				getLiquidFraction() const;
	double				getNominalRadius() const;
	double				getPermeability() const;
	double				getPressureArea() const;
	bool				intersectClamp() const;
	bool				intersectClampFix() const;
	bool				intersectClampMobile() const;
	bool				isCollide() const;
	bool				isUpdated() const;
	
	void	addLink( COME_MoleculeLink *linkN );
	void	addBuoy( COME_Vertex3D *buoyN, int index );
	void	addOrthogonalNeighbor( COME_Molecule *newNeighbor );
	void	addExternalForce( COME_TimeForce *forceN );
	void	clearExternalForces();
	void	createLink( COME_Molecule *moleculeN );
	void	createLink( COME_Molecule *moleculeN, double globalHooke );
	void	makeLocalFrame();
	void	makeOriginalFrame();
	void	initializeOrthogonalNeighbors();
	void	resetPosition();
	void	checkPosition();
	
	void		setINITPosition( COME_Point3D &newPos ); //TEMP for testing RESPONSE
	COME_Point3D	getINITPositionGlobal(); //TEMP for testing RESPONSE
		
	void		derivs( double currTime, double timestep, COME_Point3D *positionsIN, COME_Vector3D *velocitiesIN, COME_Vector3D *velocitiesOUT, COME_Vector3D *accelerationsOUT, int currIndex );
	int		updateLinks();
	COME_Force	externalForceTo( double currTime );

	// To be used in chart plotting
	COME_Force	fOriginal, fDamping, fFriction, fLocal, fGravity, fExternal, fCollision;

	double	*liquidSharingArray;
};

#endif // __COME_MOLECULE_H	
