/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
///
///  PROJECT.....: CO-ME
///  RESPONSIBLE.: 
///
///  AUTHOR......: Anderson Maciel
///  DATE........: July/31/2002
///  DESCRIPTION.: Class definition.
///
////////////////////////////////////////////////////////////////////

#include <bio/comemoleculelink.h>
#include <bio/comemoleculestissue.h>
#include <bio/comemolecule.h>
#include <math.h>

//////////////////////////////////////////////////////////////////////
/// Construction/Destruction
//////////////////////////////////////////////////////////////////////

COME_MoleculeLink::COME_MoleculeLink()
{
	instanceCount++;
}

COME_MoleculeLink::COME_MoleculeLink( COME_Molecule *e1, COME_Molecule *e2 ){

	element1 = e1;
	element2 = e2;

	// set default values
	dampingConst = 0.0;
	hookeConst = 0.0;
	flowVolume1 = 0.0;
	flowVolume2 = 0.0;
	nominalDist = COME_Vector3D( e1->getPosition() - e2->getPosition() ).vpModule();
	viscousFraction = nominalDist / ( e1->getRadius() + e2->getRadius() );
	frictionConst = 0.0;
	instanceCount++;
	//printf( "Created link: %d\n", instanceCount );
}

COME_MoleculeLink::~COME_MoleculeLink(){

	instanceCount--;
	printf( "Destroied link: %d\n", instanceCount );
}

//////////////////////////////////////////////////////////////////////
/// Setting
//////////////////////////////////////////////////////////////////////

void
COME_MoleculeLink::setDampingConst( double dampingN ){

	dampingConst = dampingN;
}

void
COME_MoleculeLink::setHookeConst( double hookeN ){

	hookeConst = hookeN;
}

void
COME_MoleculeLink::setNominalDist( double nominalN ){

	nominalDist = nominalN;
}

void
COME_MoleculeLink::setViscousFraction( double fractionN ){

	viscousFraction = fractionN;
}

void
COME_MoleculeLink::setFrictionConst( double frictionConstN ){

	frictionConst = frictionConstN;
}

void
COME_MoleculeLink::setElement( int index, COME_Molecule* elementN ){

	if( index == 2 ){
		element2 = elementN;
	} else {
		element1 = elementN;
	}
}

void
COME_MoleculeLink::setFlowVolume( COME_Molecule* element, double volumeNew ){

	if( element == element1 ){
		flowVolume1 = volumeNew;
	} else if( element == element2 ) {
		flowVolume2 = volumeNew;
	}
}


//////////////////////////////////////////////////////////////////////
/// Getting
//////////////////////////////////////////////////////////////////////

double
COME_MoleculeLink::getDampingConst() const{

	return dampingConst;
}

double
COME_MoleculeLink::getHookeConst() const{

	return hookeConst;
}

double
COME_MoleculeLink::getNominalDist() const{

	return nominalDist;
}


double
COME_MoleculeLink::getViscousFraction() const{

	return viscousFraction;
}

double
COME_MoleculeLink::getFractureDist() const{

	return element1->getRadius() + element2->getRadius();
}

double
COME_MoleculeLink::getFrictionConst() const{

	return frictionConst;
}

COME_Molecule*
COME_MoleculeLink::getElement( int index ){

	if( index == 1 ){
		return element2;
	}
	return element1;
}

COME_Molecule*
COME_MoleculeLink::getOtherElement( COME_Molecule* thisElement ){

	if( thisElement == element1 ){
		return element2;
	}
	return element1;
}

double
COME_MoleculeLink::getFlowVolume( COME_Molecule* element ) const{

	if( element == element1 ){
		return flowVolume2 - flowVolume1;
	} else if( element == element2 ) {
		return flowVolume1 - flowVolume2;
	}

	return 0.0;

}

//////////////////////////////////////////////////////////////////////
/// Return this link length (distance between its two elements).
//////////////////////////////////////////////////////////////////////
double
COME_MoleculeLink::getLength() const {

	return element1->getPosition().vpDistance( element2->getPosition() );
}


//////////////////////////////////////////////////////////////////////
/// Return a vector with all tetrahedra which this link is an edge.
//////////////////////////////////////////////////////////////////////
vector<COME_Tetra>
COME_MoleculeLink::getNeighbouringTetra() const {

	vector<COME_Tetra> tetras;

	list<COME_MoleculeLink*>::iterator iter1;
	int teste = 0;
	for (iter1 = element1->getConnectionList()->begin(); iter1 != element1->getConnectionList()->end(); iter1++, teste ++){

		COME_Molecule* tetra2 = (*iter1)->getOtherElement(element1);
		COME_Molecule* tetra3;
		bool tetra2oK = false;
		bool tetra3oK = false;
		
		if( tetra2 == element2 ){
			continue;
		}

		list<COME_MoleculeLink*>::iterator iter2;
		for (iter2 = tetra2->getConnectionList()->begin(); iter2 != tetra2->getConnectionList()->end(); iter2++){
			
			COME_Molecule* tetraTemp = (*iter2)->getOtherElement(tetra2);
			if( tetraTemp == element2 ){
				tetra2oK = true;
				break;
			}
		}

		if( tetra2oK ){
			for (iter2 = tetra2->getConnectionList()->begin(); iter2 != tetra2->getConnectionList()->end(); iter2++){
				tetra3 = (*iter2)->getOtherElement(tetra2);
				if( ( tetra3 != element2 ) && ( tetra3 != element1 ) ){
					tetra3oK = true;
					break;
				}
			}

			if( tetra3oK ){
				COME_Tetra newTetra;
				newTetra.addMolecule( element1 ); 
				newTetra.addMolecule( tetra2 );
				newTetra.addMolecule( tetra3 );
				newTetra.addMolecule( element2 );
				tetras.push_back( newTetra );
			}
		}

	}

	return tetras;
}

//////////////////////////////////////////////////////////////////////
/// Calculate spring constant for this link.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculeLink::recalculateSpring( int mode, double avgConnect ){

	if( mode == VOLUME_TETRAS ){

		double totalVE = 0.0;
		vector<COME_Tetra> tetras = getNeighbouringTetra();
		for( int i= 0; i < tetras.size(); i++ ){

			totalVE = tetras[i].getVolume()*tetras[i].getElasticity();
		}
		
		// test when there is not any tetrahedron.
		if( totalVE > 0.0 ){
			printf( "old Hooke: %f\n", hookeConst );
			hookeConst = totalVE * 1000 / ( getLength() * getLength() );
			printf( "new Hooke: %f\n", hookeConst );
		}
	} else if( mode == NUM_CONNECTIONS ){

		int N1 = element1->getConnectionList()->size();
		int N2 = element2->getConnectionList()->size();

		int D1, D2;

		if( N1 < 8 ){
			D1 = N1;
		} else if( ( N1 >= 9 ) && ( N1 < 12 ) ){
			D1 = N1 - 1;
		} else if( ( N1 >= 13 ) && ( N1 < 20 ) ){
			D1 = N1 - 4;
		} else if( N1 >= 20 ){
			D1 = N1 /2;
		}

		if( N2 < 8 ){
			D2 = N2;
		} else if( ( N2 >= 9 ) && ( N2 < 12 ) ){
			D2 = N2 - 1;
		} else if( ( N2 >= 13 ) && ( N2 < 20 ) ){
			D2 = N2 - 4;
		} else if( N2 >= 20 ){
			D2 = N2 /2;
		}

		double dd1 = (double)D1;
		double dd2 = (double)D2;

		double area1 = ( (2*element1->getRadius())*(2*element1->getRadius()) ) / (dd1/2);
		double area2 = ( (2*element2->getRadius())*(2*element2->getRadius()) ) / (dd2/2);
		
		double hooke = ( ( ( element1->getMaterial()->getYoungsModulus()*area1 ) + ( element2->getMaterial()->getYoungsModulus()*area2 ) ) / 2 ) / getNominalDist();

		hooke += (hooke / 3.0);
		printf("old hooke: %f\n", hookeConst );
		setHookeConst( hooke );
		printf("new hooke: %f\n", hookeConst );

	} else if( mode == ANGLES ){

		COME_Vector3D vThis( element2->getPosition() - element1->getPosition() );

		list<COME_MoleculeLink*>::iterator iter1;
		double discounts = 0.0;
		double totalCos = 0.0;
		for (iter1 = element1->getConnectionList()->begin(); iter1 != element1->getConnectionList()->end(); iter1++){

			COME_Molecule* temp = (*iter1)->getOtherElement(element1);
			if( temp != element2 ){

				
				double area1 = ( (2*element1->getRadius())*(2*element1->getRadius()) );
				double area2 = ( (2*temp->getRadius())*(2*temp->getRadius()) );
		
				double hooke = ( ( ( element1->getMaterial()->getYoungsModulus()*area1 ) / (*iter1)->getNominalDist() ) + ( ( temp->getMaterial()->getYoungsModulus()*area2 ) / (*iter1)->getNominalDist() ) ) / 2;

				COME_Vector3D vCurr( temp->getPosition() - element1->getPosition() );
				if( vThis.getAngle( vCurr ) < (M_PI/2) ){
					double currCos = cos( vThis.getAngle( vCurr ) );
					if( vThis.vpModule() > vCurr.vpModule() ){
						currCos = 1 / currCos;
					}
					totalCos += currCos * vCurr.vpModule();
					discounts += hooke * cos( vThis.getAngle( vCurr ) );
				}

			} else {
				COME_Vector3D vCurr( element2->getPosition() - element1->getPosition() );
				totalCos += vCurr.vpModule();
				double area1 = ( (2*element1->getRadius())*(2*element1->getRadius()) );
				double area2 = ( (2*element2->getRadius())*(2*element2->getRadius()) );
				double hooke = ( ( ( element1->getMaterial()->getYoungsModulus()*area1 ) / getNominalDist() ) + ( ( element2->getMaterial()->getYoungsModulus()*area2 ) / getNominalDist() ) ) / 2;
				discounts += hooke;
			}
		}

		COME_Vector3D vThis2( element1->getPosition() - element2->getPosition() );

		for (iter1 = element2->getConnectionList()->begin(); iter1 != element2->getConnectionList()->end(); iter1++){

			COME_Molecule* temp = (*iter1)->getOtherElement(element2);
			if( temp != element1 ){

				
				double area1 = ( (2*element2->getRadius())*(2*element2->getRadius()) );
				double area2 = ( (2*temp->getRadius())*(2*temp->getRadius()) );
		
				double hooke = ( ( ( element2->getMaterial()->getYoungsModulus()*area1 ) / (*iter1)->getNominalDist() ) + ( ( temp->getMaterial()->getYoungsModulus()*area2 ) / (*iter1)->getNominalDist() ) ) / 2;

				COME_Vector3D vCurr( temp->getPosition() - element2->getPosition() );
				if( vThis2.getAngle( vCurr ) < (M_PI/2) ){
					double currCos = cos( vThis2.getAngle( vCurr ) );
					if( vThis2.vpModule() > vCurr.vpModule() ){
						currCos = 1 / currCos;
					}
					totalCos += currCos * vCurr.vpModule();
					discounts += hooke * cos( vThis2.getAngle( vCurr ) );
				}

			} else {
				COME_Vector3D vCurr( element2->getPosition() - element1->getPosition() );
				totalCos += vCurr.vpModule();
				double area1 = ( (2*element1->getRadius())*(2*element1->getRadius()) );
				double area2 = ( (2*element2->getRadius())*(2*element2->getRadius()) );
				double hooke = ( ( ( element1->getMaterial()->getYoungsModulus()*area1 ) / getNominalDist() ) + ( ( element2->getMaterial()->getYoungsModulus()*area2 ) / getNominalDist() ) ) / 2;
				discounts += hooke;
			}
		}

		double area1 = ( (2*element1->getRadius())*(2*element1->getRadius()) );
		double area2 = ( (2*element2->getRadius())*(2*element2->getRadius()) );
		double hooke = ( ( ( element1->getMaterial()->getYoungsModulus()*area1 ) / getNominalDist() ) + ( ( element2->getMaterial()->getYoungsModulus()*area2 ) / getNominalDist() ) ) / 2;

		double newHooke =  getNominalDist() * ( discounts / totalCos );
		printf( "newHooke: %f ; hooke: %f ; discounts: %f ; totalCos: %f \n", newHooke, hooke, discounts, totalCos );

		setHookeConst( newHooke );

	} else if( mode == NUMERICALLY ){

		double increment = avgConnect; // reuse parameter
		setHookeConst( hookeConst * increment );
	}


	
}
