/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  AUTHOR......: Anderson Maciel
//  DATE........: August/05/2002
//  DESCRIPTION.: Class declaration.
//
////////////////////////////////////////////////////////////////////


#include <bio/comemoleculesbone.h>
#include <bio/comematerial.h>
#include <math.h>
#include <algebra/comemesh.h>

//#include <LinAlg/linalg.h>
//using namespace LinAlg;
//#include <MarchingCubes/marching_cubes.h>
//#include <MarchingCubes/blob.h>



//////////////////////////////////////////////////////////////////////
/// Constructors and destructors.
//////////////////////////////////////////////////////////////////////

COME_MoleculesBone::COME_MoleculesBone(){

	tissue = new COME_MoleculesTissue();
	tissue->setParent(this);
	surface = NULL;
	hashTable = NULL;
	updateMassCenter();
	globalHooke = 0.0;
	collisionRadius = 1.0;
}

COME_MoleculesBone::~COME_MoleculesBone(){

	
}

//////////////////////////////////////////////////////////////////////
/// Protected attributes access methods.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesBone::setTissue( COME_MoleculesTissue *tissueN ){

	tissue = tissueN;
}

void
COME_MoleculesBone::setMassCenter( const COME_Point3D& massC ){

	massCenter = massC;
}

COME_MoleculesTissue*
COME_MoleculesBone::getTissue(){

	return tissue;
}

COME_Point3D
COME_MoleculesBone::getMassCenter() const{

	return massCenter;
}

//Recalculates the Mesh based on the new positions of anchoring the molecules
void 
COME_MoleculesBone:: updateSkin(){
	if( surface )
		surface->updateSkin();
}

//////////////////////////////////////////////////////////////////////
/// This method re-generates the boundary mesh of the cartilage for visualization.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::updateSurface( bool regenerate ){

//	if( regenerate ){
//		updateSurface( 1.0, 150.0, 0.005 ); //1.0  120.0 0.005  
//		surface.estabilishLinkVerticesFaces();

		// para os pequenos 1.0, 80.0, 0.005
		// para os grandes 1.0, 160.0, 0.01
		//collision:   //2.3  (15)

//	} else {
		//generateEnvelopSurface();
//	}
  

	if( regenerate ){
		COME_Point3D mins, maxs;
		getEnvelop( mins, maxs );
		double volume = ( maxs.getX() - mins.getX() ) * ( maxs.getY() - mins.getY() ) * ( maxs.getZ() - mins.getZ() );
		double radiusAvg = tissue->getRadiusAverage();

		updateSurface( 1.0, 1/(radiusAvg/2), sqrt( volume )/20.0 + 0.005 );
		surface->estabilishLinkVerticesFaces();

	} else {
		surface->update();
	}
}


//////////////////////////////////////////////////////////////////////
/// This method re-generates the boundary mesh of the bone for visualization.
/// Such mesh is an implicit surface that surrounds the organ shape.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::updateSurface( double skinSphere, double stiff, double resolution ){

/*
	// Metaballs represented as the spheres with Raquel's density function
	MetaballsSphere metaSphere( skinSphere );
	
	vector<Primitive*>			primitives;
	VEC							C(3);		// sphere center

	list<COME_Molecule*>::iterator iter;	
	for( iter = tissue->getShape()->begin(); iter != tissue->getShape()->end(); iter++ ){
		//read spheres
		COME_Point3D point	= (*iter)->getPosition();

		C[0] = point.getX();
		C[1] = point.getY();
		C[2] = point.getZ();
		
		//add sphere  
		metaSphere.addSphere( Sphere( C, (*iter)->getRadius(), stiff ) );
	}
  
	//add to primitives
	primitives.push_back( &metaSphere );

	//create surface
	marchingCubes surfaceTriangles( primitives, resolution );
	

	// copy vertices and triangle indices into the Class Mesh	
	
	int iPos;

	vector<VEC>	vertexList	= surfaceTriangles.getVertices();
	vector<VEC> normalList	= surfaceTriangles.getNormals();

	// copy all vertices and the normals of the triangulated surface 
	surface->resetMesh();
	for ( iPos = 0; iPos < vertexList.size(); iPos++ ){
		
		COME_Vector3D normal = COME_Vector3D( normalList[ iPos ][ 0 ], normalList[ iPos ][ 1 ], normalList[ iPos ][ 2 ] );
		surface->addVertex( COME_Vertex3D( vertexList[ iPos ][ 0 ], vertexList[ iPos ][ 1 ], vertexList[ iPos ][ 2 ], normal ) );
	}

	//copy all indices of vertices 
	vector<Triangle>	trianglesList	= surfaceTriangles.getTriangles();
	for (vector<Triangle>::const_iterator tri=trianglesList.begin(); tri!=trianglesList.end(); tri++){     
		
		vector<int>			indicesTriangles;

		indicesTriangles.push_back( tri->indices[0] );
		indicesTriangles.push_back( tri->indices[1] );
		indicesTriangles.push_back( tri->indices[2] );

		surface->addFaceAsIndices( indicesTriangles );
	}
*/
}

//////////////////////////////////////////////////////////////////////
/// This method re-generates the boundary mesh of the bone for visualization.
/// Such mesh is the rectangular envelop for the organ shape.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::generateEnvelopSurface(){
	
	surface->resetMesh();

	COME_Point3D mins, maxs;
	getEnvelop( mins, maxs );

	COME_Vertex3D v0 = COME_Vertex3D( mins.getX(), mins.getY(), mins.getZ() );
	COME_Vertex3D v1 = COME_Vertex3D( mins.getX(), maxs.getY(), mins.getZ() );
	COME_Vertex3D v2 = COME_Vertex3D( mins.getX(), mins.getY(), maxs.getZ() );
	COME_Vertex3D v3 = COME_Vertex3D( mins.getX(), maxs.getY(), maxs.getZ() );
	COME_Vertex3D v4 = COME_Vertex3D( maxs.getX(), mins.getY(), maxs.getZ() );
	COME_Vertex3D v5 = COME_Vertex3D( maxs.getX(), maxs.getY(), maxs.getZ() );
	COME_Vertex3D v6 = COME_Vertex3D( maxs.getX(), mins.getY(), mins.getZ() );
	COME_Vertex3D v7 = COME_Vertex3D( maxs.getX(), maxs.getY(), mins.getZ() );

	v0.setNormal( COME_Vector3D( v0 - v5 ).vpNormalize() );
	v1.setNormal( COME_Vector3D( v1 - v4 ).vpNormalize() );
	v2.setNormal( COME_Vector3D( v2 - v7 ).vpNormalize() );
	v3.setNormal( COME_Vector3D( v3 - v6 ).vpNormalize() );
	v4.setNormal( COME_Vector3D( v4 - v1 ).vpNormalize() );
	v5.setNormal( COME_Vector3D( v5 - v0 ).vpNormalize() );
	v6.setNormal( COME_Vector3D( v6 - v3 ).vpNormalize() );
	v7.setNormal( COME_Vector3D( v7 - v2 ).vpNormalize() );
	
	surface->addVertex( v0 );
	surface->addVertex( v1 );
	surface->addVertex( v2 );
	surface->addVertex( v3 );
	surface->addVertex( v4 );
	surface->addVertex( v5 );
	surface->addVertex( v6 );
	surface->addVertex( v7 );

	vector<int>	indTri;

	indTri.push_back( 0 ); indTri.push_back( 3 ); indTri.push_back( 1 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 0 ); indTri.push_back( 2 ); indTri.push_back( 3 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 2 ); indTri.push_back( 5 ); indTri.push_back( 3 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 2 ); indTri.push_back( 4 ); indTri.push_back( 5 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 4 ); indTri.push_back( 7 ); indTri.push_back( 5 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 4 ); indTri.push_back( 6 ); indTri.push_back( 7 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 6 ); indTri.push_back( 1 ); indTri.push_back( 7 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 6 ); indTri.push_back( 0 ); indTri.push_back( 1 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 1 ); indTri.push_back( 5 ); indTri.push_back( 7 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 1 ); indTri.push_back( 3 ); indTri.push_back( 5 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 6 ); indTri.push_back( 2 ); indTri.push_back( 0 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();
	indTri.push_back( 6 ); indTri.push_back( 4 ); indTri.push_back( 2 );
	surface->addFaceAsIndices( indTri );
	indTri.clear();

}

//////////////////////////////////////////////////////////////////////
/// This method returns the 3D rectangular envelop of this organ.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::getEnvelop( COME_Point3D& mins, COME_Point3D& maxs ){

	tissue->getEnvelop( mins, maxs );
}

//////////////////////////////////////////////////////////////////////
/// This method calculate the new torque and force on this bone center of mass.
/// Such values may be used by higher level entities to update the global position
/// of the bone.
//////////////////////////////////////////////////////////////////////// 
//////////////////////////////////////////////////////////////////////
/// This method implements the Runge-Kutta 4 method for numeric integration
/// to calculate the new positions for all molecules of this organ.
//////////////////////////////////////////////////////////////////////// 

bool
COME_MoleculesBone::update( double timestep, double simClock ){

/*
	double xh , hh, h6;
	COME_Vector3D *dymV, *dymF, *dymT, *dymW;
	
	COME_Vector3D *dytV, *dytF, *dytT, *dytW;
	
	COME_Vector3D *ytX, *ytL, *ytP;
	COME_Matrix *ytR;

	dymV = new COME_Vector3D();
	dymF = new COME_Vector3D();
	dymT = new COME_Vector3D();
	dymW = new COME_Vector3D();
	
	dytV = new COME_Vector3D();
	dytF = new COME_Vector3D();
	dytT = new COME_Vector3D();
	dytW = new COME_Vector3D();
	
	ytX = new COME_Vector3D();
	ytL = new COME_Vector3D();
	ytP = new COME_Vector3D();
	ytR = new COME_Matrix();

	COME_Vector3D momentumTemp;
	COME_Vector3D w;

	//COME_Matrix I = COME_Matrix( rotationMatrix * inertiaTensor * rotationMatrix.getTransposed() );

	hh = timestep * 0.5;
	h6 = timestep / 6.0;
	xh = simClock + hh;
	
	// First step //////////////////////////////////////////////////////
	
	ytX->setVector3D( ( linearVelocity * hh ) + massCenter );
	ytL->setVector3D( rotationMatrix * inertiaTensor * rotationMatrix.getTransposed() * angularVelocity );
	momentumTemp.setVector3D( *ytL ); // save initial momentum
	ytP->setVector3D( linearVelocity * tissue->getMass() );
	ytR->setMatrix( rotationMatrix );
	
	// Second step //////////////////////////////////////////////////////
	derivs( xh, hh, ytX, ytL, ytP, ytR, dytV, dytF, dytT, dytW ); 

	ytX->setVector3D( ( *dytV * hh ) + massCenter );
	ytL->setVector3D( *dytW * *ytR * inertiaTensor * (*dytW * ytR->getTransposed() ) * *dytW );
	ytP->setVector3D( *dytV * tissue->getMass() );
	ytR->setMatrix( rotationMatrix );//*ytR += (*dytW); // probably non-correct

	// Third step //////////////////////////////////////////////////////
	derivs( xh, hh, ytX, ytL, ytP, ytR, dymV, dymF, dymT, dymW ); 

	ytX->setVector3D( ( *dymV * timestep ) + massCenter );
	ytL->setVector3D( *dymW * *ytR * inertiaTensor * (*dymW * ytR->getTransposed() ) * *dymW );
	ytP->setVector3D( *dymV * tissue->getMass() );
	ytR->setMatrix( rotationMatrix );//ytR->setMatrix( *( dymWR->add( *ytR ) ) );

	*dymV += *dytV;
	*dymF += *dytF;
	*dymT += *dytT;
	*dymW += *dytW;


	// Fourth step //////////////////////////////////////////////////////
	derivs( simClock + timestep, timestep, ytX, ytL, ytP, ytR, dytV, dytF, dytT, dytW ); 

	massCenter = ( ( linearVelocity + *dytV + ( *dymV * 2.0 ) ) * h6 ) + massCenter;
	COME_Vector3D rotatedVec = COME_Vector3D( angularVelocity * rotationMatrix );
	rotationMatrix = ( ( angularVelocity + (*dytW * *ytR) + ( *dymW * *ytR * 2.0 ) ) * h6 ).transMult( rotatedVec );
	angularVelocity =  rotationMatrix * inertiaTensor.getInverse() * rotationMatrix.getTransposed() * momentumTemp;
	linearVelocity = ( ( linearVelocity + *dytV + ( *dymV * 2.0 ) ) * h6 ) * tissue->getMass();
		
	delete dymV;
	delete dymF;
	delete dymT;
	delete dymW;
	
	delete dytV; 
	delete dytF;
	delete dytT;
	delete dytW;

	delete ytX;
	delete ytL;
	delete ytP;
	delete ytR;
*/
	updateSkin();
	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method derivs the two sets of variables positions and velocities
/// into the two new sets velocities and accelerations. 
/// It calls the protected method calculateForce() to compute the final
/// force being applied on this molecule and apply Newton's laws to calculate
/// the resultant acceleration and the final velocity.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesBone::derivs( double currTime, double timestep,
						    COME_Point3D *X, COME_Vector3D *L, COME_Vector3D *P, COME_Matrix *R,
							COME_Vector3D *V, COME_Vector3D *F, COME_Vector3D *T, COME_Vector3D *W ){
	
	if( !isFixed() ){
		
		V->setVector3D( *P / tissue->getMass() );
		//F = addAllForces();
		//T = ( forcePoint - massCenter ) * forceVector;
		W->setVector3D(  (*R) * inertiaTensor.getInverse() * R->getTransposed() * (*L) );
	}
}


//////////////////////////////////////////////////////////////////////
/// This method calculates the inertia tensor for this bone based on its mesh volume
/// and stores it in the inertiaTensor attribute. The parameter discretization
/// defines the sample size as a fraction of the bounding box. 
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::updateInertiaTensor( double discretization ){


	inertiaTensor.reset();
	int hits = 0;
	
	COME_Point3D mins, maxs;
	getEnvelop( mins, maxs );

	for( double x = mins.getX(); x < maxs.getX(); x += discretization *  ( maxs.getX() - mins.getX() ) ){
		for( double y = mins.getY(); y < maxs.getY(); y += discretization *  ( maxs.getY() - mins.getY() ) ){
			for( double z = mins.getZ(); z < maxs.getZ(); z += discretization *  ( maxs.getZ() - mins.getZ() ) ){

				if( surface->isInside( COME_Point3D(x,y,z) ) ){

					inertiaTensor.setValueAt( 0, 0, inertiaTensor.getValueAt( 0, 0 ) + ( y * y + z * z ) );
					inertiaTensor.setValueAt( 0, 1, inertiaTensor.getValueAt( 0, 1 ) + ( -x * y ) );
					inertiaTensor.setValueAt( 0, 2, inertiaTensor.getValueAt( 0, 2 ) + ( -x * z ) );

					inertiaTensor.setValueAt( 1, 0, inertiaTensor.getValueAt( 1, 0 ) + ( -x * y ) );
					inertiaTensor.setValueAt( 1, 1, inertiaTensor.getValueAt( 1, 1 ) + ( x * x + z * z ) );
					inertiaTensor.setValueAt( 1, 2, inertiaTensor.getValueAt( 1, 2 ) + ( -y * z ) );

					inertiaTensor.setValueAt( 2, 0, inertiaTensor.getValueAt( 2, 0 ) + ( -x * z ) );
					inertiaTensor.setValueAt( 2, 1, inertiaTensor.getValueAt( 2, 1 ) + ( -y * z ) );
					inertiaTensor.setValueAt( 2, 2, inertiaTensor.getValueAt( 2, 2 ) + ( x * x + y * y ) );

	                hits++;
				}
			}
		}
	}

	inertiaTensor.multiplyScalar( 1/hits );
	inertiaTensor.multiplyScalar( tissue->getMass() );

}

//////////////////////////////////////////////////////////////////////
/// This method calculates the center of mass for this bone based on its molecules
/// distribution and density. Then it stores it in the massCenter attribute. 
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesBone::updateMassCenter(){

	massCenter = tissue->getMassCenter();
}

//////////////////////////////////////////////////////////////////////
/// This method only generates the normals on the surface of the bone.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesBone::initializeSkinning(){

	if( this->getSurface() == NULL ){
		surface = new COME_Mesh();
		surface->setParent( this );
		updateSurface( IMPLICIT );
		printf( "update IMPLICIT BONE done \n" );
	}
	this->getSurface()->anchorToMolecules( this->getTissue(), 4 );
	cout << "initialising Faces Neighbours...";
	this->getPointerToSurface()->initFacesNeighbours();
	cout << "done \ninitialising Vertices Neighbours...";
	this->getPointerToSurface()->initVerticesNeighbours();
	cout << "done\n";
}

void
COME_MoleculesBone::respondCollision(){

	// do nothing for now
	//printf( "Entrou bone::respondcollision \n" );
}
