/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  AUTHOR......: Anderson Maciel
//  DATE........: August/05/2002
//  DESCRIPTION.: Class definition.
//
////////////////////////////////////////////////////////////////////


#include <bio/comemoleculestissue.h>
#include <bio/comemoleculescartilage.h>
#include <bio/comebiostructure.h>
#include <bio/comematerial.h>
#include <math.h>
#include <algebra/comemesh.h>
#include <algebra/comevertex3d.h>

//#include <LinAlg/linalg.h>

//#include <conio.h>

//using namespace LinAlg;

//#include <MarchingCubes/marching_cubes.h>
//#include <MarchingCubes/blob.h>

//////////////////////////////////////////////////////////////////////
/// Default constructor.
//////////////////////////////////////////////////////////////////////

COME_MoleculesTissue::COME_MoleculesTissue()
{
	radiusAverage = 0.0;
	mass = 0.0;
	accu = 0.0;
	averageNumConnectionsPerMolecule = 0;
	dymv = NULL;
	dyma = NULL;
	dytv = NULL;
	dyta = NULL;
	ytp = NULL;
	ytv = NULL;
	gpuForces = NULL;
	parent = NULL;
}

COME_MoleculesTissue::~COME_MoleculesTissue()
{
	if( dymv ) delete [] dymv;
	if( dyma ) delete [] dyma;
	if( dytv ) delete [] dytv;
	if( dyta ) delete [] dyta;
	if( ytp ) delete [] ytp;
	if( ytv ) delete [] ytv;
	if( gpuForces ) delete [] gpuForces;
	
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		delete (*iter);
	}
}

void
COME_MoleculesTissue::initForSimulation(){

	dymv = new COME_Vector3D[shape.size()];
	dyma = new COME_Vector3D[shape.size()];
	
	dytv = new COME_Vector3D[shape.size()];
	dyta = new COME_Vector3D[shape.size()];
	
	ytp = new COME_Vector3D[shape.size()];
	ytv = new COME_Vector3D[shape.size()];
	
	gpuForces = new COME_Vector3D[shape.size()];
}

list<COME_Molecule*>*
COME_MoleculesTissue::getShape(){

	return &shape;
}

void
COME_MoleculesTissue::setRadiusAverage( double radiusN ){

	radiusAverage = radiusN;
}

double
COME_MoleculesTissue::getRadiusAverage(){

	return radiusAverage;
}

int
COME_MoleculesTissue::getAverageNumConnections(){

	return averageNumConnectionsPerMolecule;
}

//////////////////////////////////////////////////////////////////////
/// Test if currentPoint is closer than a distance from any of the 
/// molecules of this tissue.
//////////////////////////////////////////////////////////////////////
bool
COME_MoleculesTissue::isTooClose( COME_Point3D currentPoint, double distance ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		if( (*iter)->getPosition().vpDistance( currentPoint ) < distance ){
			return true;
		} 
	}
	return false;
}

//////////////////////////////////////////////////////////////////////
/// The following 7 methods are based in the assumption that all molecules of
/// the tissue are equal. So, they set the same values to all molecules and
/// get allways the values for the first molecule.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::setMoleculesMaterial( COME_Material* newMaterial ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		(*iter)->setMaterial( newMaterial );
	}
}

void
COME_MoleculesTissue::setMoleculesDensity( double densityN ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		(*iter)->getMaterial()->setDensity( densityN );
	}
}

void
COME_MoleculesTissue::setMoleculesElasticity( double elasticityN ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		(*iter)->getMaterial()->setYoungsModulus( elasticityN );
	}
}

void
COME_MoleculesTissue::setMoleculesFrictionConst( double frictionN ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		(*iter)->setFrictionConst( frictionN );
	}
}

/// Returns the material of the first molecule of the tissue. Attention, a tissue may have
/// molecules of different materials.
COME_Material*
COME_MoleculesTissue::getMoleculesMaterial(){

	return shape.front()->getMaterial();
}

double
COME_MoleculesTissue::getDensity(){

	return shape.front()->getMaterial()->getDensity();
}

double
COME_MoleculesTissue::getElasticity(){

	return shape.front()->getMaterial()->getYoungsModulus();
}

double
COME_MoleculesTissue::getFrictionConst(){

	return shape.front()->getFrictionConst();
}

double
COME_MoleculesTissue::getTopStress(){

	double top = 0.0;
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++ ){
		
		if( top < (*iter)->getStress() )
			top=(*iter)->getStress();
	}
	return top;
}

COME_Force
COME_MoleculesTissue::getGPUForce( int i ){

	return gpuForces[i];

}
//////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
/// This method adds moleculeN to the list o molecules of this cartilage.
/// Connections are created automatically according to positions and radius.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::addMolecule( COME_Molecule *moleculeN ){
	
	// test for all existent molecules if distance to moleculeN is smaller than its fracture distance
	bool canAdd = true;
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		// if it is, create a connection between them
		if( ((COME_Point3D)((*iter)->getPosition())).vpDistance( moleculeN->getPosition() ) < ( (*iter)->getRadius() + moleculeN->getRadius() ) ){
			
			if( ((COME_BioStructure*)parent)->getGlobalHooke() == 0.0 ){
				moleculeN->createLink( (*iter) );
			} else {
				moleculeN->createLink( (*iter), ((COME_BioStructure*)parent)->getGlobalHooke() );
			}
			canAdd = true;
		}
		
	}

	if( canAdd ){
		radiusAverage = ( radiusAverage*shape.size() + moleculeN->getRadius() ) / ( shape.size() + 1 );
		moleculeN->setIndex( shape.size() );
		shape.push_back( moleculeN );
		mass += moleculeN->getMass();
	}
}

//////////////////////////////////////////////////////////////////////
/// This method adds moleculeN to the list o molecules of this cartilage.
/// Connections are created automatically according to positions and radius.
/// If some existent molecule is at a distance smaller than minDist DON'T ADD
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::addMolecule( COME_Molecule *moleculeN, double minDist ){
	
	
	list<COME_Molecule*>::iterator iter;

	// test if this new molecule is too close of any other in the list (test minDist)
	for (iter = shape.begin(); iter != shape.end(); iter++){
		// if it is, don't add it
		if( ((COME_Point3D)((*iter)->getPosition())).vpDistance( moleculeN->getPosition() ) >= ( minDist ) ){
			// do nothing
		} else {
			return;
		}		
	}

	// test for all existent molecules if distance to moleculeN is smaller than its fracture distance
	for (iter = shape.begin(); iter != shape.end(); iter++){
		// if it is, create a connection between them
		if( ((COME_Point3D)((*iter)->getPosition())).vpDistance( moleculeN->getPosition() ) < ( (*iter)->getRadius() + moleculeN->getRadius() ) ) {
			moleculeN->createLink( (*iter) );
		}
		
	}

	radiusAverage = ( radiusAverage*shape.size() + moleculeN->getRadius() ) / ( shape.size() + 1 );
	moleculeN->setIndex( shape.size() );
	shape.push_back( moleculeN );
	mass += moleculeN->getMass();
}

bool
COME_MoleculesTissue::updateEuler( double timestep, double simClock, string fileNameToSaveDeformation ){

	//printf( "ENTROU\n");
	simTime = simClock;
	if( !dymv ){
		initForSimulation();
	}
	int i = 0;
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++, i++){
	
		ytp[i] = (*iter)->getGlobalPosition();
		ytv[i] = (*iter)->getVelocity();
		//printf( "pos: %f vel: %f \n", (*iter)->getGlobalPosition().getY(), (*iter)->getVelocity().getY() );
	}
	
	i = 0;
	for (iter = shape.begin(); iter != shape.end(); iter++, i++){
		
		if( !(*iter)->isFixed() ){
			////////// Derive
			(*iter)->derivs( simClock, timestep, ytp, ytv, dytv, dyta, i ); 
			
			//////// Update position and velocity
			(*iter)->setPositionMinusOne( (*iter)->getGlobalPosition() );
			COME_Point3D newpos( (*iter)->getGlobalPosition() + dytv[i]*timestep );
			(*iter)->setPosition( newpos );
			(*iter)->setVelocity( dytv[i] );
		}
	}
	
	////////// Apply collision response (position and velocity)
	 if( ( COME::flagCollisionTreatment == NEIGHBORS ) || ( COME::flagCollisionTreatment == SPHERICAL_SLIDING ) ){
		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			
			if( !(*iter)->isFixed() ){
				
				vector<COME_Vertex3D*> *vertices = (*iter)->getBuois();
				int iv;
				
				COME_Vector3D dispPosition;
				COME_Vector3D dispVelocity;
				for( iv = 0; iv < vertices->size(); iv++ ){
					
					if( dispPosition.vpModule() < ( (*vertices)[iv]->getCollisionDispAvg().vpModule()  ) )
						dispPosition = ( (*vertices)[iv]->getCollisionDispAvg()  );
					if( dispVelocity.vpModule() < ( (*vertices)[iv]->getVelocityDispAvg().vpModule() ) )
						dispVelocity = ( (*vertices)[iv]->getVelocityDispAvg() );
				}
				COME_Vector3D newVeloc = dytv[i] + dispVelocity;
				dytv[i] = newVeloc;
				
				COME_Point3D newpos( (*iter)->getGlobalPosition() + dispPosition );
				
				//////// Update position and velocity
				(*iter)->setPosition( newpos );
				(*iter)->setVelocity( dytv[i] );
			}
		}
	}
	
	/// Record positions of all molecules of this tissue in this time onto the output file.
	accu += timestep;
	if( COME::flagExportToPrecalculatedFile ){
		if( ( fileNameToSaveDeformation != "" ) && ( accu >= COME::flagAnimationResolution ) ){
			//printf( "CALL accu: %f ts: \n", accu, timestep );
			saveDeformationFile( UPDATE );
			accu = 0.0;
		}
	}
		
	makeAllLocalFrames();

	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method implements the Runge-Kutta 4 method for numeric integration
/// to calculate the new positions for all molecules of this organ.
/// If ...
/// If a deformation file is loaded, use positions from file instead of calculate them.
//////////////////////////////////////////////////////////////////////// 
bool
COME_MoleculesTissue::update( double timestep, double simClock, string fileNameToSaveDeformation ){

	int i;
	double xh , hh, h6;
	list<COME_Molecule*>::iterator iter;
	
	simTime = simClock;
	
	//if( deformation.size() <= 0 ){

		if( !dymv ){
			initForSimulation();
		}
		
		#ifdef GPU_FORCES
		// Call CG function to compute current forces on each molecule
		// in this case, derivs never calls calculateForce
		gpuForces = NULL; //TODAS AS FORAS
		#endif

		hh = timestep * 0.5;
		h6 = timestep / 6.0;
		xh = simClock + hh;
		
		// First step //////////////////////////////////////////////////////
		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			ytp[i] = ( (*iter)->getVelocity() * hh ) + (*iter)->getGlobalPosition();
			ytv[i] = ( (*iter)->getAcceleration() * hh ) + (*iter)->getVelocity();
		}

		// Second step //////////////////////////////////////////////////////
		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			(*iter)->derivs( xh, hh, ytp, ytv, dytv, dyta, i ); 
		}

		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			ytp[i] = ( dytv[i] * hh ) + (*iter)->getGlobalPosition();
			ytv[i] = ( dyta[i] * hh ) + (*iter)->getVelocity();
		}

		// Third step //////////////////////////////////////////////////////
		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			(*iter)->derivs( xh, hh, ytp, ytv, dymv, dyma, i ); 
		}

		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			ytp[i] = ( dymv[i] * timestep ) + (*iter)->getGlobalPosition();
			ytv[i] = ( dyma[i] * timestep ) + (*iter)->getVelocity();
			
			dymv[i] += dytv[i];
			dyma[i] += dyta[i];
		}

		// Fourth step //////////////////////////////////////////////////////
		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			(*iter)->derivs( simClock + timestep, timestep, ytp, ytv, dytv, dyta, i ); 
		}

		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			
			ytp[i] = ( ( ( (*iter)->getVelocity() + dytv[i] + ( dymv[i] * 2.0 ) ) * h6 ) + (*iter)->getGlobalPosition() );
			ytv[i] = ( ( ( (*iter)->getAcceleration() + dyta[i] + ( dyma[i] * 2.0 ) ) * h6 ) + (*iter)->getVelocity() );
			
			if( ( timestep < 0 ) || ( ( ytv[i].vpModule() * timestep ) < ( (*iter)->getRadius() / 20.0 ) ) ){
				// do nothing
			} else {
				//return false;
			}
		}

		i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			
			if( !(*iter)->isFixed() ){
				(*iter)->setPositionMinusOne( (*iter)->getGlobalPosition() );
				(*iter)->setPosition( ytp[i] );
				//(*iter)->setAcceleration( ( ytv[i] - (*iter)->getVelocity() ) / timestep ); // a = ( V- Vo ) / deltaT (Don't need to do it because it's done in derivs.)
				(*iter)->setVelocity( ytv[i] );
				//(*iter)->makeLocalFrame();
			}
		}
		
		////////// Apply collision response (position and velocity)
		if( ( COME::flagCollisionTreatment == NEIGHBORS ) || ( COME::flagCollisionTreatment == SPHERICAL_SLIDING ) ){
			i = 0;
			for (iter = shape.begin(); iter != shape.end(); iter++, i++){
				
				if( !(*iter)->isFixed() ){
					
					vector<COME_Vertex3D*> *vertices = (*iter)->getBuois();
					int iv;
					
					COME_Vector3D dispPosition;
					COME_Vector3D dispVelocity;
					for( iv = 0; iv < vertices->size(); iv++ ){
						
						if( dispPosition.vpModule() < ( (*vertices)[iv]->getCollisionDispAvg().vpModule()  ) )
							dispPosition = ( (*vertices)[iv]->getCollisionDispAvg()  );
						if( dispVelocity.vpModule() < ( (*vertices)[iv]->getVelocityDispAvg().vpModule() ) )
							dispVelocity = ( (*vertices)[iv]->getVelocityDispAvg() );
					}
					COME_Vector3D newVeloc = ytv[i] + dispVelocity;
					ytv[i] = newVeloc;
					
					COME_Point3D newpos( (*iter)->getGlobalPosition() + dispPosition );
					
					//////// Update position and velocity
					(*iter)->setPosition( newpos );
					(*iter)->setVelocity( ytv[i] );
				}
			}
		}
		

		/// Record positions of all molecules of this tissue in this time onto the output file.
		accu += timestep;
		if( flagExportToPrecalculatedFile ){
			if( ( fileNameToSaveDeformation != "" ) && ( accu >= COME::flagAnimationResolution ) ){
				saveDeformationFile( UPDATE );
				accu = 0.0;
			}
		}

	/*} else {
		/// Load precalculated deformation 
		accu += timestep;
		

		unsigned int t = (unsigned int)(simClock / COME::flagAnimationResolution);
		int i = 0;
		for (iter = shape.begin(); iter != shape.end(); iter++, i++){
			
			(*iter)->setPositionMinusOne( (*iter)->getGlobalPosition() );
			(*iter)->setPosition( deformation[t][i] );
			(*iter)->setStress( deformationStresses[t][i] );
			(*iter)->setStrain( deformationStrains[t][i] );
			//(*iter)->makeLocalFrame();
		}
		if( accu >= COME::flagAnimationResolution ){
			accu = 0.0;
		}

	}*/
	makeAllLocalFrames();
	return true;

}


//////////////////////////////////////////////////////////////////////
/// This method inspects the positions of all molecules within this organ
/// in the global coordinate system to calculate the minimum and maximum
/// corners of the rectangular organ envelop.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesTissue::getEnvelop( COME_Point3D& mins, COME_Point3D& maxs ){

	if( shape.size() == 0 ){
		mins = COME_Point3D();
		maxs = COME_Point3D();
		return;
	}

	mins = maxs = shape.front()->getGlobalPosition();

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		COME_Point3D minsM = (*iter)->getGlobalPosition();
		COME_Point3D maxsM = (*iter)->getGlobalPosition();

		if( minsM.getX() < mins.getX() ){
			mins.setX( minsM.getX() );
		}
		if( minsM.getY() < mins.getY() ){
			mins.setY( minsM.getY() );
		}
		if( minsM.getZ() < mins.getZ() ){
			mins.setZ( minsM.getZ() );
		}

		if( maxsM.getX() > maxs.getX() ){
			maxs.setX( maxsM.getX() );
		}
		if( maxsM.getY() > maxs.getY() ){
			maxs.setY( maxsM.getY() );
		}
		if( maxsM.getZ() > maxs.getZ() ){
			maxs.setZ( maxsM.getZ() );
		}
	}

	// if spheres size change this will not work perfectly
	mins -= shape.front()->getRadius();
	maxs += shape.back()->getRadius();

}

//////////////////////////////////////////////////////////////////////
/// This method returns the molecule of this organ which is the nearest of the given point.
//////////////////////////////////////////////////////////////////////// 
COME_Molecule*
COME_MoleculesTissue::nearestTo( COME_Point3D point ){

	list<COME_Molecule*>::iterator nearest = shape.begin();
	double minDist = point.vpDistance( (*nearest)->getPosition() );

	list<COME_Molecule*>::iterator iter;	
	for (iter = shape.begin(); iter != shape.end(); iter++){

		if( minDist > point.vpDistance( (*iter)->getPosition() ) ){

			minDist = point.vpDistance( (*iter)->getPosition() );
			nearest = iter;
		}
	}

	return (*nearest);
}

//////////////////////////////////////////////////////////////////////
/// This method returns the total mass of this tissue calculated by adding masses of all molecules.
//////////////////////////////////////////////////////////////////////// 
double
COME_MoleculesTissue::getMass(){

	return mass;
}

//////////////////////////////////////////////////////////////////////
/// This method calculates the center of mass for this bone based on its molecules
/// distribution and density. Then it stores it in the massCenter attribute. 
//////////////////////////////////////////////////////////////////////// 
COME_Point3D
COME_MoleculesTissue::getMassCenter(){

	double x, y, z, mass;
	x = y = z = mass = 0.0;

	list<COME_Molecule*>::iterator iter;	
	for (iter = shape.begin(); iter != shape.end(); iter++){

		x += (*iter)->getMass() * (*iter)->getPosition().getX();
		y += (*iter)->getMass() * (*iter)->getPosition().getY();
		z += (*iter)->getMass() * (*iter)->getPosition().getZ();

		mass += (*iter)->getMass();
	}

	COME_Point3D massCenter( x/mass, y/mass, z/mass );
	return massCenter;
}

//////////////////////////////////////////////////////////////////////
/// This method start procedure to calculate all spring constants of this tissue
/// based in Young's modulus of their neighbouring molecules. 
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesTissue::recalculateAllSpringConstants( int mode, double currTime ){

	updateAverageNumConnections();
	list<COME_Molecule*>::iterator iter;	
	for (iter = shape.begin(); iter != shape.end(); iter++){

		double increment = 1.0;
		if( mode == NUMERICALLY ){
			if( chain.currentLength > 0.0 ){
				
				chain.updateLengths( currTime );

				chain.targetLength = chain.nominalLength + ( ( (chain.force*100) * chain.nominalLength ) / ( 100000.0 * 1.0 ) );

				if( chain.currentLength > chain.targetLength ){

					if( chain.currentLength > chain.lastLength ){

						// reduce k
						increment = 1.001;

					} else {

						// do nothing
						increment = 1.0;

					}

				} else if( chain.currentLength < chain.targetLength ){

					if( chain.currentLength > chain.lastLength ){

						// do nothing
						increment = 1.0;

					} else {

						// increase k
						increment = 0.999;

					}
				}
			}
		}

		list<COME_MoleculeLink*>::iterator iterLink;
		for (iterLink = (*iter)->getConnectionList()->begin(); iterLink != (*iter)->getConnectionList()->end(); iterLink++ ){

			if( mode == NUMERICALLY ){
				(*iterLink)->recalculateSpring( NUMERICALLY, increment );
			} else {
				(*iterLink)->recalculateSpring( mode, averageNumConnectionsPerMolecule );
			}
		}
	}
}

/// Works only for full diagonals lattice models :-(
void
COME_MoleculesTissue::setSurfaceMolecules(){

	list<COME_Molecule*>::iterator iter;	
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		if( (*iter)->getConnectionList()->size() <= 17 ){
			(*iter)->setOnSurface( true );
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// This method calculates and updates the average number of connections per molecule
/// in this tissue.
//////////////////////////////////////////////////////////////////////// 
void
COME_MoleculesTissue::updateAverageNumConnections(){

	unsigned int total = 0;

	list<COME_Molecule*>::iterator iter;	
	for (iter = shape.begin(); iter != shape.end(); iter++){
		
		total += (*iter)->getConnectionList()->size();
	}

	if( shape.size() > 0 )
		averageNumConnectionsPerMolecule = total / shape.size();
	else
		averageNumConnectionsPerMolecule = 0;
}

//////////////////////////////////////////////////////////////////////
/// This method loads a binary file containing only double variables.
///	These are xyz coordinates of numberOfMolecules points in n instants of time.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::loadDeformationFile( string fileName ){

	FILE *file;

	unsigned int numberOfMolecules = getShape()->size();

	if( file = fopen( fileName.c_str(), "rb" ) ){

		while( !feof( file ) ){

			COME_Point3D *localDeformation =  new COME_Point3D [numberOfMolecules];
			double *localDeformationStresses = new double [numberOfMolecules];
			double *localDeformationStrains = new double [numberOfMolecules];
			
			for( int j = 0; j < numberOfMolecules; j++ ){

				double array[5];
				fread( array, sizeof(double), 5, file );

				localDeformation[j].setXYZ( array[0], array[1], array[2] );
				localDeformationStresses[j] = array[3];
				localDeformationStrains[j] = array[4];
			}
			deformation.push_back( localDeformation );
			deformationStresses.push_back( localDeformationStresses );
			deformationStrains.push_back( localDeformationStrains );
		}		

		fclose(file);
	}

}

//////////////////////////////////////////////////////////////////////
/// This method saves xyz coordinates and stress of all molecules of this tissue now
///	onto a binary file.
/* old code for molecules
	/*list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++ ){
		
		double array[6];
		COME_Point3D localPoint = (*iter)->getGlobalPosition(); 
		array[0] = simTime;
		array[1] = localPoint.getX();
		array[2] = localPoint.getY();
		array[3] = localPoint.getZ();
		array[4] = (*iter)->getStress();
		array[5] = (*iter)->getStrain();
		fwrite( array, sizeof(double), 6, file );
	}
*/
//////////////////////////////////////////////////////////////////////
/*
void
COME_MoleculesTissue::saveDeformationFile( int fileMode ){
	
	FILE *file;
	string strMode;

	if( fileMode == NEW ){
		strMode = "wb";// create new
		fclose( fopen( ((COME_MoleculesCartilage*)parent)->getDeformationFile().c_str(), strMode.c_str() ) );
	} else {
		strMode = "a+b"; // update at the end
	
		if( file = fopen( ((COME_MoleculesCartilage*)parent)->getDeformationFile().c_str(), strMode.c_str() ) ){
	
			vector<COME_Vertex3D> *verts = ((COME_MoleculesCartilage*)parent)->getSurface()->getVerticesGlobalPt();
			//vector<COME_Vertex3D> *vertsOriginal = ((COME_MoleculesCartilage*)parent)->getSurface()->getVerticesPt();
			for ( int i = 0; i < verts->size(); i++ ){

				COME_Vertex3D vertex = verts->at(i);
				double array[6];
				array[0] = simTime;
				array[1] = vertex.x;
				array[2] = vertex.y;
				array[3] = vertex.z;
				array[4] = vertex.getStress();
				array[5] = vertex.getStrain();
				fwrite( array, sizeof(double), 6, file );
			}
			fclose(file);
		}
	}
}
*/

void
COME_MoleculesTissue::saveDeformationFile( int fileMode ){
	
	FILE *file;
	string strMode;

	if( fileMode == NEW ){
		strMode = "wb";// create new
		
		fclose( fopen( ((COME_MoleculesCartilage*)parent)->getDeformationFile().c_str(), strMode.c_str() ) );
		
		string f_namei = ((COME_MoleculesCartilage*)parent)->getDeformationFile();
		string tempStri( f_namei, 0, f_namei.size()-10 );
		f_namei = tempStri + "stress.dat";
		fclose( fopen( f_namei.c_str(), strMode.c_str() ) );

		framesOnFile = 0;

	} else {
		strMode = "a+b"; // update at the end
	
		// save vertices
		if( file = fopen( ((COME_MoleculesCartilage*)parent)->getDeformationFile().c_str(), strMode.c_str() ) ){
	
			vector<COME_Vertex3D> *verts = ((COME_MoleculesCartilage*)parent)->getSurface()->getVerticesGlobalPt();
			for ( int i = 0; i < verts->size(); i++ ){

				COME_Vertex3D vertex = verts->at(i);
				double array[3];
				array[0] = vertex.x;
				array[1] = vertex.y;
				array[2] = vertex.z;
				fwrite( array, sizeof(double), 3, file );
			}
			fclose(file);
			framesOnFile++;
		}

		// save stress
		string f_name = ((COME_MoleculesCartilage*)parent)->getDeformationFile();
		string tempStr( f_name, 0, f_name.size()-10 );
		f_name = tempStr + "stress.dat";

		if( file = fopen( f_name.c_str(), strMode.c_str() ) ){
	
			vector<COME_Vertex3D> *verts = ((COME_MoleculesCartilage*)parent)->getSurface()->getVerticesGlobalPt();
			for ( int i = 0; i < verts->size(); i++ ){

				COME_Vertex3D vertex = verts->at(i);
				double array[1];
				array[0] = vertex.getStress();
				fwrite( array, sizeof(double), 1, file );
			}
			fclose(file);
		}
	}
}

// visco //
void
COME_MoleculesTissue::flow( double timestep ){

	list<COME_Molecule*>::iterator iter;

	//if( (*iter)->getPermeability() != 0.0 ){
	
	for (iter = shape.begin(); iter != shape.end(); iter++ ){
	
		double volume = (*iter)->getLiquidFraction() * ( 4.0 / 3.0 ) * M_PI * pow( (*iter)->getNominalRadius(), 3 ) * (*iter)->getLiquidLevel();
		
		double totalFlowingVolume = (*iter)->getStress() * volume * (*iter)->getPermeability() / timestep;

		// Update radius of the current molecule from new calculated volume
		//printf( "avant: %f ", (*iter)->getRadius() );
		//(*iter)->setRadius( pow( ( 3.0 * ( (*iter)->getVolume() - totalFlowingVolume ) ) / ( 4.0 * M_PI ), 1.0/3.0 ) );
		//printf( "apres: %f \n", (*iter)->getRadius() );
		//getch();


		//if( (*iter)->isOnSurface() ){
		//	totalFlowingVolume = 0.8 * totalFlowingVolume;
		//}

		list<COME_MoleculeLink*>::iterator iterO;
		double totalStress = 0.0;
		double *stressArray = new double [(*iter)->getConnectionList()->size()];
		int iStress;
		for (iterO = (*iter)->getConnectionList()->begin(), iStress = 0; iterO != (*iter)->getConnectionList()->end(); iterO++, iStress++ ){
			
			COME_Molecule *temp;
			if( (*iterO)->getElement(0) == (*iter) ){
				temp = (*iterO)->getElement(1);
			} else {
				temp = (*iterO)->getElement(0);
			}

			totalStress += temp->getStress();
			stressArray[iStress] = temp->getStress();
		}

		for (iterO = (*iter)->getConnectionList()->begin(), iStress = (*iter)->getConnectionList()->size()-1 ; iterO != (*iter)->getConnectionList()->end(); iterO++, iStress-- ){
			
			COME_Molecule *temp;
			if( (*iterO)->getElement(0) == (*iter) ){
				temp = (*iterO)->getElement(1);
			} else {
				temp = (*iterO)->getElement(0);
			}

			// 
			if( totalStress ){
				(*iterO)->setFlowVolume( (*iter), ( stressArray[iStress] / totalStress ) * totalFlowingVolume );
			}

		}
						
	}

	for (iter = shape.begin(); iter != shape.end(); iter++ ){

		list<COME_MoleculeLink*>::iterator iterO;
		double totalFlowingVolume = 0.0;
		for (iterO = (*iter)->getConnectionList()->begin(); iterO != (*iter)->getConnectionList()->end(); iterO++ ){
			
			totalFlowingVolume += (*iterO)->getFlowVolume(*iter);
		}

		if( (*iter)->isOnSurface() ){
			
			totalFlowingVolume = 1.2 * totalFlowingVolume;
		}

		(*iter)->setRadius( pow( ( 3.0 * ( (*iter)->getVolume() + totalFlowingVolume ) ) / ( 4.0 * M_PI ), 1.0/3.0 ) );

		for (iterO = (*iter)->getConnectionList()->begin(); iterO != (*iter)->getConnectionList()->end(); iterO++ ){
		
			COME_Molecule *temp;
			if( (*iterO)->getElement(0) == (*iter) ){
				temp = (*iterO)->getElement(1);
			} else {
				temp = (*iterO)->getElement(0);
			}
			
			double incDist = fabs( ( (*iterO)->getViscousFraction() * ( temp->getRadius() + (*iter)->getRadius() ) ) - (*iterO)->getNominalDist() );

			if( (*iterO)->getElement(1)->getPosition().vpDistance( (*iterO)->getElement(2)->getPosition() ) > (*iterO)->getNominalDist() ){
				(*iterO)->setNominalDist( (*iterO)->getNominalDist() - incDist );
			} else {
				(*iterO)->setNominalDist( (*iterO)->getNominalDist() + incDist );
			}
		}

	}
}


//////////////////////////////////////////////////////////////////////
/// Apply the transformation given by the matrix M to the fixed free molecules.
/// Used to transform one side attached structures like the cartilages.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue :: transform( COME_Matrix M ){

	list<COME_Molecule*>::iterator iter;
	for( iter = shape.begin(); iter != shape.end(); iter++ ){
		
		if( (*iter)->isFixed() ){

			(*iter)->setPosition( M * (*iter)->getPosition() );

		}
	}
}

//////////////////////////////////////////////////////////////////////
/// Apply the transformation given by the matrix M to the fixed free molecules
/// and MP to the molecules fixed at the origin.
/// Used to transform two side attached structures like the ligaments.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue :: transform( COME_Matrix M, COME_Matrix MP ){

	list<COME_Molecule*>::iterator iter;
	for( iter = shape.begin(); iter != shape.end(); iter++ ){
		
		if( ( (*iter)->isFixed() ) && ( (*iter)->getDescription() == "origin" ) ){

			(*iter)->setPosition( MP * (*iter)->getPosition() );

		} else if( ( (*iter)->isFixed() ) && ( (*iter)->getDescription() == "free" ) ){

			(*iter)->setPosition( M * (*iter)->getPosition() );
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// Scale the geometry of the tissue by the factor.
/// It scales the molecules positions and radius.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue :: scale( double factorx, double factory, double factorz ){

	// Find central point
	double accX = 0.0, accY = 0.0, accZ = 0.0;
	int n = 0;
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++, n++ ){
		accX+=(*iter)->getPosition().getX();
		accY+=(*iter)->getPosition().getY();
		accZ+=(*iter)->getPosition().getZ();
	}
	accX/=(double)n; accY/=(double)n; accZ/=(double)n;
	
	// Translate to origin
	translate( -accX, -accY, -accZ );
	
	//Apply scaling
	for (iter = shape.begin(); iter != shape.end(); iter++){

		COME_Point3D newPos( (*iter)->getPosition().getX() * factorx, (*iter)->getPosition().getY() * factory, (*iter)->getPosition().getZ() * factorz );
		(*iter)->setPosition( newPos );
		(*iter)->setRadius( (*iter)->getRadius() * ( ( factorx + factory + factorz ) / 3.0 ) );
	}
	
	// Translate back to position
	translate( accX, accY, accZ );

}

//////////////////////////////////////////////////////////////////////
/// Translate all molecules by the distances.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue :: translate( double dx, double dy, double dz ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++ ){
	
		COME_Point3D newPos( (*iter)->getPosition().getX() + dx, (*iter)->getPosition().getY() + dy, (*iter)->getPosition().getZ() + dz );
		(*iter)->setPosition( newPos );
	}

}

//////////////////////////////////////////////////////////////////////
/// Rotate all molecules of the tissue by the angles in the order xyz.
/// To rotate in another order, make several consecutive calls to this
/// function with zero angles for the axis you don't want to rotate.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue :: rotate( double rx, double ry, double rz ){

	// Find central point
	double accX = 0.0, accY = 0.0, accZ = 0.0;
	int n = 0;
	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++, n++ ){
		accX+=(*iter)->getPosition().getX();
		accY+=(*iter)->getPosition().getY();
		accZ+=(*iter)->getPosition().getZ();
	}
	accX/=(double)n; accY/=(double)n; accZ/=(double)n;
	
	// Translate to origin
	translate( -accX, -accY, -accZ );
	
	// Apply rotations
	for (iter = shape.begin(); iter != shape.end(); iter++){

		COME_Point3D newPos;
		/// Rotate on X
		newPos = COME_Point3D (
			(*iter)->getPosition().getX(),
			(*iter)->getPosition().getY() * cos( rx ) + (*iter)->getPosition().getZ() * sin( rx ),
			(*iter)->getPosition().getZ() * cos( rx ) - (*iter)->getPosition().getY() * sin( rx )
		);
		(*iter)->setPosition( newPos );
		/// Rotate on Y
		newPos = COME_Point3D (
			(*iter)->getPosition().getX() * cos( ry ) - (*iter)->getPosition().getZ() * sin( ry ),
			(*iter)->getPosition().getY(),
			(*iter)->getPosition().getX() * sin( ry ) + (*iter)->getPosition().getZ() * cos( ry )
		);
		(*iter)->setPosition( newPos );
		/// Rotate on Z
		newPos = COME_Point3D (
			(*iter)->getPosition().getX() * cos( rz ) + (*iter)->getPosition().getY() * sin( rz ),
			(*iter)->getPosition().getY() * cos( rz ) - (*iter)->getPosition().getX() * sin( rz ),
			(*iter)->getPosition().getZ()
		);
		(*iter)->setPosition( newPos );
	}
	
	// Translate back to position
	translate( accX, accY, accZ );

}

void
COME_MoleculesTissue::scaleNominals( double factor ){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++ ){
		list<COME_MoleculeLink*>::iterator iterO;
		for (iterO = (*iter)->getConnectionList()->begin(); iterO != (*iter)->getConnectionList()->end(); iterO++ ){

			(*iterO)->setNominalDist( COME_Vector3D( (*iterO)->getElement(0)->getPosition() - (*iterO)->getElement(1)->getPosition() ).vpModule() * factor );
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// Calls COME_Molecule::makeLocalFrame to update the local reference frame for skinning of each molecule.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::makeAllLocalFrames(){
	
	if (COME::flagSkinning ){
		//printf( "Making all local frames\n" );
		list<COME_Molecule*>::iterator iter;
		for (iter = shape.begin(); iter != shape.end(); iter++){
			(*iter)->makeLocalFrame();
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// Save current position of all molecules as initial such that it's possible to reset later.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::checkInitialPositions(){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		(*iter)->checkPosition();
	}
}

//////////////////////////////////////////////////////////////////////
/// Load initial positions of all molecules reseting them.
//////////////////////////////////////////////////////////////////////
void
COME_MoleculesTissue::resetInitialPositions(){

	list<COME_Molecule*>::iterator iter;
	for (iter = shape.begin(); iter != shape.end(); iter++){
		(*iter)->resetPosition();
	}
}
