/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

///////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  AUTHOR......: Anderson Maciel
//  DATE........: June/05/2002
//  DESCRIPTION.: Class definition.
//
///////////////////////////////////////////////////////////////////

#include	<general/comescenario.h>
#include	<physics/comeforce.h>
#include	<physics/comecollide.h>
#include	<bio/comepatient.h>
#include	<bio/comemolecule.h>
#include	<bio/comemoleculescartilage.h>
#include	<bio/comemoleculesbone.h>
#include	<bio/comeclamp.h>
#include	<general/comesimulator.h>
#include	<general/comexml.h>
#include	<algebra/comevector3d.h>
#include	<kinematics/comepolyaxialjoint.h>
#include	<kinematics/comemovement.h>
#include	<math.h>

#ifndef _WIN32
#include	<sys/time.h>
#else 
#include	<time.h>
#endif

#include	<zthread/Mutex.h>
#define	_MECHATESTER_APP
//////////////////////////////////////////////////////////////////////
/// Construction/Destruction
//////////////////////////////////////////////////////////////////////
using namespace ZThread;

static double accu;

COME_Scenario::COME_Scenario(){

	availableMaterials = new vector<COME_Material*>;
	movement = NULL;
	gravity = COME_Vector3D( 0.0, 0.0, 0.0 );
	collisions = new COME_Collide( this );
	modified = false;
	patientList.push_back( new COME_Patient() );
	patientList.front()->setParent( this );
	mutex = new Mutex();
	accu = 0.0;
}

COME_Scenario::~COME_Scenario(){

}


//////////////////////////////////////////////////////////////////////
/// Setting
//////////////////////////////////////////////////////////////////////
void
COME_Scenario::setPatientList( list<COME_Patient*> &listN ){

	patientList = listN;
}

void
COME_Scenario::setGravity( const COME_Vector3D &gravityN ){
	
	gravity = gravityN;
}

//////////////////////////////////////////////////////////////////////
/// Getting
//////////////////////////////////////////////////////////////////////
COME_Patient*
COME_Scenario::getPatient( int index ) const {

	list<COME_Patient*>::const_iterator iter;
	int i = 0;
	for (iter = patientList.begin(); !(iter == patientList.end()); iter++){
		if( i == index ){
			return (*iter);
		}
		i++;
	}
	return NULL;
}

list<COME_Patient*>*
COME_Scenario::getPatientList() {

	return &patientList;
}

const COME_Vector3D
COME_Scenario::getGravity() const {

	return gravity;
}

COME_Collide*
COME_Scenario::getCollisionDetector() const {

	return collisions;
}

vector<COME_Material*>*
COME_Scenario::getAvailableMaterialsList() const {

	return availableMaterials;
}

//////////////////////////////////////////////////////////////////////
/// This method changes the state of the scenario to a state distant of
/// the current one of 1 timestep in the future or past depending of the
/// direction value. Note that in the case when more than one patient
/// exists, only the last inserted will be simulated.
//////////////////////////////////////////////////////////////////////
bool
COME_Scenario::simulate( double timestep, double simClock, bool direction ){

	// Variables to save axes into files
	//double currFlex = 0.0, currAdduct = 0.0, currTwist = 0.0;

	//lock();
	#ifndef _WIN32
	struct timeval *Tps, *Tpf;
	Tps = (struct timeval*) malloc(sizeof(struct timeval));
	Tpf = (struct timeval*) malloc(sizeof(struct timeval));
	#endif

	// update kinematical matrices

	int indPat = patientList.size() - 1;
	
	if( movement ){

		#ifndef _WIN32
		gettimeofday (Tps, NULL );
		#endif
		
		movement->setTime_Err( timestep / 2.0 );
		
		for( int i = 0; i < movement->getQtdMotion(); i++ ){
  	
			COME_Joint *jointAux = getPatient(indPat)->getJointByName( movement->getTimeline()[i].getJointName() );
 	
			if( ( movement->getTimeline()[i].getTimeIni() > simClock - movement->getTime_Err() ) && ( movement->getTimeline()[i].getTimeIni() <= simClock + movement->getTime_Err() ) ){
				//calculate deltaS
				double gds = 0;
				switch( movement->getTimeline()[i].getMotionType() ){
					case FLEX	:	gds = movement->getTimeline()[i].getParameter() - (((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX])->getCurrent();
								break;
					case ADDUCT	:	gds = movement->getTimeline()[i].getParameter() - (((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT])->getCurrent();
								break;
					case TWIST	:	gds = movement->getTimeline()[i].getParameter() - (((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST])->getCurrent();
								break;
				}
  	        		double gdt = ( movement->getTimeline()[i].getTimeFin() - movement->getTimeline()[i].getTimeIni() ) / timestep;
  	        		movement->getTimeline()[i].setDeltaS( gds / gdt );
			}

			// move to new position	
			double newPos = 0.0;
      			switch( movement->getTimeline()[i].getMotionType() ){
        			case FLEX   : newPos = (((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX])->getCurrent() + movement->getTimeline()[i].getDeltaS();
					((COME_PolyaxialJoint*)jointAux )->setFlexionTo( newPos );
					//printf( "FLEX:%f\n", newPos );
					//currFlex = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMin() ) * newPos + ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMin();
					break;
				case ADDUCT : newPos = (((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT])->getCurrent() + movement->getTimeline()[i].getDeltaS();
					((COME_PolyaxialJoint*)jointAux )->setAdductionTo( newPos );
					//printf( "ADDUCT:%f\n", newPos );
					//currAdduct = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMin() ) * newPos + ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMin();
					break;
				case TWIST  : newPos = (((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST])->getCurrent() + movement->getTimeline()[i].getDeltaS();
					((COME_PolyaxialJoint*)jointAux )->setTwistTo( newPos );
					//printf( "TWIST:%f\n", newPos );
					//currTwist = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMin() ) * newPos + ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMin();
					break;
			}


			if( ( movement->getTimeline()[i].getTimeFin() > simClock - movement->getTime_Err() ) && ( movement->getTimeline()[i].getTimeFin() <= simClock + movement->getTime_Err() ) ){
				//reset deltaS
				movement->getTimeline()[i].setDeltaS( 0.0000000000 );
			}
		}
	}

	// Apply changes. Should be replaced by a Patient->Update to be executed only if there are changes
	getPatient(indPat)->getRootJoint()->vpMakeGims( COME_Matrix() );
	#ifndef _WIN32
	gettimeofday (Tpf, NULL); 
	((COME_Simulator*)parent)->difKinematics += (Tpf->tv_sec-Tps->tv_sec)*1000000 + Tpf->tv_usec-Tps->tv_usec;
	gettimeofday (Tps, NULL );
	#endif
	
	/// Apply physics and deformation
	if( COME::flagCollisionTreatment == NEIGHBORS ){
	
		collisions->updateProximityStructure();
		collisions->geometricResponse();
	} else if( COME::flagCollisionTreatment == SPHERICAL_SLIDING ){
	
		collisions->sphericalSlidingResponse();
	}
	#ifndef _WIN32
	gettimeofday (Tpf, NULL); 
	((COME_Simulator*)parent)->difCollisions += (Tpf->tv_sec-Tps->tv_sec)*1000000 + Tpf->tv_usec-Tps->tv_usec;
	#endif

	
	list<COME_Patient*>::iterator iter;
	for (iter = patientList.begin(); !(iter == patientList.end()); iter++){
		#ifndef _WIN32
		gettimeofday (Tps, NULL );
		#endif
		if( direction == FUTURE ){
			if( !(*iter)->update( timestep, simClock ) ){
				return false;
			}
		} else {
			if( !(*iter)->update( -timestep, simClock )){
				return false;
			}
		}
		
		#ifndef _WIN32
		gettimeofday (Tpf, NULL); 
		((COME_Simulator*)parent)->difDeformation += (Tpf->tv_sec-Tps->tv_sec)*1000000 + Tpf->tv_usec-Tps->tv_usec;
		#endif

		#ifdef _MECHATESTER_APP
		// Update mobile clamp position during force test
		COME_Vector3D dispClamp;
		double totalVectors = 0.0;
		list<COME_BioStructure*>::const_iterator iterOrgans;
		for( iterOrgans = ((list<COME_BioStructure *>*)getPatient(indPat)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)getPatient(indPat)->getPtOrganList())->end(); iterOrgans++  ){
		
			list<COME_Molecule*>* molecules = ((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->getShape();
			list<COME_Molecule*>::iterator iterMolecules;
			for ( iterMolecules = molecules->begin(); iterMolecules != molecules->end(); iterMolecules++){
			
				if( (*iterMolecules)->intersectClampMobile() ){
					dispClamp += (*iterMolecules)->getPosition() - (*iterMolecules)->getPositionMinusOne( );
					totalVectors+=1.0;
				} 
			}
		}
		if ( totalVectors ){
			getPatient(indPat)->getOrgan("clamp1")->translate( dispClamp / totalVectors);
			getPatient(indPat)->getOrgan("clamp1")->updateSkin();
		}
		#endif // MECHATESTER
	}
	if( COME::flagCollisionTreatment == NEIGHBORS ){
		#ifndef _WIN32
		gettimeofday (Tps, NULL );
		#endif
		collisions->resetProximitiesDisps();
		#ifndef _WIN32
		gettimeofday (Tpf, NULL); 
		((COME_Simulator*)parent)->difCollisions += (Tpf->tv_sec-Tps->tv_sec)*1000000 + Tpf->tv_usec-Tps->tv_usec;
		#endif
	}
	
	
	// Detect collisions between the objects of the scene
	if( COME::flagCollisionTreatment == SPHERES ){
	
		collisions->treatCollisions();

	} else if( ( COME::flagCollisionTreatment == MESH ) || (COME::flagCollisionTreatment == HYBRID )){
		
		collisions->detectContacts();

		vector<COME_Collision> c = collisions->getCollisions();
		for (int k=0; k < c.size(); k++ ){
			
			COME_BioStructure *organ1 = c[k].getOrgan1();
			COME_BioStructure *organ2 = c[k].getOrgan2();
			
			COME_Force f1 = c[k].getForceInFace( organ1 );
			COME_Force f2 = c[k].getForceInFace( organ2 );

			int face1 = c[k].getFace( organ1 );
			int face2 = c[k].getFace( organ2 );

			organ1->getSurface()->getAFace( face1 ).getNearestMolecule()->addCollisionForce( f1 );
			organ2->getSurface()->getAFace( face2 ).getNearestMolecule()->addCollisionForce( f2 );
			
		}
	} else if( COME::flagCollisionTreatment == DISPLACEMENT ){
	
		if( collisions->detectContactsDisplacement() ){
		
			list<COME_BioStructure*>::const_iterator iterOrgans;
			COME_Patient *activePatient = getPatient(getPatientList()->size()-1);
			for( iterOrgans = activePatient->getPtOrganList()->begin(); iterOrgans != activePatient->getPtOrganList()->end(); iterOrgans++ ){
		
				if( (*iterOrgans)->isFixed() ){
					((COME_MoleculesBone*)(*iterOrgans))->respondCollision();
				} else {
					((COME_MoleculesCartilage*)(*iterOrgans))->respondCollision();
				}
			}
		}
	
	} else if( COME::flagCollisionTreatment == NEIGHBORS ){
	
		// DO NOTHING: Code is integrated in updateEuler
	
	} else if( COME::flagCollisionTreatment == SPHERICAL_SLIDING ){
	
		// DO NOTHING: Code is integrated in updateEuler
	
	} else {
		// NO COLLISION DETECTION
	}
	
	accu += timestep;
	if( flagExportToPrecalculatedFile ){
		if( accu >= COME::flagAnimationResolution ){
			saveJointIntoFile( getPatient(0)->getRootJoint( ), UPDATE );
			accu = 0.0;
		}
	}
	
	//unlock();

	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method initializes all files for precalculation
//////////////////////////////////////////////////////////////////////// 
void
COME_Scenario::createPrecalculatedFiles() {

	list<COME_Patient*>::iterator iter;
	for (iter = patientList.begin(); !(iter == patientList.end()); iter++){
		list<COME_BioStructure*>::const_iterator iterOrgans;
		for( iterOrgans = ((list<COME_BioStructure *>*)(*iter)->getPtOrganList())->begin(); iterOrgans != ((list<COME_BioStructure *>*)(*iter)->getPtOrganList())->end(); iterOrgans++  ){
			//if( !((COME_MoleculesCartilage*)(*iterOrgans))->isFixed() )
				((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->saveDeformationFile( NEW ); // to create empty file
				//((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->saveDeformationFile( UPDATE ); // to save zero position
		}
	}

	if( getPatient(0) ){
		if( getPatient(0)->getRootJoint( )->getDescription() == "root" ){
			saveJointIntoFile( getPatient(0)->getRootJoint( ), NEW ); // to create empty file
			//saveJointIntoFile( getPatient(0)->getRootJoint( ), UPDATE ); // to save zero position
		}
	}

}


//////////////////////////////////////////////////////////////////////
/// This method saves the current status of all joints of the hierarchy from jointAux into their respective file
/// It stores the transformation the joint represent in the file.
/// In other words, export all joint's LIMs to files such that one can play the motion afterwards
//////////////////////////////////////////////////////////////////////// 
void
COME_Scenario::saveJointIntoFile( COME_Joint *jointAux, int fileMode ) {
	
	// Save axes and angles into file
	FILE *file;
	string strMode;
	string fileName = COME::baseFolder + "/" + jointAux->getDescription() + "angles.dat";
	
	if( fileMode == NEW ){
		strMode = "wb";// create new
		fclose( fopen( fileName.c_str(), strMode.c_str() ) );
	} else {
		strMode = "a+b"; // update at the end

		if( file = fopen( fileName.c_str(), strMode.c_str() ) ){
		
			double *matrixAux[4];
			matrixAux[0] = new double[4];
			matrixAux[1] = new double[4];
			matrixAux[2] = new double[4];
			matrixAux[3] = new double[4];
			
			jointAux->getLim().getMatrixD( matrixAux ); 
		
			double array[16];
			array[0] = ((COME_Simulator*)parent)->getClock();
			
			array[1] = matrixAux[0][0];
			array[2] = matrixAux[1][0];
			array[3] = matrixAux[2][0];
		
			array[4] = matrixAux[0][1];
			array[5] = matrixAux[1][1];
			array[6] = matrixAux[2][1];
		
			array[7] = matrixAux[0][2];
			array[8] = matrixAux[1][2];
			array[9] = matrixAux[2][2];
		
			array[10] = matrixAux[3][0];
			array[11] = matrixAux[3][1];
			array[12] = matrixAux[3][2];
			
			array[13] = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMin() ) * ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getCurrent() + ((COME_PolyaxialJoint*)jointAux )->getDofList()[FLEX]->getMin();
		
			array[14] = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMin() ) * ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getCurrent() + ((COME_PolyaxialJoint*)jointAux )->getDofList()[ADDUCT]->getMin();
		
			array[15] = ( ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMax() - ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMin() ) * ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getCurrent() + ((COME_PolyaxialJoint*)jointAux )->getDofList()[TWIST]->getMin();
		
			fwrite( array, sizeof(double), 16, file );
			
			fclose(file);
		}
	}
	
	list<COME_Joint*>::iterator childAux;
	int childCount = 0;
	for( childAux = jointAux->getChildListPt()->begin(); childCount < jointAux->getChildListPt()->size();  childAux++, childCount++ ){
		saveJointIntoFile( (*childAux), fileMode );
	}
}

//////////////////////////////////////////////////////////////////////
/// This method creates patients and simulation parameters from a description file.
/// Such file is in XML format and "comepatient.dtd" keeps its sintaxe definition.
//////////////////////////////////////////////////////////////////////// 
bool
COME_Scenario::loadFile( string file_name, COME_Simulator *simulator ){

	COME_Xml doc;

	// load scene 
	doc.loadSceneFile( file_name, patientList, simulator, this );
	
	vector<COME_Material*>* newMaterials = doc.getAllMaterials();
	for( int i = 0; i < newMaterials->size(); i++ ){
		availableMaterials->push_back( (*newMaterials)[i] );
	}
		
	// load movements
	movement = new COME_Movement();
	movement->vpLoadMotionFile( file_name.c_str() );
	movement->setTime_Err( simulator->getTimestep() / 2.0 );
	
	parent = simulator;
	simulator->setScene( this );
	
	/// create interior lists for organs collisions
	for( int iS = 0; iS < getPatient(0)->getPtOrganList()->size(); iS++ ){

		for( int iI = 0; iI < getPatient(0)->getOrgan(iS)->internalsDesc.size(); iI++ ){
		
			getPatient(0)->getOrgan(iS)->internals.push_back( getPatient(0)->getOrgan(getPatient(0)->getOrgan(iS)->internalsDesc[iI] ) );
		}
	}
	
	modified = true;

	return true;
}


//////////////////////////////////////////////////////////////////////
/// Run all organs and create hash tables when needed
//////////////////////////////////////////////////////////////////////// 
void
COME_Scenario::initializeAllHashs(){

	for( int iS = 0; iS < getPatient(0)->getPtOrganList()->size(); iS++ ){
		if( getPatient(0)->getOrgan(iS)->internalsDesc.size() > 0 ){
			getPatient(0)->getOrgan(iS)->initializeHashStructureRayCasting();
			printf( "Organ %s initialized for spherical sliding. \n", getPatient(0)->getOrgan(iS)->getDescription().c_str() );
		}
	}
}

//////////////////////////////////////////////////////////////////////
/// This method adds an organ rt the first patient from a description file.
/// Such file is in XML format and "comepatient.dtd" keeps its sintaxe definition.
//////////////////////////////////////////////////////////////////////// 
bool
COME_Scenario::addOrganFromFile( string file_name, COME_Simulator *simulator ){

	COME_Xml doc( availableMaterials );

	// load scene 
	COME_MoleculesCartilage *newOrgan = (COME_MoleculesCartilage*)doc.loadOrganFile( file_name, patientList, simulator, this );
	
	//((COME_MoleculesCartilage*)newOrgan)->initializeSkinning();
	//printf( "skinning initialized \n" );
	//newOrgan->updateSkin();
	//printf( "skin updated \n" );
	
	// Following lines (adding new materials is not necessary because
	// availableMaterials passed above is already updated in loadOrganFile
	/*vector<COME_Material*>* newMaterials = doc.getAllMaterials();
	for( int i = 0; i < newMaterials->size(); i++ ){
		availableMaterials->push_back( (*newMaterials)[i] );
	}*/

	newOrgan->getTissue()->setMoleculesMaterial( (*availableMaterials)[0] );
	getPatient(0)->addOrgan( newOrgan );

	parent = simulator;
	simulator->setScene( this );
	
	modified = true;
	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method add all materials from a given file.
/// Such file is in XML format and "comepatient.dtd" keeps its sintaxe definition.
//////////////////////////////////////////////////////////////////////// 
bool
COME_Scenario::addMaterialsFromFile( string file_name, COME_Simulator *simulator ){

	COME_Xml doc;

	// load scene 
	doc.loadMaterialsFile( file_name, patientList, simulator, this );
	
	vector<COME_Material*>* newMaterials = doc.getAllMaterials();
	for( int i = 0; i < newMaterials->size(); i++ ){
		availableMaterials->push_back( (*newMaterials)[i] );
	}

	modified = true;
	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method saves patients and simulation parameters to a description file.
/// Such file is in XML format and "comepatient.dtd" keeps its sintaxe definition.
//////////////////////////////////////////////////////////////////////// 
bool
COME_Scenario::saveFile( string file_name, COME_Simulator *simulator ){

	COME_Xml docS;
	docS.saveSceneFile( file_name, patientList, simulator );

	return true;
}


/// Deprecated
/// Remove this function (comment date: 07/05/2004
/*bool
COME_Scenario::loadOrgans( vector<string> organFiles, int mode ){

	double xResolution = 0.01;
	double yResolution = 0.01;
	double zResolution = 0.01;
	double scaleFactorFile = 0.005; //-00


	/// Parameters derived from resolution  --- initial test
	double fracture_dist = (xResolution+yResolution+zResolution)/3 + (xResolution+yResolution+zResolution)/6;
	double friction_const = 0.3;
	double radius = fracture_dist/2;

	for( int iF = 0; iF < organFiles.size(); iF++ ){

		COME_MoleculesCartilage *newOrgan = new COME_MoleculesCartilage();
		newOrgan->setParent( patientList.front() );
		
		COME_Mesh *organMainMesh;
		//organMainMesh.loadFile( organFiles[iF], scaleFactorFile );
		organMainMesh->loadFile( organFiles[iF], scaleFactorFile, COME_Point3D( -97.9, 4.5, -25.7 ) );
		
		if( mode == MARCHING_CUBES ){

			COME_Point3D mins, maxs;
			organMainMesh->getEnvelop( mins, maxs );
			// Marchig cubes to create balls
			for( double xMcubes = mins.getX(); xMcubes <= maxs.getX(); xMcubes += xResolution ){
				for( double yMcubes = mins.getY(); yMcubes <= maxs.getY(); yMcubes += yResolution ){
					for( double zMcubes = mins.getZ(); zMcubes <= maxs.getZ(); zMcubes += zResolution ){

						COME_Point3D currentPoint( xMcubes, yMcubes, zMcubes );
						if( organMainMesh->isInside( currentPoint ) ){
							
							// Attention: Using allways the same material
							COME_Molecule *molecule = new COME_Molecule( currentPoint, radius, friction_const, (*availableMaterials)[0], COME_Vector3D(), (COME_BioStructure*)newOrgan );
							molecule->setFixed( true );
							newOrgan->getTissue()->addMolecule( molecule );

						}
						printf( "." );
					}
					printf( "\n" );
				}
				printf( "\n" );
			}
		} else if( mode == VERTICES ){

			double layersThickness = 0.045;
			double minDist = 0.016;
			radius = 0.02;
			
			int layersNumber = 0;
			COME_Mesh *organMesh = organMainMesh;
			
			for( int numFaces = 1; numFaces >= 1;  ){

				//organMesh.scaleOnNormals( layersThickness );
				organMesh->scale( 0.85 ); //1.12 for pelvis cartilage
				//numFaces = organMesh.simplify( minDist );
				//printf( "Remeshed to %d faces.\n", numFaces );
				//getchar();

				if( numFaces > 0 ){
					// Create balls on each vertex of the mesh
					for( int iVert = 0; iVert < organMesh->getVertices().size(); iVert++ ){

						COME_Point3D pointPosition = COME_Point3D( organMesh->getAVertex( iVert ));
						COME_Molecule *molecule = new COME_Molecule( pointPosition, radius, friction_const, (*availableMaterials)[0], COME_Vector3D(), newOrgan );
						if( layersNumber == 1 ){

							molecule->setMaterial( (*availableMaterials)[1] );

						}
						newOrgan->getTissue()->addMolecule( molecule, minDist );
					}
				}
				layersNumber++;
				if( layersNumber == 2 )
					numFaces = 0;
			}
		}

		newOrgan->setSurface( organMainMesh );
		patientList.front()->addOrgan( newOrgan );
		printf( "Finished loading organ.\n" );
	}

	return true;
}
*/
bool
COME_Scenario::loadOrganMesh( string organFile, int type, double scaleFactor ){

	COME_BioStructure *newOrgan;

	switch( type ){
		case CARTILAGE	: newOrgan = new COME_MoleculesCartilage(); newOrgan->setFixed( false ); break;
		case LIGAMENT	: newOrgan = new COME_MoleculesCartilage(); newOrgan->setFixed( false ); break;
		case BONE	: newOrgan = new COME_MoleculesBone(); newOrgan->setFixed( true ); break;
		case CLAMP	: newOrgan = new COME_Clamp(); newOrgan->setFixed( true ); break;
		//case LIGAMENT	: newOrgan = new COME_MoleculesLigament(); break;
	}
	newOrgan->setType( type );
	newOrgan->setParent( patientList.front() );
	patientList.front()->addOrgan( newOrgan );

	COME_Mesh *organMainMesh = new COME_Mesh();
	// translation for pubofemoral to RNW04
	//organMainMesh->loadFile( organFile, scaleFactor, COME_Point3D( 21.0, -4.5, 8.76 ) );
	// translation for pubofemoral to DEMO
	//organMainMesh->loadFile( organFile, scaleFactor, COME_Point3D( 21.0, -4.5, 8.76 ) );
	
	// APPLY ALL TRANSLATIONS
	COME_Point3D oldhip( -97.9, 4.5, -25.7 );
	//COME_Point3D labrum( -119.8, 8.535, -34.84 );
	//COME_Point3D pubofemoral( -118.9, 9, -34.46 );
	COME_Point3D labrum( 21.9, -4.035, 9.14 );
	COME_Point3D pubofemoral( 21.0, -4.5, 8.76 );
	COME_Point3D comedemo( 0.0, 0.0, -2.5 );
	
	string descOrgan = "fuck"; //"comedemo"; //////////////////////////////////////////// CHANGE THE STRING HERE TO SWITCH TRANSLATIONS
	if( descOrgan == "labrum")
		organMainMesh->loadFile( organFile, scaleFactor, labrum );
	else if( ( descOrgan == "pubofemoral") || ( descOrgan == "ischiofemoral") )
		organMainMesh->loadFile( organFile, scaleFactor, pubofemoral );
	else if( descOrgan == "pelviscart")
		organMainMesh->loadFile( organFile, scaleFactor, oldhip );
	else if( descOrgan == "pelvisbone")
		organMainMesh->loadFile( organFile, scaleFactor, oldhip );
	else if( descOrgan == "femurcart")
		organMainMesh->loadFile( organFile, scaleFactor, oldhip );
	else if( descOrgan == "femurbone")
		organMainMesh->loadFile( organFile, scaleFactor, oldhip );
	else if( descOrgan == "comedemo")
		organMainMesh->loadFile( organFile, scaleFactor, comedemo );
	else {
		organMainMesh->loadFile( organFile, scaleFactor, COME_Point3D( 0, 0, 0 ) );
	}
	//organMainMesh->loadFile( organFile, scaleFactor, COME_Point3D( 0.0, 0.0, 0.0 ) );
	
	newOrgan->setSurface( organMainMesh );
	switch( type ){
		case CARTILAGE	: ((COME_MoleculesCartilage*)newOrgan)->initializeSkinning(); break;
		case LIGAMENT	: ((COME_MoleculesCartilage*)newOrgan)->initializeSkinning(); break;
		case BONE	: ((COME_MoleculesBone*)newOrgan)->initializeSkinning();break;
		case CLAMP	: ((COME_Clamp*)newOrgan)->initializeSkinning();break;
		//case LIGAMENT	: ((COME_MoleculesLigament*)newOrgan)->initializeSkinning() break;
	}
	printf( "skinning initialized \n" );
	newOrgan->updateSkin();
	printf( "skin updated \n" );
	
	return true;
}

//////////////////////////////////////////////////////////////////////
/// This method inspects the envelops of all patients within the scene and sets the 
/// parameters mins and maxs to the minimum and maximum corners of the rectangular
/// scene envelop.
//////////////////////////////////////////////////////////////////////// 
void
COME_Scenario::getEnvelop( COME_Point3D& mins, COME_Point3D& maxs ) const {

	patientList.front()->getEnvelop( mins, maxs );

	list<COME_Patient*>::const_iterator iter;
	for (iter = patientList.begin(); !(iter == patientList.end()); iter++){
		
		COME_Point3D minsP , maxsP;
		(*iter)->getEnvelop( minsP, maxsP );
		
		if( minsP.getX() < mins.getX() ){
			mins.setX( minsP.getX() );
		}
		if( minsP.getY() < mins.getY() ){
			mins.setY( minsP.getY() );
		}
		if( minsP.getZ() < mins.getZ() ){
			mins.setZ( minsP.getZ() );
		}

		if( maxsP.getX() > maxs.getX() ){
			maxs.setX( maxsP.getX() );
		}
		if( maxsP.getY() > maxs.getY() ){
			maxs.setY( maxsP.getY() );
		}
		if( maxsP.getZ() > maxs.getZ() ){
			maxs.setZ( maxsP.getZ() );
		}
	}

}
