/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

///////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  AUTHOR......: Anderson Maciel
//  DATE........: September/03/2002
//  DESCRIPTION.: COME_Xml class methods definition.
//
///////////////////////////////////////////////////////////////////


#include	<general/come.h>
#include	<general/comexml.h>
#include	<general/comesimulator.h>
#include	<general/comescenario.h>
#include	<bio/comemolecule.h>
#include	<bio/comemoleculescartilage.h>
#include	<bio/comemoleculesbone.h>
#include	<physics/cometimeforce.h>
#include	<kinematics/comepolyaxialjoint.h>
#include	<kinematics/comedof.h>
#include	<kinematics/comemodifier.h>
#include	<kinematics/comemovement.h>

#include	<xercesc/dom/DOM.hpp>
#include	<xercesc/util/XercesDefs.hpp>
#include	<xercesc/util/PlatformUtils.hpp>

#ifndef _WIN32
#include	<xercesc/util/Platforms/Linux/LinuxDefs.hpp>
else
#include	<xercesc/util/Platforms/Win32/Win32Defs.hpp>
#endif

#include	<xercesc/util/XMLString.hpp>
#include	<xercesc/sax/SAXException.hpp>
#include	<xercesc/sax/SAXParseException.hpp>
#include	<xercesc/parsers/XercesDOMParser.hpp>
#include	<xercesc/dom/DOMException.hpp>
#include	<xercesc/framework/LocalFileFormatTarget.hpp>

#include	<string.h>
#include	<stdlib.h>
#include	<stdio.h>
#include	<math.h>

using namespace XERCES_CPP_NAMESPACE;

// Header to avoid multiple definition of ostream
ostream& operator<<(ostream& target, const StrX& toDump);

/// Default constructor
COME_Xml::COME_Xml(){

	// Never free this vector because it will be used by molecules
	materials = new vector<COME_Material*>;

}

COME_Xml::COME_Xml( vector<COME_Material*>* existentMaterials ){

	materials = existentMaterials;

}

/// Default destructor
COME_Xml::~COME_Xml(){

}

vector<COME_Material*>*
COME_Xml::getAllMaterials(){

	return materials;
}

void
COME_Xml::loadMaterialsFile( string fileName, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene ){


    // Initialize the XML4C system
    try
    {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& toCatch)
    {
         printf( "Error during initialization! : %s\n", toCatch.getMessage() );
         return;
    }

    XercesDOMParser::ValSchemes    valScheme = XercesDOMParser::Val_Auto; //Val_Never Val_Always
    bool                     doNamespaces    = false;

    // Instantiate the DOM parser.
    XercesDOMParser parser;
    parser.setValidationScheme(valScheme);
    parser.setDoNamespaces(doNamespaces);

    // And create our error handler and install it
    DOMCountErrorHandler errorHandler;
    parser.setErrorHandler(&errorHandler);

    //
    //  Get the starting time and kick off the parse of the indicated
    //  file. Catch any exceptions that might propogate out of it.
    //
    unsigned long duration;
    try
    {
        const unsigned long startMillis = XMLPlatformUtils::getCurrentMillis();
        parser.parse( fileName.c_str() );
        const unsigned long endMillis = XMLPlatformUtils::getCurrentMillis();
        duration = endMillis - startMillis;
    }

    catch (const XMLException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << StrX(toCatch.getMessage()) << "\n" << endl;
        return;
    }
    catch (const DOMException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << toCatch.msg << "\n" << endl;
        XMLPlatformUtils::Terminate();
        return;
    }
    catch (...)
    {
       cerr << "\nUnexpected exception during parsing: '" << fileName << "'\n";
        XMLPlatformUtils::Terminate();
        return;
    }

    //
    //  Extract the DOM tree, get the list of all the elements and report the
    //  length as the count of elements.
    //
    if (errorHandler.getSawErrors())
    {
        printf( "\nErrors occured, no output available\n" );
    }
     else
    {
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        DOMDocument *doc = parser.getDocument();
	unsigned int elementCount = doc->getElementsByTagName(XMLString::transcode("*"))->getLength();
			
	/// Test main tag
	root = doc->getDocumentElement();
	if( strcmp( XMLString::transcode(root->getTagName()), "come" ) ){
		printf( "Invalid XML file.\n" );
		exit(0);
	}

	simulatorXml = simulator;

	DOMNodeList *materialNodes = root->getElementsByTagName( XMLString::transcode("material") );

	loadMaterials( materialNodes );
    }
    return;
}

COME_BioStructure*
COME_Xml::loadOrganFile( string fileName, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene ){


    // Initialize the XML4C system
    try
    {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& toCatch)
    {
         printf( "Error during initialization! : %s\n", toCatch.getMessage() );
         return NULL;
    }

    XercesDOMParser::ValSchemes    valScheme = XercesDOMParser::Val_Auto; //Val_Never Val_Always
    bool                     doNamespaces    = false;

    // Instantiate the DOM parser.
    XercesDOMParser parser;
    parser.setValidationScheme(valScheme);
    parser.setDoNamespaces(doNamespaces);

    // And create our error handler and install it
    DOMCountErrorHandler errorHandler;
    parser.setErrorHandler(&errorHandler);

    //
    //  Get the starting time and kick off the parse of the indicated
    //  file. Catch any exceptions that might propogate out of it.
    //
    unsigned long duration;
    try
    {
        const unsigned long startMillis = XMLPlatformUtils::getCurrentMillis();
        parser.parse( fileName.c_str() );
        const unsigned long endMillis = XMLPlatformUtils::getCurrentMillis();
        duration = endMillis - startMillis;
    }

    catch (const XMLException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << StrX(toCatch.getMessage()) << "\n" << endl;
        return NULL;
    }
    catch (const DOMException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << toCatch.msg << "\n" << endl;
        XMLPlatformUtils::Terminate();
        return NULL;
    }
    catch (...)
    {
       cerr << "\nUnexpected exception during parsing: '" << fileName << "'\n";
        XMLPlatformUtils::Terminate();
        return NULL;
    }

    //
    //  Extract the DOM tree, get the list of all the elements and report the
    //  length as the count of elements.
    //
    if (errorHandler.getSawErrors())
    {
        printf( "\nErrors occured, no output available\n" );
    }
     else
    {
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        DOMDocument *doc = parser.getDocument();
	unsigned int elementCount = doc->getElementsByTagName(XMLString::transcode("*"))->getLength();
			
	/// Test main tag
	root = doc->getDocumentElement();
	if( strcmp( XMLString::transcode(root->getTagName()), "come" ) ){
		printf( "Invalid XML file.\n" );
		exit(0);
	}

	simulatorXml = simulator;

	DOMNodeList *organs = root->getElementsByTagName( XMLString::transcode("molecule_organ") );

	return loadShapes( organs )[0];
    }
    return NULL;
}	

void
COME_Xml::loadSceneFile( string fileName, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene ){


    // Initialize the XML4C systemvoid
    try
    {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& toCatch)
    {
         printf( "Error during initialization! : %s\n", toCatch.getMessage() );
         return;
    }

    XercesDOMParser::ValSchemes    valScheme = XercesDOMParser::Val_Auto; //Val_Never Val_Always
    bool                     doNamespaces    = false;

    // Instantiate the DOM parser.
    XercesDOMParser parser;
    parser.setValidationScheme(valScheme);
    parser.setDoNamespaces(doNamespaces);

    // And create our error handler and install it
    DOMCountErrorHandler errorHandler;
    parser.setErrorHandler(&errorHandler);

    //
    //  Get the starting time and kick off the parse of the indicated
    //  file. Catch any exceptions that might propogate out of it.
    //
    unsigned long duration;
    try
    {
        const unsigned long startMillis = XMLPlatformUtils::getCurrentMillis();
        parser.parse( fileName.c_str() );
        const unsigned long endMillis = XMLPlatformUtils::getCurrentMillis();
        duration = endMillis - startMillis;
    }

    catch (const XMLException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << StrX(toCatch.getMessage()) << "\n" << endl;
        return;
    }
    catch (const DOMException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << toCatch.msg << "\n" << endl;
        XMLPlatformUtils::Terminate();
        return;
    }
    catch (...)
    {
       cerr << "\nUnexpected exception during parsing: '" << fileName << "'\n";
        XMLPlatformUtils::Terminate();
        return;
    }

    //
    //  Extract the DOM tree, get the list of all the elements and report the
    //  length as the count of elements.
    //
    if (errorHandler.getSawErrors())
    {
        printf( "\nErrors occured, no output available\n" );
    }
     else
    {
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        DOMDocument *doc = parser.getDocument();
	unsigned int elementCount = doc->getElementsByTagName(XMLString::transcode("*"))->getLength();
			
	/// Test main tag
	root = doc->getDocumentElement();
	if( strcmp( XMLString::transcode(root->getTagName()), "come" ) ){
		printf( "Invalid XML file.\n" );
		exit(0);
	}

	simulatorXml = simulator;

	DOMNodeList *scene = root->getElementsByTagName( XMLString::transcode("scene") );

	for( int s = 0; s < scene->getLength(); s++ ){
	
		if( !XMLString::compareString( scene->item(s)->getNodeName(), XMLString::transcode("scene" ) ) ){
			DOMNamedNodeMap *attr = scene->item(s)->getAttributes();
			double x = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("gX") )->getNodeValue() ) );
			double y = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("gY") )->getNodeValue() ) );
			double z = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("gZ") )->getNodeValue() ) );
			parentScene->setGravity( COME_Vector3D( x, y, z ) );
		}
	}

	DOMNodeList *simulation = root->getElementsByTagName( XMLString::transcode("simulation") );

	for( int sim = 0; sim < simulation->getLength(); sim++ ){
	
		if( !XMLString::compareString( simulation->item(sim)->getNodeName(), XMLString::transcode("simulation" ) ) ){
			DOMNamedNodeMap *attr = simulation->item(sim)->getAttributes();
			simulator->setFPS( atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("fps") )->getNodeValue() ) ) );
			simulator->setDuration( atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("duration") )->getNodeValue() ) ) );
		}
	}

		
	/// Load patients
	////////// only one scene is allowed at this moment ////
	DOMNodeList *patients_forces = scene->item(0)->getChildNodes();
			
	for( int i = 0; i < patients_forces->getLength(); i++ ){
		
		// Load patient attributes
		if( !XMLString::compareString( patients_forces->item(i)->getNodeName(), XMLString::transcode("patient" ) ) ){
			DOMNamedNodeMap *attr = patients_forces->item(i)->getAttributes();
			string name = XMLString::transcode( attr->getNamedItem( XMLString::transcode("name") )->getNodeValue());
			bool gender;
			if( !XMLString::compareString(attr->getNamedItem( XMLString::transcode("gender") )->getNodeValue(), XMLString::transcode("M") ) ) {
				gender = true;
			} else {
				gender = false;
			}
			float weight = (float)atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("weight") )->getNodeValue() ) );
			float height = (float)atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("height") )->getNodeValue() ) );
			short age = (short)atoi( XMLString::transcode( attr->getNamedItem( XMLString::transcode("age") )->getNodeValue() ) );

			// Load patient materials
			DOMNodeList *materialNodes = patients_forces->item(i)->getChildNodes();
							
			for( int io = 0; io < materialNodes->getLength(); io++ ){

				if( !XMLString::compareString( materialNodes->item(io)->getNodeName(), XMLString::transcode("materials" ) ) ){

					// Load patient materials
					DOMNodeList *materials_list = materialNodes->item(io)->getChildNodes();
					loadMaterials( materials_list );
				}
			}

			/// List to be filled while loading organs into joints bellow
			list<COME_BioStructure*> organList;

			/// Find root joint and load joints tree
			DOMNodeList *joints = root->getElementsByTagName(XMLString::transcode("joint"));
			COME_Joint *rootJoint;

			for( int ij = 0; ij < joints->getLength(); ij++ ){
				DOMNamedNodeMap *attr = joints->item(ij)->getAttributes();
				if( !XMLString::compareString( attr->getNamedItem( XMLString::transcode("description" ) )->getNodeValue(), XMLString::transcode("root") ) ){
					auxVec = new COME_Dof*[joints->getLength()*6];
					iIndDofAux = 0;
					rootJoint = loadJoint( joints->item(ij), NULL, organList );
					loadModifiers( joints );
					delete [] auxVec;
					break;
				}
			}

			// Build patient
			COME_Patient* patientN = new COME_Patient( name, age, gender, weight, height, rootJoint, organList, parentScene );

			// Add patient to the list
			patientL.clear();
			patientL.push_back( patientN );
						
		} 
			// FUTURE IMPLEMENTATION
			// Load global forces (gravity, etc.)		
			//DOMNodeList *gForces = scene->getElementsByTagName("global_forces");
								
	}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Print out the stats that we collected and time taken.
        cout << fileName << ": " << duration << " ms ("
             << elementCount << " elems)." << endl;
    }
    // And call the termination method
    //XMLPlatformUtils::Terminate();

    return;
}

COME_Joint*
COME_Xml::loadJoint( DOMNode *node, COME_Joint *parent, list<COME_BioStructure*> &patientOrganList ){

	/// Create a Joint and its dependencies
	DOMNamedNodeMap *attrNode = node->getAttributes();
	DOMNodeList *dofs_shapes = node->getChildNodes();
	COME_Dof *dofAux = loadDofs( dofs_shapes );
	COME_Joint *newJoint = NULL;
	
	if( !XMLString::compareString( attrNode->getNamedItem( XMLString::transcode("type") )->getNodeValue() , XMLString::transcode("plane") ) ){
		//newJoint = new COME_PlaneJoint( parent, &(dofAux[0]), &(dofAux[1]), &(dofAux[2]), &(dofAux[3]), &(dofAux[4]), &(dofAux[5]) );
	} else if( !XMLString::compareString( attrNode->getNamedItem( XMLString::transcode("type") )->getNodeValue() , XMLString::transcode("uniaxial") ) ){
		//newJoint = new COME_UniaxialJoint( parent, &(dofAux[0]) );
	} else if( !XMLString::compareString( attrNode->getNamedItem( XMLString::transcode("type") )->getNodeValue() , XMLString::transcode("biaxial") ) ){
		//newJoint = new COME_BiaxialJoint( parent, &(dofAux[0]), &(dofAux[1]) );
	} else if( !XMLString::compareString( attrNode->getNamedItem( XMLString::transcode("type") )->getNodeValue() , XMLString::transcode("triaxial") ) ){
		newJoint = new COME_PolyaxialJoint( parent, &(dofAux[0]), &(dofAux[1]), &(dofAux[2]) );
	} else if( !XMLString::compareString( attrNode->getNamedItem( XMLString::transcode("type") )->getNodeValue() , XMLString::transcode("noaxial") ) ){
		newJoint = new COME_PolyaxialJoint( parent );
	}
	newJoint->setDescription( XMLString::transcode( attrNode->getNamedItem( XMLString::transcode( "description") )->getNodeValue() ) );

	vector<COME_BioStructure*> shapeAux = loadShapes( dofs_shapes );
	for( int i = 0; i < shapeAux.size(); i++ ){
		newJoint->vpAddShape( shapeAux[ i ] );
		patientOrganList.push_back( shapeAux[ i ] );
	}
	
	/// Call loadJoint recursively to create the tree
	DOMNodeList *joints = root->getElementsByTagName( XMLString::transcode("joint") );
	for( int j = 0; j < joints->getLength(); j++ ){
		DOMNamedNodeMap *attrAux = joints->item(j)->getAttributes();
		if( !XMLString::compareString( attrAux->getNamedItem( XMLString::transcode("parent") )->getNodeValue(), attrNode->getNamedItem( XMLString::transcode("description") )->getNodeValue() ) ){
			COME_Joint* auxJoint = loadJoint( joints->item(j), newJoint, patientOrganList );
			newJoint->vpAddChild( auxJoint );
		}
	}
	return newJoint;
}

COME_Dof*
COME_Xml::loadDofs( DOMNodeList *paramList ){

	/// Scan dofs of the list
	COME_Dof* listDof = new COME_Dof[6];
	int idof = 0;
	for( int i = 0; i < paramList->getLength(); i++ ){
		if( !XMLString::compareString( paramList->item(i)->getNodeName(), XMLString::transcode( "dof") ) ){
			/// Get dof properties
			COME_Vector3D axis;
			COME_Point3D position;
			COME_Curve* evoluta = new COME_Bezier();
			float min = 0.0, max = 0.0 , curr = 0.0, rest = 0.0;
			DOMNodeList *dof = paramList->item(i)->getChildNodes();
			for( int j = 0; j < dof->getLength(); j++ ){
				if( !XMLString::compareString( dof->item(j)->getNodeName(), XMLString::transcode( "position") ) ){
					DOMNamedNodeMap *attrPos = dof->item(j)->getAttributes();
					position = COME_Point3D( (float)atof( XMLString::transcode( attrPos->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) ),
											 (float)atof( XMLString::transcode( attrPos->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) ),
											 (float)atof( XMLString::transcode( attrPos->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) )
											);
				} else if( !XMLString::compareString( dof->item(j)->getNodeName(), XMLString::transcode( "axis") )  ){
					DOMNamedNodeMap *attrAxis = dof->item(j)->getAttributes();
					axis = COME_Vector3D( (float)atof( XMLString::transcode( attrAxis->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) ),
										  (float)atof( XMLString::transcode( attrAxis->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) ),
										  (float)atof( XMLString::transcode( attrAxis->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) )
										);
				} else if( !XMLString::compareString( dof->item(j)->getNodeName(), XMLString::transcode( "range") )  ){
					DOMNamedNodeMap *attrRange = dof->item(j)->getAttributes();
					min = (float)atof( XMLString::transcode( attrRange->getNamedItem( XMLString::transcode("min") )->getNodeValue() ) );
					max = (float)atof( XMLString::transcode( attrRange->getNamedItem( XMLString::transcode("max") )->getNodeValue() ) );
				} else if( !XMLString::compareString( dof->item(j)->getNodeName(), XMLString::transcode( "evoluta") )  ){
					DOMNodeList *pts = dof->item(j)->getChildNodes();
					COME_Vertex3D*	ctrlPts = new COME_Vertex3D[4];
					int cp = 0;
					for( int k = 0; k < pts->getLength(); k++ ){
						if( !XMLString::compareString( pts->item(k)->getNodeName(), XMLString::transcode("ctrl_point") ) ){
  						DOMNamedNodeMap *attrPt = pts->item(k)->getAttributes();
  						ctrlPts[cp++].setXYZ( (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) ),
  											  (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) ),
  											  (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) )
  											);
  						}
					}
					evoluta = new COME_Bezier( ctrlPts );
				}
			}
			/// Create DOF
			listDof[idof] = COME_Dof( axis, position, evoluta, min, max, curr, rest );
			DOMNamedNodeMap *attrAux = paramList->item(i)->getAttributes();
			listDof[idof].setDescription( XMLString::transcode( attrAux->getNamedItem( XMLString::transcode("description") )->getNodeValue() ) );
			listDof[idof].vpRest();
			
			//put dof in the temporary list for modifiers
			auxVec[iIndDofAux++] = &(listDof[idof]);
			
			idof++;
		}
	}
	return listDof;
}

// Load patient molecule organs
vector<COME_BioStructure*>
COME_Xml::loadShapes( DOMNodeList *paramList ){

	vector<COME_BioStructure*> organList;

	for( int i = 0; i < paramList->getLength(); i++ ){

		if( !XMLString::compareString( paramList->item(i)->getNodeName(), XMLString::transcode( "molecule_organ") ) ){

			COME_BioStructure *tempOrgan = loadMoleculeOrgan( paramList->item(i) );
			DOMNamedNodeMap *attr = paramList->item(i)->getAttributes();
			string fileMesh = XMLString::transcode( attr->getNamedItem( XMLString::transcode("mesh_file") )->getNodeValue() );
			if( !( fileMesh.find( COME::baseFolder, 0 ) == 0 ) )
				fileMesh = COME::baseFolder + "/" + fileMesh;
			string fileTexture = COME::baseFolder + "/" + XMLString::transcode( attr->getNamedItem( XMLString::transcode("texture_file") )->getNodeValue() );
			string descOrgan = XMLString::transcode( attr->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
			string fileDeformation = XMLString::transcode( attr->getNamedItem( XMLString::transcode("deformation_file") )->getNodeValue() );
			
			if( fileDeformation.size() == 0 )
				fileDeformation = fileMesh;

			if( !( fileDeformation.find( COME::baseFolder, 0 ) == 0 ) )
				fileDeformation = COME::baseFolder + "/" + fileDeformation;

			if( fileDeformation.find( ".vtk", 0 ) != string::npos ){
				string tempStr( fileDeformation, 0, fileDeformation.size()-4 );
				fileDeformation = tempStr + "points.dat";
			} else if( fileDeformation.find( ".iv", 0 ) != string::npos ){
				string tempStr( fileDeformation, 0, fileDeformation.size()-3 );
				fileDeformation = tempStr + "points.dat";
			}
			
			double fileScaleFactor = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("mesh_scale_factor") )->getNodeValue() ) );
			double fileTansX = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("mesh_translate_X") )->getNodeValue() ) );
			double fileTansY = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("mesh_translate_Y") )->getNodeValue() ) );
			double fileTansZ = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("mesh_translate_Z") )->getNodeValue() ) );
			double fileCollisionRadius = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("collision_radius") )->getNodeValue() ) );
	
			tempOrgan->setDescription(descOrgan);
			tempOrgan->setCollisionRadius(fileCollisionRadius);
			
			DOMNodeList *internals = paramList->item(i)->getChildNodes();
			for( int ip = 0; ip < internals->getLength(); ip++ ){
				if( !XMLString::compareString( internals->item(ip)->getNodeName(), XMLString::transcode( "internals") ) ){
					
					DOMNodeList *internal = internals->item(ip)->getChildNodes();
					for( int iq = 0; iq < internal->getLength(); iq++ ){
						if( !XMLString::compareString( internal->item(iq)->getNodeName(), XMLString::transcode( "internal") ) ){
							DOMNamedNodeMap *attrInt = internal->item(iq)->getAttributes();
							string intDesc = XMLString::transcode( attrInt->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
							tempOrgan->internalsDesc.push_back( intDesc );
						}
					}
				}
				
			}
			
			// Load mesh if it exists
			printf( "fileMesh: %s base: %s \n", fileMesh.c_str(), ( COME::baseFolder + "/" ).c_str() );
			if( fileMesh != ( COME::baseFolder + "/" )  ){
				COME_Mesh *newSurface = new COME_Mesh();
				//tempOrgan->getSurface()->loadFile( fileMesh, fileScaleFactor, COME_Point3D( -97.9, 4.5, -25.7 ) );
				
				// APPLY ALL TRANSLATIONS
				COME_Point3D oldhip( -97.9, 4.5, -25.7 );
				//COME_Point3D labrum( -119.8, 8.535, -34.84 );
				//COME_Point3D pubofemoral( -118.9, 9, -34.46 );
				COME_Point3D labrum( 21.9, -4.035, 9.14 );
				COME_Point3D pubofemoral( 21.0, -4.5, 8.76 );
				COME_Point3D comedemo( 0.0, 0.0, -2.5 );
				
				if( descOrgan == "labrum")
					newSurface->loadFile( fileMesh, fileScaleFactor, labrum );
				else if( ( descOrgan == "pubofemoral") || ( descOrgan == "ischiofemoral") )
					newSurface->loadFile( fileMesh, fileScaleFactor, pubofemoral );
				else if( descOrgan == "pelviscart")
					newSurface->loadFile( fileMesh, fileScaleFactor, oldhip );
				else if( descOrgan == "pelvisbone")
					newSurface->loadFile( fileMesh, fileScaleFactor, oldhip );
				else if( descOrgan == "femurcart")
					newSurface->loadFile( fileMesh, fileScaleFactor, oldhip );
				else if( descOrgan == "comedemo")
					newSurface->loadFile( fileMesh, fileScaleFactor, comedemo );
				else if( descOrgan == "femurbone"){
					newSurface->loadFile( fileMesh, fileScaleFactor, oldhip ); //////////////
					//newSurface->translate( 0.117, 0, 0.352 ); ///OPERATION
				} else {
					newSurface->loadFile( fileMesh, fileScaleFactor, COME_Point3D( -fileTansX, -fileTansY, -fileTansZ ) );
				}
				//if( descOrgan == "ligament"){
				//	newSurface->rotate( -M_PI/2.0, 0.0, 0.0 );
				//}
				
				if( fileTexture != "" ){
					newSurface->loadImage( fileTexture );
				}
				tempOrgan->setSurface( newSurface );
				if( !tempOrgan->isFixed() )
					((COME_MoleculesCartilage*)tempOrgan)->initializeSkinning();
				else ((COME_MoleculesBone*)tempOrgan)->initializeSkinning();
				printf( "skinning initialized \n" );
				tempOrgan->updateSkin();
				printf( "skin updated \n" );
			} else {
				if( !tempOrgan->isFixed() ){
					((COME_MoleculesCartilage*)tempOrgan)->initializeSkinning();
					printf( "skinning initialized \n" );
					((COME_MoleculesCartilage*)tempOrgan)->updateSkin();
					printf( "skin updated \n" );
				} else {
					((COME_MoleculesBone*)tempOrgan)->initializeSkinning();
					printf( "skinning initialized \n" );
					((COME_MoleculesBone*)tempOrgan)->updateSkin();
					printf( "skin updated \n" );
				}
			}
			// Load precalculated deformation if it exists.
			// Or set and initialize output file for offline simulation.
			// Or do nothing for normal simulation.
			if( !tempOrgan->isFixed()  ){
				((COME_MoleculesCartilage*)tempOrgan)->setDeformationFile( fileDeformation );
			}

			if( tempOrgan->getGlobalHooke() == 0.0 ){
				tempOrgan->getTissue()->recalculateAllSpringConstants( simulatorXml->getHowRecalculate(), 0.0 );
				printf( "Recalculating springs by: %d \n", simulatorXml->getHowRecalculate() );
			}
			organList.push_back( tempOrgan );
		}
	}

	return organList;
}


void
COME_Xml::loadModifiers( DOMNodeList *joints ){

	for( int i = 0; i < joints->getLength(); i++ ){
		DOMNodeList *xmlDofs = joints->item(i)->getChildNodes();
		for( int j = 0; j < xmlDofs->getLength(); j++ ){
			if( !XMLString::compareString( xmlDofs->item(j)->getNodeName(), XMLString::transcode( "dof") ) ){
				DOMNamedNodeMap *attrDof = xmlDofs->item(j)->getAttributes();
 				DOMNodeList *xmlModif = xmlDofs->item(j)->getChildNodes();
 				for( int k = 0; k < xmlModif->getLength(); k++ ){
 					if( !XMLString::compareString( xmlModif->item(k)->getNodeName(), XMLString::transcode("modifier") ) ){
 						DOMNodeList *xmlRelation = xmlModif->item(k)->getChildNodes();
 						int numRelat = 0;
 						for( int x = 0; x <	xmlRelation->getLength(); x++ ){
   						if( !XMLString::compareString( xmlRelation->item(x)->getNodeName(), XMLString::transcode("relation") ) ) numRelat++;
   					}
   					COME_Dof** modifierDof = new COME_Dof*[numRelat];
 					COME_Dof* modifiedDof = NULL;
 					COME_Curve *min = new COME_Curve[numRelat];
 					COME_Curve *max = new COME_Curve[numRelat];
 						
 					int ll = 0;
   					for( int l = 0; l <	xmlRelation->getLength(); l++ ){
   						while( XMLString::compareString( xmlRelation->item(l)->getNodeName(), XMLString::transcode("relation") ) ){
   							l++;
   							if( l >= xmlRelation->getLength() ) break;
   						}
   						if( l >= xmlRelation->getLength() ) break;
   						DOMNamedNodeMap *attrRelation = xmlRelation->item(l)->getAttributes();
   						int mm = 0;
    					for( int m = 0; m <	iIndDofAux; m++ ){
    						if( !XMLString::compareString( attrRelation->getNamedItem( XMLString::transcode("dof") )->getNodeValue(), XMLString::transcode( auxVec[m]->getDescription().c_str() ) ) ){
     							modifierDof[mm++] = auxVec[m];
    						}
    						if( !XMLString::compareString( attrDof->getNamedItem( XMLString::transcode("description") )->getNodeValue(), XMLString::transcode( auxVec[m]->getDescription().c_str() ) ) ){
     							modifiedDof = auxVec[m];
    						}
    					}
    					
    					// get the points of the relation curves
    					DOMNodeList *xmlPoints = xmlRelation->item(l)->getChildNodes();
    					COME_Vertex3D*	Ptm = new COME_Vertex3D[4];
    					COME_Vertex3D*	PtM = new COME_Vertex3D[4];
    					int nn = 0;
      				for( int n = 0; n < xmlPoints->getLength(); n++ ){
      					if( !XMLString::compareString( xmlPoints->item(n)->getNodeName(), XMLString::transcode( "ctrl_point") ) ){
      						DOMNamedNodeMap *attrPt = xmlPoints->item(n)->getAttributes();
        					if( nn < 4 ){
                		Ptm[nn].setXYZ( (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) ),
                						(float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) ),
                						(float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) ) );
                	}
                	if( nn >= 4 ){
                		PtM[nn-4].setXYZ( (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) ),
                						  (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) ),
                						  (float)atof( XMLString::transcode( attrPt->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) ) );
                	}
                	nn++;
                }
              }

              min[ll].setControlPoints(Ptm);
              max[ll].setControlPoints(PtM);
    					ll++;
    				}
    				// Create and set the modifier
   					COME_Modifier *modif = new COME_Modifier( modifierDof, min, max, numRelat );
   					modifiedDof->setRangeModifier( modif );
   				}
   			}
			}
		}
	}
}


void
COME_Xml::loadMaterials( DOMNodeList *materialList ){

	for( int im = 0; im < materialList->getLength(); im++ ){

		if( !XMLString::compareString( materialList->item(im)->getNodeName(), XMLString::transcode("material" ) ) ){
			DOMNamedNodeMap *attr = materialList->item(im)->getAttributes();
			///shit
			string materialDesc;
			materialDesc = XMLString::transcode( attr->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
			
			double cR = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("R") )->getNodeValue() ) );
			double cG = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("G") )->getNodeValue() ) );
			double cB = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("B") )->getNodeValue() ) );
			double damping_const = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("damping_const") )->getNodeValue() ) );
			double density = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("density") )->getNodeValue() ) );
			double youngs_modulus = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("youngs_modulus") )->getNodeValue() ) );
			double medium_density = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("medium_density") )->getNodeValue() ) );
			double liquidFraction = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("liquidFraction") )->getNodeValue() ) );
			double permeability = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("permeability") )->getNodeValue() ) );
			double aI = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("anisoI") )->getNodeValue() ) );
			double aJ = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("anisoJ") )->getNodeValue() ) );
			double aK = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("anisoK") )->getNodeValue() ) );
			double mx_stress = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("max_stress") )->getNodeValue() ) );
			COME_Material *newMaterial = new COME_Material( COME_Vector3D( cR, cG, cB ), density, damping_const, youngs_modulus, medium_density, liquidFraction, permeability, COME_Vector3D( aI, aJ, aK ) );
			newMaterial->setDescription( materialDesc );
			newMaterial->setMaxStress( mx_stress );
			materials->push_back( newMaterial );
		}
	}
}



COME_BioStructure *
COME_Xml::loadMoleculeOrgan( DOMNode *node ){
	
	
	COME_BioStructure *organ = NULL;
	DOMNodeList *moleculeList = node->getChildNodes();

	int type;
	DOMNamedNodeMap *organAtt = node->getAttributes();
	if( !XMLString::compareString( organAtt->getNamedItem( XMLString::transcode("type") )->getNodeValue(), XMLString::transcode("cartilage" ) ) ){
	 
		organ = new COME_MoleculesCartilage();
		type = CARTILAGE;
		organ->setFixed( false );
	} else if( !XMLString::compareString( organAtt->getNamedItem( XMLString::transcode("type") )->getNodeValue(), XMLString::transcode("ligament" ) ) ){
	 
		organ = new COME_MoleculesCartilage();
		type = LIGAMENT;
		organ->setFixed( false );
	} else {

		organ = new COME_MoleculesBone(); //COME_MoleculesCartilage();
		type = BONE;
		organ->setFixed( true );
	}
	organ->setType( type );

	string description = XMLString::transcode( organAtt->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
	organ->setDescription( description );
	double globalHooke = atof( XMLString::transcode( organAtt->getNamedItem( XMLString::transcode("global_hooke") )->getNodeValue() ) );
	organ->setGlobalHooke( globalHooke );

	for( int im = 0; im < moleculeList->getLength(); im++ ){

		if( !XMLString::compareString( moleculeList->item(im)->getNodeName(), XMLString::transcode("molecule" ) ) ){
			DOMNamedNodeMap *attr = moleculeList->item(im)->getAttributes();
			string material = XMLString::transcode( attr->getNamedItem( XMLString::transcode("material") )->getNodeValue() );
			double x = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) );
			double y = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) );
			double z = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) );
			double friction_const = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("friction_const") )->getNodeValue() ) );
			double radius = atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("radius") )->getNodeValue() ) );
			string fixed = XMLString::transcode( attr->getNamedItem( XMLString::transcode("fixed") )->getNodeValue() );
			string molecName = XMLString::transcode( attr->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
						
			//COME_Point3D position = COME_Point3D( x, y, z ) + COME_Point3D( 0.117, 0, 0.352 ); ///OPERATION
			COME_Point3D position = COME_Point3D( x, y, z );
			
			if( materials->size() == 0 ){
				materials->push_back( new COME_Material() );
				(*materials)[0]->setDescription( "empty" );
			}
			
			/// this block was strangely commented. Reverify next time Mechatester is compiled. Molecule constructor was outside with (*materials)[0]
			COME_Molecule *molecule = new COME_Molecule();
			int i;
			for( i = 0; i < materials->size(); i ++ ){
				if( (*materials)[i]->getDescription() == material ){
					delete molecule;
					molecule = new COME_Molecule( position, radius, friction_const, (*materials)[i], COME_Vector3D(), organ );
					break;
				}
			}
						
			molecule->setDescription( molecName );
			if( fixed == "true" ){

				molecule->setFixed(true);
			}
			
			/// Load external forces here
			DOMNodeList *forceList = moleculeList->item(im)->getChildNodes();
			for( int iF = 0; iF < forceList->getLength(); iF++ ){
			
				if( !XMLString::compareString( forceList->item(iF)->getNodeName(), XMLString::transcode("force" ) ) ){
					DOMNamedNodeMap *attrF = forceList->item(iF)->getAttributes();
					double xf = atof( XMLString::transcode( attrF->getNamedItem( XMLString::transcode("x") )->getNodeValue() ) );
					double yf = atof( XMLString::transcode( attrF->getNamedItem( XMLString::transcode("y") )->getNodeValue() ) );
					double zf = atof( XMLString::transcode( attrF->getNamedItem( XMLString::transcode("z") )->getNodeValue() ) );
					double t = atof( XMLString::transcode( attrF->getNamedItem( XMLString::transcode("time") )->getNodeValue() ) );
					COME_TimeForce *force = new COME_TimeForce( xf, yf, zf, t );
					molecule->addExternalForce( force );
				}
			}
			if( ( type == CARTILAGE ) || ( type == LIGAMENT ) ){
				((COME_MoleculesCartilage*)organ)->getTissue()->addMolecule( molecule );
				printf( "ADDED MOLECULE \n" );
			} else if( type == BONE ){
				((COME_MoleculesBone*)organ)->getTissue()->addMolecule( molecule );
				//((COME_MoleculesCartilage*)organ)->getTissue()->addMolecule( molecule );
			}

			//organ->getTissue()->setSurfaceMolecules();

			/// Numerical method test
			if( material == "chain"){
				organ->getTissue()->chain.molecules.push_back( molecule );
				organ->getTissue()->chain.updateLengths( 0.0 );
				organ->getTissue()->chain.setNominal();
			}
			///
		}
	}

	return organ;

}

//------------------------------------------------------------------------------------------------------------------------

int
COME_Xml::saveSceneFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator) {

	printf("Just came in \n");
	// Initialize the XML4C system
	try
	{
		XMLPlatformUtils::Initialize();
	}
	
	catch (const XMLException& toCatch)
	{
		printf( "Error during initialization! : %s\n", toCatch.getMessage() );
		return 0;
	}

	XMLCh tempStr[100];
        XMLString::transcode("LS", tempStr, 99, XMLPlatformUtils::fgMemoryManager);
        DOMImplementation *impl = DOMImplementationRegistry::getDOMImplementation(tempStr);
        DOMWriter* theSerializer = ((DOMImplementationLS*)impl)->createDOMWriter();

	// Set some features on this serializer
        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true))
            theSerializer->setFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true);

        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true))
             theSerializer->setFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true);
	
	printf("after exception, before funny things \n");

	XMLString::transcode("\n", tempStr, 99);
	theSerializer->setNewLine( tempStr );

        // Create document
	XMLString::transcode("come", tempStr, 99);
	DOMDocumentType *comeDocType = impl->createDocumentType( tempStr, 0, XMLString::transcode("comescene.dtd") );
        DOMDocument*   doc = impl->createDocument(0, tempStr, comeDocType );
        DOMElement*   come = doc->getDocumentElement();

	char chrPtTemp[100];

	printf("Start adding tags \n");
	
	// Add scene tag to document
        XMLString::transcode("scene", tempStr, 99);
        DOMElement*   scene = doc->createElement(tempStr);
        come->appendChild(scene);
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gX"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gY"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gZ"), XMLString::transcode(chrPtTemp) );

	// Add Simulation tag and attributes to document
        XMLString::transcode("simulation", tempStr, 99);
        DOMElement*   simulation = doc->createElement(tempStr);
        scene->appendChild(simulation);
		
	sprintf( chrPtTemp, "%10.6f", simulator->getFPS() ) ;
	simulation->setAttribute( XMLString::transcode("fps"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getDuration() ) ;
	simulation->setAttribute( XMLString::transcode("duration"), XMLString::transcode(chrPtTemp) );
	simulation->setAttribute( XMLString::transcode("cycle"), XMLString::transcode("off") );
	simulation->setAttribute( XMLString::transcode("type"), XMLString::transcode("offline") );

	printf("Start adding patient \n" );
	// Add patients to document
	//   the front of the patient's list is used because there is only one patient in the scene
	XMLString::transcode("patient", tempStr, 99);
        DOMElement*   patient = doc->createElement(tempStr);
        scene->appendChild(patient);
		
	patient->setAttribute( XMLString::transcode("name"), XMLString::transcode(patientL.front()->getName().c_str() ) );
	if( patientL.front()->getGender() ){
		patient->setAttribute( XMLString::transcode("gender"), XMLString::transcode("M") );
	} else {
		patient->setAttribute( XMLString::transcode("gender"), XMLString::transcode("F") );
	}
	sprintf( chrPtTemp, "%10.6f", patientL.front()->getWeight() );
	patient->setAttribute( XMLString::transcode("weight"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", patientL.front()->getHeight() );
	patient->setAttribute( XMLString::transcode("height"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%d", patientL.front()->getAge() );
	patient->setAttribute( XMLString::transcode("age"), XMLString::transcode(chrPtTemp) );

	printf("before savematerials \n" );
	saveMaterials( doc, patient, patientL.front() );
	
	// Add a standard joint just to keep format
	XMLString::transcode("joint", tempStr, 99);
        DOMElement* joint = doc->createElement(tempStr);
	patient->appendChild( joint );
	joint->setAttribute( XMLString::transcode("description"), XMLString::transcode("root" ) );
	joint->setAttribute( XMLString::transcode("parent"), XMLString::transcode("null" ) );
	joint->setAttribute( XMLString::transcode("type"), XMLString::transcode("noaxial" ) );
	
	// Add MoleculeOrgans of this patient
	printf("before savemoleculeorgan \n" );
	saveMoleculeOrgan( doc, joint, patientL.front() );
	printf("after savemoleculeorgan \n" );

        // StdOutFormatTarget prints the resultant XML stream
        // to stdout once it receives any thing from the serializer.
        XMLFormatTarget *myFormTarget = new LocalFileFormatTarget( file_name.c_str() );

        try {
            // do the serialization through DOMWriter::writeNode();
            theSerializer->writeNode(myFormTarget, *doc);
        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            printf( "Exception message is: \n %s\n ", message );
            delete [] message;
            return 0;
        }
        catch (const DOMException& toCatch) {
            char* message = XMLString::transcode(toCatch.msg);
            printf( "Exception message is: \n %s\n ", message );
            delete [] message;
            return 0;
        }
        catch (...) {
            cout << "Unexpected Exception \n" ;
            return 0;
        }

        theSerializer->release();
        delete myFormTarget;
        return 1;
}

    

void
COME_Xml::saveMaterials( DOMDocument *doc, DOMElement *patientNode, COME_Patient patient ){

	DOMElement *materialList = doc->createElement( XMLString::transcode("materials" ) );
		
	vector<string> includedMaterials;
	list<COME_MoleculesCartilage*>::const_iterator iter;
	for( iter = ((list<COME_MoleculesCartilage *>*)patient.getPtOrganList())->begin(); iter != ((list<COME_MoleculesCartilage *>*)patient.getPtOrganList())->end(); iter++ ){
	
		list<COME_Molecule*>::const_iterator iterM;
		for( iterM = (*iter)->getTissue()->getShape()->begin(); iterM != (*iter)->getTissue()->getShape()->end(); iterM++ ){
		
			if( ( (*iterM)->getMaterial()->getDescription() != "" ) && ( find( includedMaterials.begin(), includedMaterials.end(), (*iterM)->getMaterial()->getDescription() ) == includedMaterials.end() ) ){
				includedMaterials.push_back( (*iterM)->getMaterial()->getDescription() );

				DOMElement *materialNode = doc->createElement( XMLString::transcode("material" ) );

				char chrPtTemp[100];
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getMaterial()->getDensity() );
				materialNode->setAttribute( XMLString::transcode("density"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getMaterial()->getDamping() );
				materialNode->setAttribute( XMLString::transcode("damping_const"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getMaterial()->getYoungsModulus() );
				materialNode->setAttribute( XMLString::transcode("youngs_modulus"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getMaterial()->getMediumDensity() );
				materialNode->setAttribute( XMLString::transcode("medium_density"), XMLString::transcode(chrPtTemp) );
				materialNode->setAttribute( XMLString::transcode("description"), XMLString::transcode((*iterM)->getMaterial()->getDescription().c_str() ) );
				COME_Vector3D color = (*iterM)->getMaterial()->getColor();
				sprintf( chrPtTemp, "%10.6f", color.getX() );
				materialNode->setAttribute( XMLString::transcode("R"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", color.getY() );
				materialNode->setAttribute( XMLString::transcode("G"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", color.getZ() );
				materialNode->setAttribute( XMLString::transcode("B"), XMLString::transcode(chrPtTemp) );
				COME_Vector3D anisotropy = (*iterM)->getMaterial()->getAnisotropyVector();
				sprintf( chrPtTemp, "%10.6f", anisotropy.getX() );
				materialNode->setAttribute( XMLString::transcode("anisoI"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", anisotropy.getY() );
				materialNode->setAttribute( XMLString::transcode("anisoJ"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", anisotropy.getZ() );
				materialNode->setAttribute( XMLString::transcode("anisoK"), XMLString::transcode(chrPtTemp) );

				materialList->appendChild( materialNode );
			}
		}
		// if saving an organ without any material specified
		if( includedMaterials.size() == 0 ){
		
			DOMElement *materialNode = doc->createElement( XMLString::transcode("material" ) );

			materialNode->setAttribute( XMLString::transcode("density"), XMLString::transcode("1.0") );
			materialNode->setAttribute( XMLString::transcode("damping_const"), XMLString::transcode("0.2") );
			materialNode->setAttribute( XMLString::transcode("youngs_modulus"), XMLString::transcode("1000.0") );
			materialNode->setAttribute( XMLString::transcode("medium_density"), XMLString::transcode("1.0") );
			materialNode->setAttribute( XMLString::transcode("description"), XMLString::transcode("void") );
			materialNode->setAttribute( XMLString::transcode("R"), XMLString::transcode("1.0") );
			materialNode->setAttribute( XMLString::transcode("G"), XMLString::transcode("1.0") );
			materialNode->setAttribute( XMLString::transcode("B"), XMLString::transcode("1.0") );
			
			materialList->appendChild( materialNode );
		}

	}

	patientNode->appendChild( materialList );
}


void
COME_Xml::saveMoleculeOrgan( DOMDocument *doc, DOMElement *jointNode, COME_Patient patient ){
	
	
	list<COME_MoleculesCartilage*>::const_iterator iter;
	for( iter = ((list<COME_MoleculesCartilage *>*)patient.getPtOrganList())->begin(); iter != ((list<COME_MoleculesCartilage *>*)patient.getPtOrganList())->end(); iter++ ){
	
		if( ( (*iter)->getDescription() != "clamp0" ) && ( (*iter)->getDescription() != "clamp1" ) ){
			
			char chrPtTemp[100];
			DOMElement *organNode = doc->createElement( XMLString::transcode("molecule_organ" ) );
			organNode->setAttribute( XMLString::transcode("description"), XMLString::transcode((*iter)->getDescription().c_str()) );
			organNode->setAttribute( XMLString::transcode("mesh_file"), XMLString::transcode((*iter)->getSurface()->getFileName().c_str() ) );
			sprintf( chrPtTemp, "%10.6f", (*iter)->getGlobalHooke() );
			organNode->setAttribute( XMLString::transcode("global_hooke"), XMLString::transcode( chrPtTemp ) );
	
			bool isLigament = false;
			list<COME_Molecule*>::const_iterator iterM;
			for( iterM = (*iter)->getTissue()->getShape()->begin(); iterM != (*iter)->getTissue()->getShape()->end(); iterM++ ){
	
				DOMElement *moleculeNode = doc->createElement( XMLString::transcode("molecule" ) );
				char chrPtTemp[100];
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getRadius() );
				moleculeNode->setAttribute( XMLString::transcode("radius"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getFrictionConst() );
				moleculeNode->setAttribute( XMLString::transcode("friction_const"), XMLString::transcode(chrPtTemp) );
				moleculeNode->setAttribute( XMLString::transcode("material"), XMLString::transcode((*iterM)->getMaterial()->getDescription().c_str() ) );
				COME_Point3D position = (*iterM)->getPosition();
				sprintf( chrPtTemp, "%10.6f", position.getZ() );
				moleculeNode->setAttribute( XMLString::transcode("z"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", position.getY() );
				moleculeNode->setAttribute( XMLString::transcode("y"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", position.getX() );
				moleculeNode->setAttribute( XMLString::transcode("x"), XMLString::transcode(chrPtTemp) );
				if( (*iterM)->isFixed() ){
					moleculeNode->setAttribute( XMLString::transcode("fixed"), XMLString::transcode( "true" ) );
				} else {
					moleculeNode->setAttribute( XMLString::transcode("fixed"), XMLString::transcode( "false" ) );
				}
				moleculeNode->setAttribute( XMLString::transcode("description"), XMLString::transcode( (*iterM)->getDescription().c_str() ) );
				if( (*iterM)->getDescription() == "origin" ) isLigament = true;

				organNode->appendChild( moleculeNode );
			}
			if( isLigament ){
				organNode->setAttribute( XMLString::transcode("type"), XMLString::transcode("ligament") );
			} else {
				organNode->setAttribute( XMLString::transcode("type"), XMLString::transcode("cartilage") );
			}
			jointNode->appendChild( organNode );
			
			saveSurface( (*iter)->getSurface(), (*iter)->getDescription() + ".ivc" );
		}
	}
}


void
COME_Xml::saveSurface( COME_Mesh *mesh, string f_name ){

	FILE *file;
	string strMode;

	strMode = "wt";// create new

	if( file = fopen( f_name.c_str(), strMode.c_str() ) ){
	
		if( mesh ){
			vector<COME_Vertex3D> *verts = mesh->getVerticesGlobalPt();
			//vector<COME_Vertex3D> *vertsOriginal = ((COME_MoleculesCartilage*)parent)->getSurface()->getVerticesPt();
			for ( int i = 0; i < verts->size(); i++ ){
	
				COME_Vertex3D vertex = verts->at(i);
				COME_Point3D minusoldhip( -97.9, 4.5, -25.7 );
				COME_Point3D minusfemuroperation( -0.117, 0, -0.352 );
				vertex+=minusfemuroperation;
				vertex.x = vertex.x / 0.005;
				vertex.y = vertex.y / 0.005;
				vertex.z = vertex.z / 0.005;
				vertex+= minusoldhip;
	
				fprintf( file, "%f %f %f,\n", vertex.x, vertex.y, vertex.z );
			}
			fclose(file);
		}
	}

}

//------------------------------------------------------------------------------------------------------------------------

int
COME_Xml::saveAnimationFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator) {

	// Initialize the XML4C system
	try
	{
		XMLPlatformUtils::Initialize();
	}
	
	catch (const XMLException& toCatch)
	{
		printf( "Error during initialization! : %s\n", toCatch.getMessage() );
		return 0;
	}

	XMLCh tempStr[100];
        XMLString::transcode("LS", tempStr, 99, XMLPlatformUtils::fgMemoryManager);
        DOMImplementation *impl = DOMImplementationRegistry::getDOMImplementation(tempStr);
        DOMWriter* theSerializer = ((DOMImplementationLS*)impl)->createDOMWriter();

	// Set some features on this serializer
        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true))
            theSerializer->setFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true);

        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true))
             theSerializer->setFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true);
	
	XMLString::transcode("\n", tempStr, 99);
	theSerializer->setNewLine( tempStr );

        // Create document
	XMLString::transcode("animation", tempStr, 99);
	DOMDocumentType *comeDocType = impl->createDocumentType( tempStr, 0, XMLString::transcode("comeanimation.dtd") );
        DOMDocument*   doc = impl->createDocument(0, tempStr, comeDocType );
        DOMElement*   come = doc->getDocumentElement();

	char chrPtTemp[100];
	DOMText* textValue;
	
	// Add version tag to document
        DOMElement* version = doc->createElement(XMLString::transcode("version"));
	textValue = doc->createTextNode(XMLString::transcode("0.1"));
	version->appendChild(textValue);
        come->appendChild(version);
	
	// Add description tag to document
        DOMElement* description = doc->createElement(XMLString::transcode("description"));
	textValue = doc->createTextNode(XMLString::transcode(file_name.c_str()));
	description->appendChild(textValue);
        come->appendChild(description);
	
	// Add length tag to document
        DOMElement* length = doc->createElement(XMLString::transcode("length"));
	sprintf( chrPtTemp, "%10.6f", simulator->getDuration() );
	textValue = doc->createTextNode(XMLString::transcode(chrPtTemp));
	length->appendChild(textValue);
        come->appendChild(length);
	
	// Add timestep tag to document
        DOMElement* timestep = doc->createElement(XMLString::transcode("timestep"));
	sprintf( chrPtTemp, "%10.6f", simulator->getTimestep() );
	textValue = doc->createTextNode(XMLString::transcode(chrPtTemp));
	timestep->appendChild(textValue);
        come->appendChild(timestep);
	
	// Add number of frames tag to document
        DOMElement* nFrames = doc->createElement(XMLString::transcode("nFrames"));
	//sprintf( chrPtTemp, "%d", (int)( ceil(simulator->getDuration()/COME::flagAnimationResolution) ));
		framesOnFile = framesOnFile / ((list<COME_BioStructure *>*)(patientL.front()->getPtOrganList()))->size();
		sprintf( chrPtTemp, "%d", framesOnFile );
	textValue = doc->createTextNode(XMLString::transcode(chrPtTemp));
	nFrames->appendChild(textValue);
        come->appendChild(nFrames);
	
	// Add organs tag to document
        DOMElement* organs = doc->createElement(XMLString::transcode("organs"));
        come->appendChild(organs);
	saveOrgansAnimation(doc, organs, patientL.front());
	
	// Add joints tag to document
        DOMElement* joints = doc->createElement(XMLString::transcode("joints"));
        come->appendChild(joints);
	saveJointsAnimation( doc, joints, patientL.front()->getRootJoint() );


        // StdOutFormatTarget prints the resultant XML stream
        // to stdout once it receives any thing from the serializer.
	file_name = COME::baseFolder + "/" + file_name;
        XMLFormatTarget *myFormTarget = new LocalFileFormatTarget( file_name.c_str() );

        try {
            // do the serialization through DOMWriter::writeNode();
            theSerializer->writeNode(myFormTarget, *doc);
        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            printf( "Exception message is: \n %s\n ", message );
            delete [] message;
            return -1;
        }
        catch (const DOMException& toCatch) {
            char* message = XMLString::transcode(toCatch.msg);
            printf( "Exception message is: \n %s\n ", message );
            delete [] message;
            return -1;
        }
        catch (...) {
            cout << "Unexpected Exception \n" ;
            return -1;
        }

        theSerializer->release();
        delete myFormTarget;
        return 0;
}

// Add all organs tags to document
void
COME_Xml::saveOrgansAnimation( DOMDocument *doc, DOMElement *organsNode, COME_Patient *patient ){

	list<COME_BioStructure*>::const_iterator iter;
	for( iter = ((list<COME_BioStructure *>*)patient->getPtOrganList())->begin(); iter != ((list<COME_BioStructure *>*)patient->getPtOrganList())->end(); iter++ ){
	
		if( ( (*iter)->getDescription() != "clamp0" ) && ( (*iter)->getDescription() != "clamp1" ) ){
		
			DOMElement* organ = doc->createElement(XMLString::transcode("organ"));
			organsNode->appendChild(organ);
			organ->setAttribute( XMLString::transcode("description"), XMLString::transcode((*iter)->getDescription().c_str() ) );
			organ->setAttribute( XMLString::transcode("mesh"), XMLString::transcode((*iter)->getSurface()->getFileName().c_str() ) );
			
			string pts_file = ((COME_MoleculesCartilage*)(*iter))->getDeformationFile();
			string tempStr( pts_file, 0, pts_file.size()-10 );
			string stress_file = tempStr + "stress.dat";

			organ->setAttribute( XMLString::transcode("points"), XMLString::transcode(pts_file.c_str() ) );
			organ->setAttribute( XMLString::transcode("stress"), XMLString::transcode(stress_file.c_str() ) );
		}
	}
}

// Add all joints tags to document
void
COME_Xml::saveJointsAnimation( DOMDocument *doc, DOMElement *jointsNode, COME_Joint *currJoint ){

	// Add the current joint
	if( currJoint ){
		DOMElement* joint = doc->createElement(XMLString::transcode("joint"));
		jointsNode->appendChild(joint);
		joint->setAttribute( XMLString::transcode("description"), XMLString::transcode(currJoint->getDescription().c_str() ) );
		string file_a = COME::baseFolder + "/" + currJoint->getDescription() + "angles.dat";
		joint->setAttribute( XMLString::transcode("angles_file"), XMLString::transcode(file_a.c_str() ) );
		
		// Call this function recursively to add all joints
		list<COME_Joint*>::iterator childAux;
		int conta = 0;
		for( childAux = currJoint->getChildListPt()->begin(); conta < currJoint->getChildListPt()->size();  childAux++, conta++ ){
			// strangely the test childAux != currJoint->getChildList().end() doesn't work to stop the loop, so I used "conta" and implemented getChildPt
			saveJointsAnimation( doc, jointsNode, (*childAux) );
		}
	}
}


/////////////////////// methods for integration process /////////////////////
int
COME_Xml::loadIntegrationFile( string fileName, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene ){


    // Initialize the XML4C systemvoid
    try
    {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& toCatch)
    {
         printf( "Error during initialization! : %s\n", toCatch.getMessage() );
         return 0;
    }

    XercesDOMParser::ValSchemes    valScheme = XercesDOMParser::Val_Auto; //Val_Never Val_Always
    bool                     doNamespaces    = false;

    // Instantiate the DOM parser.
    XercesDOMParser parser;
    parser.setValidationScheme(valScheme);
    parser.setDoNamespaces(doNamespaces);

    // And create our error handler and install it
    DOMCountErrorHandler errorHandler;
    parser.setErrorHandler(&errorHandler);

    //
    //  Get the starting time and kick off the parse of the indicated
    //  file. Catch any exceptions that might propogate out of it.
    //
    unsigned long duration;
    try
    {
        const unsigned long startMillis = XMLPlatformUtils::getCurrentMillis();
        parser.parse( fileName.c_str() );
        const unsigned long endMillis = XMLPlatformUtils::getCurrentMillis();
        duration = endMillis - startMillis;
    }

    catch (const XMLException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << StrX(toCatch.getMessage()) << "\n" << endl;
        return 0;
    }
    catch (const DOMException& toCatch)
    {
        cerr << "\nError during parsing: '" << fileName << "'\n"
             << "Exception message is:  \n"
             << toCatch.msg << "\n" << endl;
        XMLPlatformUtils::Terminate();
        return 0;
    }
    catch (...)
    {
       cerr << "\nUnexpected exception during parsing: '" << fileName << "'\n";
        XMLPlatformUtils::Terminate();
        return 0;
    }

    //
    //  Extract the DOM tree, get the list of all the elements and report the
    //  length as the count of elements.
    //
    if (errorHandler.getSawErrors())
    {
        printf( "\nErrors occured, no output available\n" );
    }
     else
    {
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        DOMDocument *doc = parser.getDocument();
	unsigned int elementCount = doc->getElementsByTagName(XMLString::transcode("*"))->getLength();
			
	/// Test main tag
	root = doc->getDocumentElement();
	if( strcmp( XMLString::transcode(root->getTagName()), "model" ) ){
		printf( "Invalid XML file.\n" );
		exit(0);
	}

	simulatorXml = simulator;

	// Load organs [OK]
	list<COME_BioStructure*> shapesList;
	DOMNodeList *organs = root->getElementsByTagName( XMLString::transcode("organ") );
	for( int oi = 0; oi < organs->getLength(); oi++ ){
	
		if( !XMLString::compareString( organs->item(oi)->getNodeName(), XMLString::transcode("organ" ) ) ){
			DOMNamedNodeMap *attr = organs->item(oi)->getAttributes();
			string name = XMLString::transcode( attr->getNamedItem( XMLString::transcode("name") )->getNodeValue() );
			string file = XMLString::transcode( attr->getNamedItem( XMLString::transcode("file") )->getNodeValue() );

			// Load materials from organ file
			loadMaterialsFromOrganFile( file );

			// load organ from file
			COME_BioStructure* organAux = loadOrganFile( file, patientL, simulator, parentScene );
			organAux->setDescription( name );
			shapesList.push_back( organAux );
		}
	}

	// Load joints [OK]
	DOMNodeList *joints = doc->getDocumentElement()->getElementsByTagName( XMLString::transcode("joint") );
	vector<COME_Joint *> tempJointList;
	for( int ji = 0; ji < joints->getLength(); ji++ ){
	
		if( !XMLString::compareString( joints->item(ji)->getNodeName(), XMLString::transcode("joint" ) ) ){
			DOMNamedNodeMap *attr = joints->item(ji)->getAttributes();
			string description = XMLString::transcode( attr->getNamedItem( XMLString::transcode("description") )->getNodeValue() );
			
			COME_Matrix jointMatrix;
			jointMatrix.setValueAt( 0, 0, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m00") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 0, 1, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m01") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 0, 2, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m02") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 1, 0, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m10") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 1, 1, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m11") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 1, 2, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m12") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 2, 0, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m20") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 2, 1, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m21") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 2, 2, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m22") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 0, 3, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m03") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 1, 3, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m13") )->getNodeValue() ) ) );
			jointMatrix.setValueAt( 2, 3, atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("m23") )->getNodeValue() ) ) );
			
			COME_Vector3D vectorI = COME_Vector3D( jointMatrix.getValueAt(0,0), jointMatrix.getValueAt(0,1), jointMatrix.getValueAt(0,2) );
			COME_Vector3D vectorJ = COME_Vector3D( jointMatrix.getValueAt(1,0), jointMatrix.getValueAt(1,1), jointMatrix.getValueAt(1,2) );
			COME_Vector3D vectorK = COME_Vector3D( jointMatrix.getValueAt(2,0), jointMatrix.getValueAt(2,1), jointMatrix.getValueAt(2,2) );
			COME_Point3D pointPos = COME_Vector3D( jointMatrix.getValueAt(0,3), jointMatrix.getValueAt(1,3), jointMatrix.getValueAt(2,3) );

			// create Dofs [OK]
			DOMNodeList *dofElements = joints->item(ji)->getChildNodes();
			COME_Dof *dofAux = new COME_Dof[3];
			int idof = 0;
			for( int iD = 0; iD < dofElements->getLength(); iD++ ){
		
				if( !XMLString::compareString( dofElements->item(iD)->getNodeName(), XMLString::transcode("limit" ) ) ){
					DOMNamedNodeMap *attr = dofElements->item(iD)->getAttributes();
					string type = XMLString::transcode( attr->getNamedItem( XMLString::transcode("type") )->getNodeValue());
					float min = ( (float)atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("min") )->getNodeValue() ) ) ) * M_PI / 180.0;
					float max = ( (float)atof( XMLString::transcode( attr->getNamedItem( XMLString::transcode("max") )->getNodeValue() ) ) ) * M_PI / 180.0;
					COME_Curve* evoluta = new COME_Bezier();
					float curr = 0.0, rest = 0.0;

					if( type == "FLEX" ){
						dofAux[0] = COME_Dof( vectorI, pointPos, evoluta, min, max, curr, rest );
						dofAux[0].setDescription( type );
						dofAux[0].vpRest();
					}
					if( type == "ADDUCT" ){
						dofAux[1] = COME_Dof( vectorJ, pointPos, evoluta, min, max, curr, rest );
						dofAux[1].setDescription( type );
						dofAux[1].vpRest();
					}
					if( type == "TWIST" ){
						dofAux[2] = COME_Dof( vectorK, pointPos, evoluta, min, max, curr, rest );
						dofAux[2].setDescription( type );
						dofAux[2].vpRest();
					}
				}
			}

			// create Joint
			COME_Joint *newJoint = new COME_PolyaxialJoint( NULL, &(dofAux[0]), &(dofAux[1]), &(dofAux[2]) );
			newJoint->setDescription( description );
			tempJointList.push_back( newJoint );
		}
	}

	// build joint hierarchy
	COME_Joint *rootJoint;
	for( int ih = 0; ih < tempJointList.size(); ih++ ){

		int ihh;
		if( tempJointList[ih]->getDescription() == "root" ){
			tempJointList[ih]->setParent( NULL );
			rootJoint = tempJointList[ih];
		} else
		if( tempJointList[ih]->getDescription() == "hip" ){
			for( ihh = 0; ihh < tempJointList.size(); ihh++ ){
				if( tempJointList[ihh]->getDescription() == "root" ){
					tempJointList[ih]->setParent( tempJointList[ihh] );
					tempJointList[ihh]->vpAddChild( tempJointList[ih] );
				}
			}
		} else
		if( tempJointList[ih]->getDescription() == "knee" ){
			for( ihh = 0; ihh < tempJointList.size(); ihh++ ){
				if( tempJointList[ihh]->getDescription() == "hip" ){
					tempJointList[ih]->setParent( tempJointList[ihh] );
					tempJointList[ihh]->vpAddChild( tempJointList[ih] );
				}
			}
		}
	}

	// Run joints from the root to apply composed transformations
	// This simplified version reads only the first child of every joint, not allowing bifurcation
	COME_Joint *rJoint = rootJoint;
	COME_Matrix composedMatrix, parentMatrix;
	do{
		parentMatrix = composedMatrix;
		composedMatrix = rJoint->getLim() * composedMatrix;
		// add organs to the joint
		list<COME_BioStructure*>::const_iterator iter;
		for( iter = shapesList.begin(); iter != shapesList.end(); iter++ ){

			// test if current organ belongs to this joint then add
			string strROOT = ROOT_PARTS;
			string strHIP = HIP_PARTS;
			if( ( ( rJoint->getDescription() == "root" ) && ( strROOT.find( (*iter)->getDescription(), 0 ) != string::npos ) ) ||
				( ( rJoint->getDescription() == "hip" ) && ( strHIP.find( (*iter)->getDescription(), 0 ) != string::npos ) ) ){

				(*iter)->globalToLocal( composedMatrix, parentMatrix );
				rJoint->vpAddShape( (*iter) );
			}
		}
		if( rJoint->getChildListPt()->empty() )
			break;
	}while( rJoint = rJoint->getChildListPt()->front() );


	// Load movements
	DOMNodeList *simulation = doc->getElementsByTagName( XMLString::transcode("simulation") );
	for( int sim = 0; sim < simulation->getLength(); sim++ ){
	
		if( !XMLString::compareString( simulation->item(sim)->getNodeName(), XMLString::transcode("simulation" ) ) ){
			
			parentScene->movement = new COME_Movement();

			DOMNodeList *motionElements = simulation->item(sim)->getChildNodes();

			int numMotion = 0;
			for( int jM = 0; jM < motionElements->getLength(); jM++ ){
				if( !XMLString::compareString( motionElements->item(jM)->getNodeName(), XMLString::transcode("motion" ) ) ) numMotion++;
			}
			COME_JointMotion *tempTimeLine = new COME_JointMotion [ numMotion ];
			int qtdMotion = 0;
			int tFinal = 0;
			
			int iMotion = 0;
			for( int iM = 0; iM < motionElements->getLength(); iM++ ){
		
				if( !XMLString::compareString( motionElements->item(iM)->getNodeName(), XMLString::transcode("motion" ) ) ){
					DOMNamedNodeMap *attrMotion = motionElements->item(iM)->getAttributes();
					string jointName = XMLString::transcode( attrMotion->getNamedItem( XMLString::transcode("joint_name") )->getNodeValue());
					string typeMotion = XMLString::transcode( attrMotion->getNamedItem( XMLString::transcode("type") )->getNodeValue());
					float t0 = ( (float)atof( XMLString::transcode( attrMotion->getNamedItem( XMLString::transcode("t0") )->getNodeValue() ) ) );
					float tf = ( (float)atof( XMLString::transcode( attrMotion->getNamedItem( XMLString::transcode("tf") )->getNodeValue() ) ) );
					float parameter = ( (float)atof( XMLString::transcode( attrMotion->getNamedItem( XMLString::transcode("parameter") )->getNodeValue() ) ) );
					int motionType = FLEX;
					if( !XMLString::compareString( XMLString::transcode( typeMotion.c_str() ), XMLString::transcode( "ADDUCT" ) ) ){
						motionType = ADDUCT;
					} else if( !XMLString::compareString( XMLString::transcode( typeMotion.c_str() ), XMLString::transcode( "TWIST" ) ) ){
						motionType = TWIST;
					}
					
					tempTimeLine[qtdMotion++] = *( new COME_JointMotion( jointName, motionType, t0, tf, parameter ) );
					if( tf > tFinal ) tFinal = tf;
				}
			}
			parentScene->movement->setTimeFinal( tFinal);
			simulator->setDuration( tFinal );
			parentScene->movement->setQtdMotion( qtdMotion );
			parentScene->movement->setTimeLine( tempTimeLine );
		}
	}

	// Build patient
	COME_Patient* patientN = new COME_Patient( "loaded", 30, MALE, 80, 1.80, rootJoint, shapesList, parentScene );

	// Add patient to the list
	patientL.clear();
	patientL.push_back( patientN );


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Print out the stats that we collected and time taken.
        cout << fileName << ": " << duration << " ms ("
             << elementCount << " elems)." << endl;
    }
    // And call the termination method
    //XMLPlatformUtils::Terminate();

    return 1;
}


void
COME_Xml::loadMaterialsFromOrganFile( string fileName ){

	// Initialize the XML4C system
    try
    {
        XMLPlatformUtils::Initialize();
    }

    catch (const XMLException& toCatch)
    {
         printf( "Error during initialization! : %s\n", toCatch.getMessage() );
         return;
    }

	XercesDOMParser::ValSchemes    valScheme = XercesDOMParser::Val_Auto; //Val_Never Val_Always
	bool doNamespaces    = false;

	// Instantiate the DOM parser.
	XercesDOMParser parser;
	parser.setValidationScheme(valScheme);
	parser.setDoNamespaces(doNamespaces);

	// And create our error handler and install it
	DOMCountErrorHandler errorHandler;
	parser.setErrorHandler(&errorHandler);

	//
	//  Get the starting time and kick off the parse of the indicated
	//  file. Catch any exceptions that might propogate out of it.
	//
	unsigned long duration;
	try
	{
		const unsigned long startMillis = XMLPlatformUtils::getCurrentMillis();
		parser.parse( fileName.c_str() );
		const unsigned long endMillis = XMLPlatformUtils::getCurrentMillis();
		duration = endMillis - startMillis;
	}

	catch (const XMLException& toCatch)
	{
		cerr << "\nError during parsing: '" << fileName << "'\n"
			<< "Exception message is:  \n"
			<< StrX(toCatch.getMessage()) << "\n" << endl;
		return;
	}
	catch (const DOMException& toCatch)
	{
		cerr << "\nError during parsing: '" << fileName << "'\n"
			<< "Exception message is:  \n"
			<< toCatch.msg << "\n" << endl;
		XMLPlatformUtils::Terminate();
		return;
	}
	catch (...)
	{
	cerr << "\nUnexpected exception during parsing: '" << fileName << "'\n";
		XMLPlatformUtils::Terminate();
		return;
	}

	//
	//  Extract the DOM tree, get the list of all the elements and report the
	//  length as the count of elements.
	//
	if (errorHandler.getSawErrors())
	{
		printf( "\nErrors occured, no output available\n" );
	}
	else
	{

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////
		DOMDocument *doc = parser.getDocument();
		DOMNodeList *materialNodes = doc->getElementsByTagName( XMLString::transcode("material") );
		loadMaterials( materialNodes );
	}
}

int
COME_Xml::exportIntegrationFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator) {

	printf("Just came in \n");
	// Initialize the XML4C system
	try
	{
		XMLPlatformUtils::Initialize();
	}
	
	catch (const XMLException& toCatch)
	{
		printf( "Error during initialization! : %s\n", toCatch.getMessage() );
		return 0;
	}

	XMLCh tempStr[100];
        XMLString::transcode("LS", tempStr, 99, XMLPlatformUtils::fgMemoryManager);
        DOMImplementation *impl = DOMImplementationRegistry::getDOMImplementation(tempStr);
        DOMWriter* theSerializer = ((DOMImplementationLS*)impl)->createDOMWriter();

	// Set some features on this serializer
        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true))
            theSerializer->setFeature(XMLUni::fgDOMWRTDiscardDefaultContent, true);

        if (theSerializer->canSetFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true))
             theSerializer->setFeature(XMLUni::fgDOMWRTFormatPrettyPrint, true);
	
	printf("after exception, before funny things \n");

	XMLString::transcode("\n", tempStr, 99);
	theSerializer->setNewLine( tempStr );

        // Create document
	XMLString::transcode("come", tempStr, 99);
	DOMDocumentType *comeDocType = impl->createDocumentType( tempStr, 0, XMLString::transcode("comescene.dtd") );
        DOMDocument*   doc = impl->createDocument(0, tempStr, comeDocType );
        DOMElement*   come = doc->getDocumentElement();

	char chrPtTemp[100];

	printf("Start adding tags \n");
	
	// Add scene tag to document
    XMLString::transcode("scene", tempStr, 99);
    DOMElement*   scene = doc->createElement(tempStr);
    come->appendChild(scene);
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gX"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gY"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getScene()->getGravity().getX() ) ;
	scene->setAttribute( XMLString::transcode("gZ"), XMLString::transcode(chrPtTemp) );

	// Add Simulation tag and attributes to document
	XMLString::transcode("simulation", tempStr, 99);
    DOMElement*   simulation = doc->createElement(tempStr);
    scene->appendChild(simulation);
		
	sprintf( chrPtTemp, "%10.6f", simulator->getFPS() ) ;
	simulation->setAttribute( XMLString::transcode("fps"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", simulator->getDuration() ) ;
	simulation->setAttribute( XMLString::transcode("duration"), XMLString::transcode(chrPtTemp) );
	simulation->setAttribute( XMLString::transcode("cycle"), XMLString::transcode("off") );
	simulation->setAttribute( XMLString::transcode("type"), XMLString::transcode("offline") );

	//Add motion to document
	COME_JointMotion *timeline = simulator->getScene()->movement->getTimeline();
	int numMotion = simulator->getScene()->movement->getQtdMotion();
	for( int iMotion = 0; iMotion < numMotion; iMotion++ ){

		XMLString::transcode("motion", tempStr, 99);
		DOMElement*   motion = doc->createElement(tempStr);
		simulation->appendChild(motion);
		motion->setAttribute( XMLString::transcode("joint_name"), XMLString::transcode( timeline[iMotion].getJointName().c_str() ) );
		if( timeline[iMotion].getMotionType() == TWIST )
			motion->setAttribute( XMLString::transcode("type"), XMLString::transcode( "TWIST" ) );
		if( timeline[iMotion].getMotionType() == FLEX )
			motion->setAttribute( XMLString::transcode("type"), XMLString::transcode( "FLEX" ) );
		if( timeline[iMotion].getMotionType() == ADDUCT )
			motion->setAttribute( XMLString::transcode("type"), XMLString::transcode( "ADDUCT" ) );

		sprintf( chrPtTemp, "%10.6f", timeline[iMotion].getTimeIni() );
		motion->setAttribute( XMLString::transcode("t0"), XMLString::transcode( chrPtTemp ) );
		sprintf( chrPtTemp, "%10.6f", timeline[iMotion].getTimeFin() );
		motion->setAttribute( XMLString::transcode("tf"), XMLString::transcode( chrPtTemp ) );
		sprintf( chrPtTemp, "%10.6f", timeline[iMotion].getParameter() );
		motion->setAttribute( XMLString::transcode("parameter"), XMLString::transcode( chrPtTemp ) );
	}


	printf("Start adding patient \n" );
	// Add patients to document
	// the front of the patient's list is used because there is only one patient in the scene
	XMLString::transcode("patient", tempStr, 99);
    DOMElement*   patient = doc->createElement(tempStr);
    scene->appendChild(patient);
		
	patient->setAttribute( XMLString::transcode("name"), XMLString::transcode(patientL.front()->getName().c_str() ) );
	if( patientL.front()->getGender() == MALE ){
		patient->setAttribute( XMLString::transcode("gender"), XMLString::transcode("M") );
	} else {
		patient->setAttribute( XMLString::transcode("gender"), XMLString::transcode("F") );
	}
	sprintf( chrPtTemp, "%10.6f", patientL.front()->getWeight() );
	patient->setAttribute( XMLString::transcode("weight"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%10.6f", patientL.front()->getHeight() );
	patient->setAttribute( XMLString::transcode("height"), XMLString::transcode(chrPtTemp) );
	sprintf( chrPtTemp, "%d", patientL.front()->getAge() );
	patient->setAttribute( XMLString::transcode("age"), XMLString::transcode(chrPtTemp) );

	saveMaterials( doc, patient, patientL.front() );
	
	// Add joints and organs
	COME_Joint *rJoint = patientL.front()->getRootJoint();
	COME_Matrix composedMatrix, parentMatrix;
	do{

		XMLString::transcode("joint", tempStr, 99);
		DOMElement* joint = doc->createElement(tempStr);
		patient->appendChild( joint );
		joint->setAttribute( XMLString::transcode("description"), XMLString::transcode( rJoint->getDescription().c_str() ) );
		string parentDesc = "null";
		if( rJoint->getParent() )
			parentDesc = rJoint->getParent()->getDescription();
		joint->setAttribute( XMLString::transcode("parent"), XMLString::transcode( parentDesc.c_str() ) );
		joint->setAttribute( XMLString::transcode("type"), XMLString::transcode("triaxial" ) );

		// Add Dofs of this joint (only triaxial)
		COME_Dof** dofList = rJoint->getDofList();
		for( int iD = 0; iD < 3; iD++ ){

			XMLString::transcode("dof", tempStr, 99);
			DOMElement* dof = doc->createElement(tempStr);
			joint->appendChild( dof );
			dof->setAttribute( XMLString::transcode("description"), XMLString::transcode( dofList[iD]->getDescription().c_str() ) );
			
			COME_Point3D dofPos = dofList[iD]->getPosition();
			XMLString::transcode("position", tempStr, 99);
			DOMElement* position = doc->createElement(tempStr);
			dof->appendChild( position );

			sprintf( chrPtTemp, "%10.6f", dofPos.getX() );
			position->setAttribute( XMLString::transcode("x"), XMLString::transcode( chrPtTemp ) );
			sprintf( chrPtTemp, "%10.6f", dofPos.getY() );
			position->setAttribute( XMLString::transcode("y"), XMLString::transcode( chrPtTemp ) );
			sprintf( chrPtTemp, "%10.6f", dofPos.getZ() );
			position->setAttribute( XMLString::transcode("z"), XMLString::transcode( chrPtTemp ) );

			COME_Vector3D dofAxis = dofList[iD]->getAxis();
			XMLString::transcode("axis", tempStr, 99);
			DOMElement* axis = doc->createElement(tempStr);
			dof->appendChild( axis );
			sprintf( chrPtTemp, "%10.6f", dofAxis.getX() );
			axis->setAttribute( XMLString::transcode("x"), XMLString::transcode( chrPtTemp ) );
			sprintf( chrPtTemp, "%10.6f", dofAxis.getY() );
			axis->setAttribute( XMLString::transcode("y"), XMLString::transcode( chrPtTemp ) );
			sprintf( chrPtTemp, "%10.6f", dofAxis.getZ() );
			axis->setAttribute( XMLString::transcode("z"), XMLString::transcode( chrPtTemp ) );

			XMLString::transcode("range", tempStr, 99);
			DOMElement* range = doc->createElement(tempStr);
			dof->appendChild( range );
			sprintf( chrPtTemp, "%10.6f", dofList[iD]->getMin() );
			range->setAttribute( XMLString::transcode("min"), XMLString::transcode( chrPtTemp ) );
			sprintf( chrPtTemp, "%10.6f", dofList[iD]->getMax() );
			range->setAttribute( XMLString::transcode("max"), XMLString::transcode( chrPtTemp ) );
		}
		
		// Add MoleculeOrgans of this joint
		saveJointOrgans( doc, joint, rJoint );

		// Add specification for functional information export

		
		// Test if all the joints have been included
		if( rJoint->getChildListPt()->empty() )
			break;

	}while( rJoint = rJoint->getChildListPt()->front() );
	
    // StdOutFormatTarget prints the resultant XML stream
    // to stdout once it receives any thing from the serializer.
    XMLFormatTarget *myFormTarget = new LocalFileFormatTarget( file_name.c_str() );

    try {
        // do the serialization through DOMWriter::writeNode();
        theSerializer->writeNode(myFormTarget, *doc);
    }
    catch (const XMLException& toCatch) {
        char* message = XMLString::transcode(toCatch.getMessage());
        printf( "Exception message is: \n %s\n ", message );
        delete [] message;
        return 0;
    }
    catch (const DOMException& toCatch) {
        char* message = XMLString::transcode(toCatch.msg);
        printf( "Exception message is: \n %s\n ", message );
        delete [] message;
        return 0;
    }
    catch (...) {
        cout << "Unexpected Exception \n" ;
        return 0;
    }

    theSerializer->release();
    delete myFormTarget;
    return 1;
}


void
COME_Xml::saveJointOrgans( DOMDocument *doc, DOMElement *jointNode, COME_Joint *jointThis ){
	
	
	list<COME_MoleculesCartilage*>::const_iterator iter;
	for( iter = ((list<COME_MoleculesCartilage *>*)jointThis->getShapeListPt())->begin(); iter != ((list<COME_MoleculesCartilage *>*)jointThis->getShapeListPt())->end(); iter++ ){
	
		if( ( (*iter)->getDescription() != "clamp0" ) && ( (*iter)->getDescription() != "clamp1" ) ){
			
			char chrPtTemp[100];
			DOMElement *organNode = doc->createElement( XMLString::transcode("molecule_organ" ) );
			organNode->setAttribute( XMLString::transcode("description"), XMLString::transcode((*iter)->getDescription().c_str()) );
			organNode->setAttribute( XMLString::transcode("mesh_file"), XMLString::transcode((*iter)->getSurface()->getFileName().c_str() ) );
			sprintf( chrPtTemp, "%10.6f", (*iter)->getGlobalHooke() );
			organNode->setAttribute( XMLString::transcode("global_hooke"), XMLString::transcode( chrPtTemp ) );
	
			bool isLigament = false;
			list<COME_Molecule*>::const_iterator iterM;
			for( iterM = (*iter)->getTissue()->getShape()->begin(); iterM != (*iter)->getTissue()->getShape()->end(); iterM++ ){
	
				DOMElement *moleculeNode = doc->createElement( XMLString::transcode("molecule" ) );
				char chrPtTemp[100];
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getRadius() );
				moleculeNode->setAttribute( XMLString::transcode("radius"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", (*iterM)->getFrictionConst() );
				moleculeNode->setAttribute( XMLString::transcode("friction_const"), XMLString::transcode(chrPtTemp) );
				moleculeNode->setAttribute( XMLString::transcode("material"), XMLString::transcode((*iterM)->getMaterial()->getDescription().c_str() ) );
				COME_Point3D position = (*iterM)->getPosition();
				sprintf( chrPtTemp, "%10.6f", position.getZ() );
				moleculeNode->setAttribute( XMLString::transcode("z"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", position.getY() );
				moleculeNode->setAttribute( XMLString::transcode("y"), XMLString::transcode(chrPtTemp) );
				sprintf( chrPtTemp, "%10.6f", position.getX() );
				moleculeNode->setAttribute( XMLString::transcode("x"), XMLString::transcode(chrPtTemp) );
				if( (*iterM)->isFixed() ){
					moleculeNode->setAttribute( XMLString::transcode("fixed"), XMLString::transcode( "true" ) );
				} else {
					moleculeNode->setAttribute( XMLString::transcode("fixed"), XMLString::transcode( "false" ) );
				}
				moleculeNode->setAttribute( XMLString::transcode("description"), XMLString::transcode( (*iterM)->getDescription().c_str() ) );
				if( (*iterM)->getDescription() == "origin" ) isLigament = true;

				organNode->appendChild( moleculeNode );
			}
			if( isLigament ){
				organNode->setAttribute( XMLString::transcode("type"), XMLString::transcode("ligament") );
			} else {
				organNode->setAttribute( XMLString::transcode("type"), XMLString::transcode("cartilage") );
			}
			jointNode->appendChild( organNode );
			
			//saveSurface( (*iter)->getSurface(), (*iter)->getDescription() + ".ivc" );
		}
	}
}



//------------------------------------------------------------------------------------------------------------------------

ostream& operator<<(ostream& target, const StrX& toDump){
    target << toDump.localForm();
    return target;
}

//---------------------------Error methods---------------------------

DOMCountErrorHandler::DOMCountErrorHandler() :

    fSawErrors(false)
{
}

DOMCountErrorHandler::~DOMCountErrorHandler()
{
}


// ---------------------------------------------------------------------------
//  DOMCountHandlers: Overrides of the SAX ErrorHandler interface
// ---------------------------------------------------------------------------
void DOMCountErrorHandler::error(const SAXParseException& e)
{
    fSawErrors = true;
    cerr << "\nError at file " << StrX(e.getSystemId())
         << ", line " << e.getLineNumber()
         << ", char " << e.getColumnNumber()
         << "\n  Message: " << StrX(e.getMessage()) << endl;
}

void DOMCountErrorHandler::fatalError(const SAXParseException& e)
{
    fSawErrors = true;
    cerr << "\nFatal Error at file " << StrX(e.getSystemId())
         << ", line " << e.getLineNumber()
         << ", char " << e.getColumnNumber()
         << "\n  Message: " << StrX(e.getMessage()) << endl;
}

void DOMCountErrorHandler::warning(const SAXParseException& e)
{
    cerr << "\nWarning at file " << StrX(e.getSystemId())
         << ", line " << e.getLineNumber()
         << ", char " << e.getColumnNumber()
         << "\n  Message: " << StrX(e.getMessage()) << endl;
}

void DOMCountErrorHandler::resetErrors()
{
}


bool DOMCountErrorHandler::getSawErrors() const
{
    return fSawErrors;
}
//----------------------------------------------------
