/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

///////////////////////////////////////////////////////////////////
//
//  PROJECT.....: CO-ME
//  RESPONSIBLE.: 
//
//  FILE........: comexml.h
//  DESCRIPTION.: Contain the COME_Xml class declarations.
//
//  AUTHOR......: Anderson Maciel
//  DATE........: September/10/2002
//  DESCRIPTION.: Class and new methods declaration.
//
///////////////////////////////////////////////////////////////////

#ifndef __COME_XML_H
#define __COME_XML_H

#include	<general/come.h>
#include	<bio/comematerial.h>
#include	<bio/comepatient.h>
#include	<bio/comebiostructure.h>
#include	<physics/comeforce.h>

#include	<xercesc/util/XercesDefs.hpp>
#include	<xercesc/util/PlatformUtils.hpp>
#include	<xercesc/sax/SAXException.hpp>
#include	<xercesc/sax/SAXParseException.hpp>
#include	<xercesc/parsers/XercesDOMParser.hpp>
#include	<xercesc/dom/DOM.hpp>

using namespace XERCES_CPP_NAMESPACE;

#ifndef __COME_SIMULATOR_H	
class COME_Simulator;
#endif

#define ROOT_PARTS "ilium|lunate|laburum"
#define HIP_PARTS "femur|femoralcap|iliofemoral|ischiofemoral|pubofemoral|teres|transverse|capsular"

class COME_Xml : public COME {

	private:
		vector<COME_Material*>*	materials;
		DOMElement	*root;
		COME_Simulator	*simulatorXml;

		COME_Dof	**auxVec; /// Temporary vector of DOF pointers to every dof in the tree
		int		iIndDofAux; /// Index for the vector above.

	protected:		
		COME_BioStructure *	loadMoleculeOrgan( DOMNode *node );		
		void			loadMaterials( DOMNodeList *node );		

		void	saveMoleculeOrgan( DOMDocument *doc, DOMElement *jointNode, COME_Patient patient );
		void	saveMaterials( DOMDocument *doc, DOMElement *patientNode, COME_Patient patient );

		COME_Joint*			loadJoint( DOMNode *node, COME_Joint *parent, list<COME_BioStructure*> &patientOrganList );
		COME_Dof*			loadDofs( DOMNodeList *list );
		vector<COME_BioStructure*>	loadShapes( DOMNodeList *list );
		void				loadModifiers( DOMNodeList *joints );

	public:
		COME_Xml();
		COME_Xml( vector<COME_Material*>* existentMaterials );
		~COME_Xml();
		
		vector<COME_Material*>*	getAllMaterials();

		void	loadSceneFile( string file_name, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene );
		void	loadMaterialsFile( string file_name, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene );
		COME_BioStructure* loadOrganFile( string file_name, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene );
		
		int	saveSceneFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator );
		void	saveSurface( COME_Mesh *mesh, string f_name );
		int	saveAnimationFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator);
		void	saveOrgansAnimation( DOMDocument *doc, DOMElement *organsNode, COME_Patient *patient );
		void	saveJointsAnimation( DOMDocument *doc, DOMElement *jointsNode, COME_Joint *currJoint );

		int		loadIntegrationFile( string fileName, list<COME_Patient*> &patientL, COME_Simulator *simulator, COME_Scenario *parentScene );
		void	loadMaterialsFromOrganFile( string fileName );
		int		exportIntegrationFile( string file_name, const list<COME_Patient*> &patientL, COME_Simulator *simulator);
		void	saveJointOrgans( DOMDocument *doc, DOMElement *jointNode, COME_Joint *jointThis );
};


#include <xercesc/sax/ErrorHandler.hpp>

//class SAXParseException;


// ---------------------------------------------------------------------------
//  Simple error handler deriviative to install on parser
// ---------------------------------------------------------------------------
class DOMCountErrorHandler : public ErrorHandler
{
public:
    // -----------------------------------------------------------------------
    //  Constructors and Destructor
    // -----------------------------------------------------------------------
    DOMCountErrorHandler();
    ~DOMCountErrorHandler();


    // -----------------------------------------------------------------------
    //  Getter methods
    // -----------------------------------------------------------------------
    bool getSawErrors() const;


    // -----------------------------------------------------------------------
    //  Implementation of the SAX ErrorHandler interface
    // -----------------------------------------------------------------------
    void warning(const SAXParseException& e);
    void error(const SAXParseException& e);
    void fatalError(const SAXParseException& e);
    void resetErrors();


private :
    // -----------------------------------------------------------------------
    //  Unimplemented constructors and operators
    // -----------------------------------------------------------------------
    DOMCountErrorHandler(const DOMCountErrorHandler&);
    void operator=(const DOMCountErrorHandler&);


    // -----------------------------------------------------------------------
    //  Private data members
    //
    //  fSawErrors
    //      This is set if we get any errors, and is queryable via a getter
    //      method. Its used by the main code to suppress output if there are
    //      errors.
    // -----------------------------------------------------------------------
    bool    fSawErrors;
};


// ---------------------------------------------------------------------------
//  This is a simple class that lets us do easy (though not terribly efficient)
//  trancoding of XMLCh data to local code page for display.
// ---------------------------------------------------------------------------
class StrX
{
public :
    // -----------------------------------------------------------------------
    //  Constructors and Destructor
    // -----------------------------------------------------------------------
    StrX(const XMLCh* const toTranscode)
    {
        // Call the private transcoding method
        fLocalForm = XMLString::transcode(toTranscode);
    }//DOM

    ~StrX()
    {
        delete [] fLocalForm;
    }


    // -----------------------------------------------------------------------
    //  Getter methods
    // -----------------------------------------------------------------------
    const char* localForm() const
    {
        return fLocalForm;
    }

private :
    // -----------------------------------------------------------------------
    //  Private data members
    //
    //  fLocalForm
    //      This is the local code page form of the string.
    // -----------------------------------------------------------------------
    char*   fLocalForm;
};



#endif
