/***************************************************************************
 *   Copyright (C) 2000 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

#include	<stdio.h>
#include	<math.h>

#include	<kinematics/comedof.h>
#include	<kinematics/comejoint.h>
#include	<kinematics/comemodifier.h>

COME_Dof::COME_Dof( void ){

	axis = COME_Vector3D( 0, 0, 1 );
	evoluta = new COME_Bezier();
	position = COME_Point3D( 0, 0, 0 );
	lim = COME_Matrix();
	minAngle = 0;
	maxAngle = 0;
	comfortMinAngle = 0;
	comfortMaxAngle = 0;
	currentPosition = 0;
	restPosition = 0;
	rangeModifier = NULL;
}

COME_Dof::COME_Dof( COME_Vector3D za, COME_Point3D pos, COME_Bezier *evol ){

	axis = za;
	evoluta = evol;
	position = pos;
	lim = COME_Matrix();
	minAngle = 0;
	maxAngle = 0;
	comfortMinAngle = 0;
	comfortMaxAngle = 0;
	currentPosition = 0;
	restPosition = 0;
	rangeModifier = NULL;
}

COME_Dof::COME_Dof( COME_Vector3D za, COME_Point3D pos, COME_Curve *evol, float min, float max, float cur, float rest ){
	
	axis = za;
	evoluta = (COME_Bezier *)evol;
	position = pos;
	lim = COME_Matrix();
	minAngle = min;
	maxAngle = max;
	comfortMinAngle = 0;
	comfortMaxAngle = 0;
	currentPosition = cur;
	restPosition = rest;
	rangeModifier = NULL;
}

COME_Dof::COME_Dof( COME_Vector3D za, COME_Point3D pos, COME_Bezier *evol, float min, float max, float cur, float rest, COME_Modifier *rm ){
	
	axis = za;
	evoluta = evol;
	position = pos;
	lim = COME_Matrix();
	minAngle = min;
	maxAngle = max;
	comfortMinAngle = 0;
	comfortMaxAngle = 0;
	currentPosition = cur;
	restPosition = rest;
	rangeModifier = rm;
}

void
COME_Dof::setDescription( char* szDesc ){
	description = szDesc;
}

void
COME_Dof::setDescription( string sDesc ){
	description = sDesc;
}

string
COME_Dof::getDescription(){
	return description;
}

COME_Vector3D
COME_Dof::getAxis( void ){
	
	return axis;
}

COME_Point3D
COME_Dof::getOrigin( void ){
	
	COME_Point3D p;
	p = lim * p;
	return p;
}

COME_Point3D
COME_Dof::getPosition( void ){
	
	return position;
}

COME_Curve
COME_Dof::getEvoluta( void ){
	
	return *evoluta;
}

COME_Matrix
COME_Dof::getLim( void ){
	
	return lim;
}

COME_Matrix
COME_Dof::getBim( void ){
	
	
	/// Build BIM.
	axis.vpNormalize();
	COME_Vector3D vx = COME_Vector3D( 1, 0, 0 );
	COME_Vector3D vy = axis.vpCrossProduct(vx);
	// Verify linearity
	if( vy == COME_Vector3D( 0, 0, 0 ) ){
		vx = COME_Vector3D( 0, 0, 1 );
		vy = axis.vpCrossProduct(vx);
	}
	vy.vpNormalize();
	vx = vy.vpCrossProduct(axis);
	vx.vpNormalize();
	double			fm[4][4];

 	fm[0][0] = vx.getX();
	fm[1][0] = vx.getY();
	fm[2][0] = vx.getZ();
	fm[3][0] = 0;
	
	fm[0][1] = vy.getX();
	fm[1][1] = vy.getY();
	fm[2][1] = vy.getZ();
	fm[3][1] = 0;
	
	fm[0][2] = axis.getX();
	fm[1][2] = axis.getY();
	fm[2][2] = axis.getZ();
	fm[3][2] = 0;
	
	fm[0][3] = 0;
 	fm[1][3] = 0;
 	fm[2][3] = 0;
 	fm[3][3] = 1;
 	
	COME_Matrix lBim = COME_Matrix( fm );
	return lBim;
}

float
COME_Dof::getMin( void ){
	
	return minAngle;
}

float
COME_Dof::getMax( void ){
	
	return maxAngle;
}

float
COME_Dof::getComfortMin( void ){
	
	return comfortMinAngle;
}

float
COME_Dof::getComfortMax( void ){
	
	return comfortMaxAngle;
}

float
COME_Dof::getCurrentMin( void ){
	
	float minModif = minAngle;
	if( rangeModifier ){
		minModif = rangeModifier->getMin();
	}
	if( minAngle < minModif ){
		return minModif;
	}
	return minAngle;
}

float
COME_Dof::getCurrentMax( void ){
	
	float maxModif = maxAngle;
	if( rangeModifier ){
		maxModif = rangeModifier->getMax();
	}
	if( maxAngle > maxModif ){
		return maxModif;
	}
	return maxAngle;
}

float
COME_Dof::getCurrent( void ){
	
	return currentPosition;
}

float
COME_Dof::getRest( void ){
	
	return restPosition;
}

COME_Joint*
COME_Dof::getOwnerJoint( void ){
	
	return ownerJoint;
}

void
COME_Dof::setAxis( COME_Vector3D za ){
	
	axis = za;
}

void
COME_Dof::setEvoluta( COME_Bezier *evol ){
	
	evoluta = evol;
}

void
COME_Dof::setLim( COME_Matrix m ){
	
	lim = m;
}

void
COME_Dof::setMin( float min ){
	
	minAngle = min;
}

void
COME_Dof::setMax( float max ){
	
	maxAngle = max;
}

void
COME_Dof::setComfortMax( float max ){
	
	comfortMaxAngle = max;
}

void
COME_Dof::setComfortMin( float min ){
	
	comfortMinAngle = min;
}

void
COME_Dof::vpMoveTo( float pos ){
	
	// Avoid values out of range
	if( pos > 1.0 ) pos = 1.0;
	if( pos < 0.0 ) pos = 0.0;
	
	// Get resources (obtain angles and points by interpolation)
	COME_Point3D tgtpos = evoluta->getPointAsPoint( pos ); //target position
	float tgtang = getCurrentMin() + ( ( getCurrentMax() - getCurrentMin() ) * pos ); //target angle
	
	COME_Point3D p = position + COME_Point3D( tgtpos );
		
	// Build transformation matrix
	double r[4][4];
	r[0][0] = cos( tgtang );
	r[0][1] = sin( tgtang );
	r[1][0] = -sin( tgtang );
	r[1][1] = cos( tgtang );
	r[3][0] = p.getX();
	r[3][1] = p.getY();
	r[3][2] = p.getZ();
	r[2][2] = r[3][3] = 1;
	r[0][2] = r[1][2] = r[2][0] = r[2][1] = r[0][3] = r[1][3] = r[2][3] = 0;
	COME_Matrix mt = COME_Matrix( r );
	
	// update to new position
	// lim = bim * T * inv(bim)
	//setLim( (getBim() * mt ) * ( getBim().getTransposed() ) );
	//setLim( ( getBim().getTransposed() ) * (getBim() * mt ) );
	setLim( ( getBim().getTransposed() ) * ( mt * getBim() ) );
	currentPosition = pos;
}


void
COME_Dof::setOwnerJoint( COME_Joint *ow ){
	ownerJoint = ow;
}

void
COME_Dof::setRest( float rest ){
	
	restPosition = rest;
}

void
COME_Dof::vpRest( void ){
	
	vpMoveTo( restPosition );
}

void
COME_Dof::setRangeModifier( COME_Modifier *m ){
	rangeModifier = m;
}

COME_Modifier*
COME_Dof::getRangeModifier( void ){
	return rangeModifier;
}


void
COME_Dof::vpPrintLim( void ){
	printf( "%f %f %f %f \n",getLim().getValueAt(0,0), getLim().getValueAt(0,1),
  												getLim().getValueAt(0,2), getLim().getValueAt(0,3) );
  printf( "%f %f %f %f \n",getLim().getValueAt(1,0), getLim().getValueAt(1,1),
  												getLim().getValueAt(1,2), getLim().getValueAt(1,3) );
	printf( "%f %f %f %f \n",getLim().getValueAt(2,0), getLim().getValueAt(2,1),
  												getLim().getValueAt(2,2), getLim().getValueAt(2,3) );
	printf( "%f %f %f %f \n\n",getLim().getValueAt(3,0), getLim().getValueAt(3,1),
  												getLim().getValueAt(3,2), getLim().getValueAt(3,3) );
}
