/*@#-------------------------------------------------------------------
 PROJECT 	    : ESPRIT 6709			       HUMANOID
 MODULE		    : HUMAN DATA STRUCTURE

 FILE               : geom.h
 CREATION DATE      : 26.02.93
 CREATION AUTHOR(S) : R. Boulic
 ----------------------------------------------------------------------
 KEYWORD(S)	    : 
 PUBLIC TYPE(S)     :
 RELATED TYPE(S)    :

 FUNCTIONALITY      : geometric transformations
 ----------------------------------------------------------------------
 UPDATE    $Revision: 1.7 $
             $Author: abaci $
 ----------------------------------------------------------------------
 LANGAGE	    : ANSI-C
 SYSTEM             : UNIX V 4. / SGI
 GRAPHIC LIBRARY    : SGI GL  
 ----------------------------------------------------------------------
 LAB DIRECTOR       : D. THALMANN
 LAB                : EPFL- LIG
 ------------------------------------------------------------------#@*/

/*! \file
    \ingroup Main
    \brief   A set of functions to perform basic algebraic operations on vectors, matrices and quaternions.
*/

# ifndef libgeom_h
# define libgeom_h


/*
** -------------------------------------
** -- Include of Header files ----------
** -------------------------------------
*/

#include	<stdio.h>	    /* for FILE type	    */


namespace geom {

/*
** -------------------------------------
** -- Public define, typedef, functions
** -------------------------------------
*/
#define M_PI		3.14159265358979323846
#define M_PI_2		1.57079632679489661923
#define LIBGEOM_REAL	double

enum {	ATTRACTION_NULLE     = 0,
	ATTRACTION_COMPLETE  = 1,
	ATTRACTION_PARTIELLE = 2 };

const float EPSIL_MR3D	= 0.00001f;
const float EPSIL_VT3D	= 0.001f;

/* some epsilon value to round off zero	*/

#define EPSIL_ZERO	0.000001
#define EPSIL_BIG_ZERO  0.00001

/* index for vectors, matrices and quaternions */

enum {	GEOM_X_INDEX = 0,
	GEOM_Y_INDEX = 1,
	GEOM_Z_INDEX = 2,
	GEOM_W_INDEX = 3 };

/* m etant un pointeur sur un espace a une dimension mais utilise ici*/
/* pour deux dimensions ; jmax est la taille d'une ligne (= nb de col.)*/
/* le resultat correspond a :  m[i][j]	     (avec int i,j,jmax)*/

#define VIJ(m,i,j,jmax)		(*((m) + (i)*(jmax) + (j)))

typedef LIBGEOM_REAL	MatriX[4][4];	/*!< A 4x4 matrix. Like "gl.h" MatriX   */
typedef LIBGEOM_REAL	MR3D[3][3];	/*!< A 3x3 rotation matrix (orthogonal) */
typedef LIBGEOM_REAL	VT3D[3];	/*!< A 3-components vector              */
typedef LIBGEOM_REAL	VHOM[4];	/*!< A 4-components homogeneous vector  */

typedef	MatriX  MatriX_t;
typedef MR3D    MR3D_t;
typedef VT3D    VT3D_t;
typedef VHOM    VHOM_t;

typedef LIBGEOM_REAL   QUATERNION[4];	/*!< A quaternion : ((x,y,z), w) */


/*----------------------------------------------------------------*/
/*! ARRAY1D represents a 1-dimensional array of double values.
    It is typically used to represent vectors of arbitrary size.
*/
typedef struct {

  double *data;    /*!< Pointer to data */
  int    lines;    /*!< Number of lines (rows) */

} ARRAY1D;

/*! Macro to access an element in a ARRAY1D by its index. */

#define AI(array1d, line)       (*((array1d)->data + (line)))

/*----------------------------------------------------------------*/
/*! ARRAY2D represents a 2-dimensional array of double values.
    It is typically used to represent matrices of arbitrary size.
*/

typedef struct {

  double *data;    /*!< Pointer to data */
  int    lines,    /*!< Number of lines (rows) */
         cols;     /*!< Number of columns */

} ARRAY2D;

/*! Macro to access an element in a ARRAY2D by its two indices. */

#define AIJ(array2d, line, col) (*((array2d)->data + (col) + (line) * (array2d)->cols))

/*-------------------------------------------------------------------*/
/* Use the following functions instead of acos and asin, if you
   are not sure that your argument is in the valid range (because of
   numerical errors, for example). These functions clamp the argument
   to its valid range.
*/

 double safe_acos (double f);  /*!< Unlike acos, this function is tolerant to numerical errors near domain boundaries. */
 double safe_asin (double f);  /*!< Unlike asin, this function is tolerant to numerical errors near domain boundaries. */

/*-------------------------------------------------------------------*/

/*@{ @name MatriX functions.
*/

  /*! Initializes a matrix with a 4x4 identity matrix. */

	void	matrix_idt_ (MatriX_t matrix);

  /*! Copy the "src" matrix into the "dst" matrix. */

	void	matrix_copy (const MatriX_t src, 
			     MatriX_t dst);

  /*! Builds a 4x4 matrix from given MR3D and VT3D components. */

	void	matrix_compose (const MR3D_t   mr, 
				const VT3D_t	 vt, 
				MatriX_t matrix);

  /*! Decompose a given 4x4 matrix into its rotatin (mr) and
      translation (vt) components.
  */

	void	matrix_decompose (MR3D_t   mr, 
				  VT3D_t   vt, 
				  const MatriX_t matrix);

  /*! Produces a copy of a given matrix, but with null translation
      component.
  */

	void    matrix_rotpart   (MatriX_t In, 
				  MatriX_t Res);

  /*! Computes the transpose (at) of a given matrix a */

	void	matrix_transpose (const MatriX_t a,
				  MatriX_t at);

  /*!  */

	void	matrix_mult_vhom (const MatriX_t a, 
				  const VHOM_t   u,
				  VHOM_t   v);

  /*! Multiplies a 4x4 matrix by a 3-vector.

        @remark The term vector may be confusing since in many graphics libs (e.g. Performer), 
	this operation is rather called point multiplication. If you want a true matrix-vector 
	multiply, extract the 3x3 rotation matrix and use mr3d_mult.

	@warning The matrix is supposed to be affine (i.e. rotation and translation components).

	@param a The input 4x4 matrix
	@param u The input vector
	@param v The output vector
  */

	void	matrix_mult_vt3d (const MatriX_t a, 
				  const VT3D_t   u,
				  VT3D_t   v);

  /*! Computes the inverse of a given matrix.

	@warning The input matrix is supposed to be affine,
             i.e. a combination of rotation and translation only.
  */

	void	matrix_inverse (const MatriX_t matrix,
				MatriX_t matrix_inv);

  /*! Computes the inverse of a given matrix. Its determinant must be provided.

	@warning The deteminant must be non-zero (i.e. the matrix must be inversible).
  */

	void	matrix_inverse_det (MatriX_t res,
								LIBGEOM_REAL    determinant,
								const MatriX_t source);

  /*! Returns the cofactor of a given matrix at given indices (i,j) */

	LIBGEOM_REAL	matrix_cofactor (const MatriX_t matrix,
				 int	  i,
				 int	  j);

  /*! Computes the determinant of a 4x4 matrix. */

	LIBGEOM_REAL	matrix_determinant (const MatriX_t matrix);

  /*! Computes matrix multiplication : mc = ma * mb */

	void	matrix_mult (const MatriX_t	ma,
			     const MatriX_t	mb,
			     MatriX_t	mc);	

  /*! Computes matrix multiplication : mc = ma * mb, 
    optimized, *for affine matrices only* */
        void    matrix_mult_opt (const MatriX ma, const MatriX mb, MatriX mc);

  /*! Compares two matrices.

	@return Returns TRUE if both matrices are equal, FALSE otherwise.
  */

	bool matrix_egalite (const MatriX_t a,
				  const MatriX_t b);

  /*! This procedure tries to attract a given matrix to the identity.

	@return ATTRACTION_NULLE, ATTRACTION_PARTIELLE, ATTRACTION_TOTALE

  */

	int   matrix_attraction_idt (MatriX_t	a);

  /*! ??? */

	int	matrix_proximite (const MatriX_t  a,
				  const MatriX_t  b, 
				  LIBGEOM_REAL	    seuil_mr,
				  LIBGEOM_REAL	    seuil_vt);

  /*! Compute interpolation of two matrices, which
      is a spherical linear interpolation of corresponding quaternions.

	@param a Input matrix
	@param b Input matrix
	@param u interpolation parameter (between 0 and 1)
	@param result Output matrix
  */

	void   matrix_lerp (const MatriX_t a,
			    const MatriX_t b,
			    LIBGEOM_REAL    u,
			    MatriX   result);

  /*! Writes a given matrix to a file. */

	bool  matrix_write (const MatriX_t m, 
				 FILE * f);

  /*! Reads a given matrix from a file. */

	bool  matrix_read  (MatriX_t m, 
				 FILE * f);

/*@}*/

/*-------------------------------------------------------------------*/

/*@{ @name MR3D functions.
*/

  /*! Initializes an MR3D matrix with the identity matrix. */

	void 	mr3d_idt_ (MR3D_t    mr);

  /*! Copies the rotation component of a given 4x4 matrix into an MR3D matrix. */

	void	mr3d_get  (const MatriX_t  matrix, 
			   MR3D_t a);

  /*! Copies an MR3D matrix from "src" to "dest". */

	void	mr3d_copy (const MR3D_t src, 
			   MR3D_t dst);

  /*! Copies an MR3D matrix into the rotation component of a 4x4 matrix. */

	void	mr3d_load (MatriX_t matrix, 
			   const MR3D_t a);			 

  /*! Computes the transpose of a given MR3D matrix (a) into at. */

	void	mr3d_transpose (const MR3D_t a,
				MR3D_t at);

  /*! Transposes a given MR3D matrix. */

	void	mr3d_self_transpose (MR3D_t ma);

  /*! Circular permutation of the rows. The first row
      becomes the third row; the second row becomes the
      first row; the third row becomes the second row. The
      result is stored into the "b" matrix.
  */

	void	mr3d_perm_circ (MR3D_t a, 
				MR3D_t b);

  /*! Like mr3d_perm_circ, but the "z" vector (the third row) is negated as well.
      The result is stored in the "b" matrix.
  */

	void	mr3d_opposite (MR3D_t a, 
			       MR3D_t b);

  /*! Negates the "x" and "y" vectors (the first and second rows).
      The result is stored in the "b" matrix.
  */

	void	mr3d_symtxy (MR3D_t a, 
			     MR3D_t b);

  /*! Multiplies each element of the "self" matrix by the given scalar.
      The result is stored in the "res" matrix.
  */

	void    mr3d_scale (const MR3D_t self, 
			    LIBGEOM_REAL scalar, 
			    MR3D_t res);

  /*! Matrix substraction: res = left - right
   */

	void    mr3d_sub  (const MR3D_t left, 
			   const MR3D_t right, 
			   MR3D_t res);

  /*! Matrix addition: res = left + right
  */

	void    mr3d_add  (const MR3D_t left, 
			   const MR3D_t right, 
			   MR3D_t res);

  /*! Matrix multiplication: c = a * b
  */

	void	mr3d_mult (const MR3D_t a, 
			   const MR3D_t b, 
			   MR3D_t c);

  /*! Computes matrix-by-vector multiplication: vv = ma * vu
  */

	void	mr3d_mult_vt3d (const MR3D_t ma,
				const VT3D_t vu,
				VT3D_t	vv);

  /*! Converts an axis-angle vector representation of orientation into
      an MR3D matrix.
  */

	void	mr3d_from_axis_angle (MR3D_t mr, 
				      const VT3D_t axis);

  /*! Converts a given MR3D to an axis-angle vector representation.
  */

	void	mr3d_to_axis_angle   (VT3D_t axis, 
				      const MR3D_t mr);

  /*! Returns the angle of rotation of the given matrix. The result
      is in the interval [0, PI].
  */

	double	mr3d_axis_angle      (const MR3D_t mr);

  /*! Compares two MR3D matrices, with a given tolerance (seuil_mr).
      Returns TRUE if the difference between two corresponding elements
      is always lower than the tolerance.
  */

	bool mr3d_egalite (const MR3D_t a,
			   const MR3D_t b,
			   LIBGEOM_REAL  seuil_mr = EPSIL_MR3D);

  /*! Attraction towards the identity matrix.
  */

	int     mr3d_attraction_idt (MR3D_t a, LIBGEOM_REAL  seuil_mr);

  /*! ??? */

	int     mr3d_attraction_ini (MR3D_t a, LIBGEOM_REAL  seuil_mr);

  /*! conversion de la matrice ligne m en trois angles XYZ mobile
      ang1 et ang3 sont normalises entre + ou - PI, ang2 entre + ou - PI/2
  */

	void 	mr3d_conv_xyz_mob (const MR3D_t m, 
				   LIBGEOM_REAL *ang1,
				   LIBGEOM_REAL *ang2,
				   LIBGEOM_REAL *ang3);

  /*! Conversion de la matrice ligne m en trois angles YXZ mobile
      ang1 et ang3 sont normalises entre + ou - PI, ang2 entre +- PI/2
      correspond a la sequence pour l'analyse des mvts en orthopedie.
  */

	void 	mr3d_conv_yxz_mob (const MR3D_t m, 
				   LIBGEOM_REAL *ang1,
				   LIBGEOM_REAL *ang2,
				   LIBGEOM_REAL *ang3);

  /*! */

	void    mr3d_from_angle_seq_xyz (MR3D mx,
					 LIBGEOM_REAL angX, 
					 LIBGEOM_REAL angY, 
					 LIBGEOM_REAL angZ);

  /*! */

	void    mr3d_from_angle_seq_yxz (MR3D mx, 
					 LIBGEOM_REAL angY, 
					 LIBGEOM_REAL angX, 
					 LIBGEOM_REAL angZ);

  /*! Return rotation matrix R, which transforms vector 'a' into vector 'b', i.e. a = b.R

      a and b are untouched and need not be of unit length
  */

	void    mr3d_from_two_vectors(const VT3D_t a, const VT3D_t b, MR3D R);

  /*! Re-orthogonalize a given MR3D matrix.

	@param limit ???
  */

	void	mr3d_reorthogonalize (MR3D_t R, int limit);

  /*! Write the MR3D matrix to a file.
  */

	bool  mr3d_write (const MR3D_t self, FILE * f);

  /*! Read an MR3D matri
  */

	bool  mr3d_read  (MR3D_t self, FILE * f);

/*@}*/


/*-------------------------------------------------------------------*/

/*@{ @name VT3D functions.
*/

  /*! Gets the translation component of a 4x4 matrix into the "vt" vector.
  */

	void	vt3d_get (const MatriX_t      matrix,
			  VT3D_t	v);

  /*! Puts a given vector into the translation component of a 4x4 matrix.
  */

	void	vt3d_load (MatriX_t	matrix,
			   const VT3D_t	v);

  /*! Copies a source (src) vector to a destination (dst) vector.
  */

	void	vt3d_copy (const VT3D_t src,
			   VT3D_t dst);

  /*! Set to zero the three elements of the given vector.
  */

	void	vt3d_set_null (VT3D_t	v);

  /*! Computes a normalized vector (v) from a given arbitrary vector u.
      If the given vector is close to the null vector, the "v" vector is
      set to zero and FAILURE is returned, otherwise SUCCESS is returned.

  */

	bool vt3d_normalize (const VT3D_t u,
				  VT3D_t v);

  /*! Computes the cross product of two vectors: w = u X v */

	void	vt3d_cross   (const VT3D_t	u,
			      const VT3D_t	v,
			      VT3D_t	w);

  /*! Scales a given vector: v = scale * u */

	void	vt3d_scale   (const VT3D_t  scale,
			      const VT3D_t	u,
			      VT3D_t	v);

  /*! Returns the dot product of the two given vectors. */

	LIBGEOM_REAL	vt3d_get_dot (const VT3D_t	u,
			      const VT3D_t	v);

  /*! Transcripts a given 3-vector (vt3d) into a 4-components
      homogeneous vector (vhom), whose fourth component is 1.

	\deprecated

  */

	void	vt3d_vhom (const VT3D_t vt3d, 
			   VHOM_t vhom);

  /*! Transcripts a given 4-components homogeneous vector (vhom),
      into a 3-components vector (vt3d).

	\deprecated
  */

	void	vhom_vt3d (const VHOM_t vhom, 
			   VT3D_t vt3d);

  /*! Initializes the three components of a vector (w) with
      given values x, y and z.
  */

	void	vt3d_let (LIBGEOM_REAL x, LIBGEOM_REAL y, LIBGEOM_REAL z, VT3D_t w);

  /*! Computes vector addition: w = u + v
  */

	void	vt3d_add (const VT3D_t u, const VT3D_t v, VT3D_t w);

  /*! Computes vector substraction: w = u - v
  */

	void	vt3d_sub (const VT3D_t u, const VT3D_t v, VT3D_t w);

  /*! Vector multiplication, component-wise : w_i = u_i * v_i
  */

	void	vt3d_mult (const VT3D_t u, const VT3D_t v, VT3D_t w);

  /*! Vector division, component-wise : w_i = u_i / v_i
  */

	void	vt3d_div (const VT3D_t u, const VT3D_t v, VT3D_t w);

  /*! Vector scaling (multiplication by a scalar s) : w = s * v */

	void	vt3d_mults (const VT3D_t v, LIBGEOM_REAL s, VT3D_t w);

  /*! Vector scaling (division by a scalar s) : w = v / s */

	void	vt3d_divs (const VT3D_t v, LIBGEOM_REAL s, VT3D_t w);

  /*! Vector negation: dest = - src */

	void  vt3d_neg (const VT3D_t src, VT3D_t dest);

  /*! Returns the norm of the given vector. */

	LIBGEOM_REAL	vt3d_get_norm (const VT3D_t v);

  /*! Returns the squared norm of the given vector v, i.e. the
      dot product of v by itself.
  */

	LIBGEOM_REAL	vt3d_get_norm2 (const VT3D_t v);

  /*! Returns the norm of the vector u-v */

	LIBGEOM_REAL	vt3d_get_dist (const VT3D_t u, const VT3D_t v);

  /*! Returns the distance between a point p and a line described by
      a point (q) and a unit direction vector (n).
  */

	LIBGEOM_REAL	vt3d_get_distptline (const VT3D_t p, const VT3D_t q, const VT3D_t n);

  /*! Returns the distance between a point p and a plane described by
      a point (q) and a unit normal vector to the plane (n).
  */

	LIBGEOM_REAL	vt3d_get_distptplane (const VT3D_t p, const VT3D_t q, const VT3D_t n);

  /*! Rotates a given vector (v) around a special axis (x, y or z)
      by an angle whose sine and cosine must be provided.

	@param v  The vector to be rotated
	@param sa The sine of the rotation angle
	@param ca The cosine of the rotation angle
	@param k  A char that indicates which axis is to be used ('x', 'y' or 'z')
	@param w  The resulting vector
  */

	void	vt3d_rotv (const VT3D_t v, LIBGEOM_REAL sa, LIBGEOM_REAL ca, char k, VT3D_t w);

  /*! Rotate v around k passing by point c by angle A radians, whose sine (sa)
      and cosine (ca) must be provided. */

	void	vt3d_rotvect (const VT3D_t v, const VT3D_t c, VT3D_t k, 
			      LIBGEOM_REAL sa, LIBGEOM_REAL ca, VT3D_t w);

  /*! Tests equality of two VT3D vectors, within a given tolerance (seuil_vt).

	@return Returns TRUE if the vectors are considered equal, and FALSE otherwise.
  */

	bool vt3d_egalite (const VT3D_t a,
				const VT3D_t b, 
				LIBGEOM_REAL	seuil_vt);

  /*! */

	int     vt3d_attraction_nul (VT3D_t a, 
				     LIBGEOM_REAL  seuil_vt);

  /*! Test for point inclusion inside a 2D polygon, non necessarily convex, and lying in the X-Y plane.

	@param point The point to be tested. Only (x,y) components are considered.
	@param p A pointer to an array of points, which are the vertices of the polygon.
	@param N The number of vertices of the polygon.

	@return TRUE if the point is inside the polygon, FALSE otherwise.

	\warning The vertices must be ordered in a clock-wise way.
  */

	bool vt3d_inside_polygon (VT3D point, VT3D *p, int N);

	/*! Linear interpolation between two 3D vectors.
	
	  @param u The interpolation factor between 0. and 1.
	\return The result ranges from p1 (when u=0) to p2 (when u=1)
  */

	void vt3d_lerp (const VT3D p1, const VT3D p2, LIBGEOM_REAL u, VT3D result);


  /*! Spherical linear interpolation between two unit 3D vectors. The
      result is also a unit norm vector.

	@param u The interpolation factor between 0. and 1.

	\return The result ranges from p1 (when u=0) to p2 (when u=1)

	\remark This is a sub-case of the 4D version for quaternions (see quat_slerp()).
  */

	void vt3d_slerp (const VT3D p1, const VT3D p2, LIBGEOM_REAL u, VT3D result);

  /*! Writes a VT3D vector (self) to a file. */

	bool vt3d_write (const VT3D_t self, FILE * f);

  /*! Reads a VT3D vector (self) from a file. */

	bool vt3d_read  (VT3D_t self, FILE *f);

  /*! Rotates a given vector (v_in) by an axis-angle vector. The result
      is stored in the v_out vector. This is based on Rodrigues formula.
  */

	void      vt3d_rotate_around_axis_angle (const VT3D  axis_angle,
                                                 const VT3D  v_in,
                                                 VT3D  v_out);

/*@}*/

/*------------------------------------------------------------------*/

/*@{ @name VTN functions.

  \remark These functions should not be used. Use ARRAY1D functions instead.

*/

  double	*vtn_creation(int	n);
  void	vtn_destr(double	*p);
  void	vtn_nul(double		*p, 
			    int		n);
  double	vtn_norm(const double		*p, 
			    int		n);
  double	vtn_anorm(const double	*p, 
			    int		n);
  void	vtn_copy(const double	*psrc,
			    double	*pdst, 
			    int		n);
  void	vtn_add(const double		*p1,
		const double	*p2,
			    double	*pdst, 
			    int		n);
  void	vtn_sub(const double		*p1,
		const double	*p2,
			    double	*pdst, 
			    int		n);
  void    vtn_mults (const double  *psrc, 
			   double  *pdst, 
			   int n, 
			   double  alpha);
  double	vtn_dot(const double		*p1,
			const double	*p2,
			    int		n);
  double	vtn_inv(const double		*p,
			    double	*p_inv,
			    int		n);

/*@}*/

/*------------------------------------------------------------------*/

/*@{ @name MatMxN functions.

  \remark These functions should not be used. Use ARRAY2D functions instead.

*/

  double	*matmxn_creation (int	lmax,
				  int	cmax);
  void	matmxn_destr (double	*p);
  void	matmxn_copy  (const double * a, double * b, int lmax, int cmax);

  void	matmxn_nul (double	*p, 
			    int		lmax,
			    int		cmax);
  void	matmxn_idt (double	*p, 
			    int		lmax,
			    int		cmax);
  void	matmxn_mult (const double	*a,
		     const double	*b,
			     double	*c, 
			     int	imax,
			     int	jmax,
			     int	kmax);
  void	matmxn_mult_vtn (const double	*a,
			 const double	*b,
				 double	*c, 
				 int	lmax ,
				 int	cmax);
  void    matmxn_write (double *d,
                              int    height,
                              int    width,
                              FILE   *f);
  void    matmxn_transpose (const double *src,
                                  double *dest,
                                  int    src_width,
                                  int    src_height);
  bool  matmxn_pseudo_inverse (double  *m,
					  int	lmax,
					  int	cmax, 
					  double *mp);
  void matmxn_damped_least_square_inverse (double *j_inv, 
						 double *j_mat, 
						 int j_width, 
						 int j_height, 
						 double damping);
  void matmxn_svdcmp(double **a,int m,int n,double *w,double **v);	
                   /* Given a matrix a[1..m][1..n], this routine computes 
		      its singular value decomposition. 
		      A = U * W * Vt. The matrix U replaces a on output. The
		      diagonal matrix of singular values W is output as
		      vector w[1..n]. The matrix V (not the transpose Vt)
		      is output as v[1..n][1..n]. 
		      "Numerical Recipes in C", 2nd edition ! 
		      */

/*@}*/

/*------------------------------------------------------------------*/
/*
 * QUATERNIONS
 *
 * References:    SIGGRAPH 93, course 60 (Ken Shoemake)
 *                Graphic Gems, Vol 1,2,3,4
 *
 */

/*@{ @name QUATERNION functions.
*/

  /*! Sets quaternion self to have components w, x, y, z 
      \remark Note the unusual order or parameters. */

	void  quat_set	(QUATERNION self,
			     LIBGEOM_REAL w, 
			     LIBGEOM_REAL x, 
			     LIBGEOM_REAL y, 
			     LIBGEOM_REAL z);  

  /*! Sets self to the zero quaternion. */

	void  quat_set_null  (QUATERNION self);	

  /*! Sets self to the unit (identity) quaternion */

	void  quat_set_unit  (QUATERNION self);	

  /*! self = newQuat */

	void  quat_assign    (QUATERNION self, 
			      QUATERNION newQuat);	

  /*! dest = src */

	void quat_copy (const QUATERNION src,
			QUATERNION dest);

  /*! Sets inv to the inverse of self

	\remark The product of a quaternion by its inverse is the unit quaternion.
  */

	void  quat_get_inv	(const QUATERNION self, 
				 QUATERNION inv); 

  /*! Sets conj to be the conjugate of self.

	\remark The conjugate of a unit quaternion represents the opposite rotation (i.e.
                it is equivalent to a transpose in matrix format).
  */

	void  quat_get_conj   (const QUATERNION self, 
			       QUATERNION conj); 

  /*! Returns the norm of quaternion self */

	LIBGEOM_REAL quat_get_norm  (const QUATERNION self);	

  /*! Returns the distance between left and right */

	LIBGEOM_REAL quat_get_distance (const QUATERNION left,
				 const QUATERNION right); 

  /*! Returns the shortest distance between left and right */

	LIBGEOM_REAL quat_get_sdistance (const QUATERNION left,
				  const QUATERNION right);

  /*! Negates a quaternion (i.e. negates all its components), to optain its opposite.

	\warning A unit quaternion and its opposite represent the same orientation.
  */

	void  quat_neg       (QUATERNION self);	

  /*! Normalizes a quaternion (self) */

	void  quat_normalize (QUATERNION self); 

  /*! Returns the dot product of two quaternions: left * right */

	LIBGEOM_REAL quat_dot       (const QUATERNION left,
			      const QUATERNION right);

  /*! Raise a normalized quaternion to the power alpha */

	void  quat_pow       (const QUATERNION self, 
			      LIBGEOM_REAL alpha, 
			      QUATERNION result); 

  /*! Exponentiate quaternion, assuming LIBGEOM_REAL part 0 (w==0) */

	void  quat_exp       (const QUATERNION self, 
			      QUATERNION result); 

  /*! Take the natural logarithm of a normalized quaternion */

	void  quat_ln        (const QUATERNION self, 
			      QUATERNION result); 

  /*! Quaternion scaling: result = self * s */

	void  quat_mults      (const QUATERNION self, 
			       LIBGEOM_REAL s, 
			       QUATERNION result); 

  /*! Quaternion multiplication : result = left * right

	\remark This operation is not commutative : left * right != right * left
  */

	void  quat_mult		(const QUATERNION left, 
				 const QUATERNION right, 
				 QUATERNION result);

  /*! Quaternion substraction : result = left - right */

	void  quat_sub        (const QUATERNION left, 
			       const QUATERNION right, 
			       QUATERNION result);

  /*! Quaternion addition : result = left + right */

	void  quat_add        (const QUATERNION left, 
			       const QUATERNION right, 
			       QUATERNION result);

  /*! Spherical linear interpolation between left and right. 
     As u goes from 0 to 1, result goes from left to right. */

	void  quat_slerp      (const QUATERNION left,
			       const QUATERNION right,
			       LIBGEOM_REAL u, 
			       QUATERNION result); 				

  /*! Writes the quaternion to stdout */

	void  quat_dump      (const QUATERNION self);	

  /*! given an angle (radians) and an axis:
     self = (cos(angle/2),sin(angle/2) * axis) */ 

	void  quat_from_axis  (QUATERNION self,
			       const VT3D axis,
			       LIBGEOM_REAL angle);

  /*! Transforms a given axis-angle vector into an equivalent unit quaternion.
  */

	void  quat_from_axis_angle (QUATERNION q,
        	                    const VT3D       axis_angle);

  /*! Transforms a given unit quaternion into an equivalent axis-angle vector,
      whose norm (angle) is included in the interval [0, PI]. */

	void  quat_to_axis_angle (const QUATERNION q,
        	                  VT3D       axis_angle);

  /*! Transforms a rotation matrix (4x4) into a unit quaternion.
  */

	void  quat_from_matrix (const MatriX m, 
				QUATERNION result);

  /*! Transforms a unit quaternion into a 4x4 rotation matrix.
  */

	void  quat_to_matrix   (const QUATERNION self, 
				MatriX result); 

  /*! Transforms a rotation matrix (3x3) into a unit quaternion.
  */

	void  quat_from_mr3d   (const MR3D   m, 
				QUATERNION result);

  /*! Transforms a unit quaternion into a 3x3 rotation matrix.
  */

	void  quat_to_mr3d     (const QUATERNION self, 
				MR3D   result); 

		
  /*! Computes the unit quaternion that represents the direct
      rotation that transforms a given vector v1 into another given vector v2.
  */

	void quat_from_two_vectors (const VT3D       v1,
 	                            const VT3D       v2,
        	                    QUATERNION q);

  /*! Converts a swing parametrization into a unit quaternion.

	\remark The swing parametrization is described in Grassia's paper (JGT 98).
  */

	void quat_from_swing (QUATERNION q,
			      const LIBGEOM_REAL      swing [2]);


  /*! Converts a (swing, twist) parametrization into a unit quaternion.

	\remark The swing-twist parametrization is described in Grassia's paper (JGT 98).
  */

	void quat_from_swing_and_twist (QUATERNION q,
	                                const LIBGEOM_REAL      swing [2],
	                                LIBGEOM_REAL      twist);

  /*! Converts a unit quaternion into a (swing, twist) parametrization.

      \arg The norm of the swing component is bounded by PI.
      \arg The twist component is bounded by -2*PI and +2*PI.

	\warning The swing-twist parametrization possesses a singularity
                 located at a rotation angle of +- PI, due to the swing
                 component.
  */

	void quat_to_swing_and_twist (const QUATERNION q,
	                              LIBGEOM_REAL      swing [2],
	                              LIBGEOM_REAL      *twist);

/*@}*/

/*--------------------------------------------------------------------*/
/* --------------------- array1d functions -------------------------- */

/*@{ @name ARRAY1D functions.

  \brief This module is built on top of vtn_ functions, and replaces them.
*/

  /*! Creates an ARRAY1D object describing an 1D array of doubles, with a given
    number of lines, and returns a pointer to it.
  */

	ARRAY1D *array1d_creation (int lines);

  /*! Destroys an ARRAY1D object, thus freeing its memory.
  */

	void array1d_destroy (ARRAY1D *array1d);

  /*! Initializes each entry of an ARRAY1D to zero.
  */

	void array1d_nul (ARRAY1D *array1d);

  /*! Returns the Euclidean norm of the given vector.
  */

	double array1d_norm (ARRAY1D *array1d);

  /*! Returns the sum of the absolute values of each entry of the given vector.
  */

	double array1d_anorm (ARRAY1D *array1d);

  /*! Copies the contennt of the "src" array into the "dest" array.
  */

	void array1d_copy (ARRAY1D *src,
	                   ARRAY1D *dest);

  /*! Vector addition: dest = src_left + src_right
  */

	void array1d_add (ARRAY1D *src_left,
	                  ARRAY1D *src_right,
	                  ARRAY1D *dest);

  /*! Vector substraction: dest = src_left - src_right
  */

	void array1d_sub (ARRAY1D *src_left,
	                  ARRAY1D *src_right,
	                  ARRAY1D *dest);

  /*! Vector scaling by a given factor: dest = factor * src
  */

	void array1d_mults (ARRAY1D *src,
	                    ARRAY1D *dest,
	                    double  factor);

  /*! Returns the dot product of vectors src1 and src2, i.e. src1 * src2
  */

	double array1d_dot (ARRAY1D *src1,
	                    ARRAY1D *src2);

  /*! Computes an array "dest" from a given array "src": dest_i = src_i / norm2,
      where norm2 is the squared norm of src1.
   */

	double array1d_inv (ARRAY1D *src,
	                    ARRAY1D *dest);

  /*! Writes the content of an ARRAY1D object to a given file.
  */

	void array1d_write (ARRAY1D *a,
	                    FILE    *file);

/*@}*/

/*--------------------------------------------------------------------*/
/* ----------------------- array2d functions ------------------------ */

/*@{ @name ARRAY2D functions.

	\brief This module is built on top of matmxn_ functions, and replaces them.

*/

  /*! Creates an ARRAY2D object with a given number of lines and columns,
      and returns a pointer to it.
  */

	ARRAY2D *array2d_creation (int lines,
	                           int cols);

  /*! Desrtoys an ARRAY2D object, and thus frees its memory.
  */

	void array2d_destroy (ARRAY2D *array2d);

  /*! Returns TRUE if the two given ARRAY2D objects have same size, i.e.
      same number of rows (lines) and same number of columns.
  */

	int  array2d_same_size (ARRAY2D *a,
	                        ARRAY2D *b);

  /*! Initializes each element of the given ARRAY2D object to zero.
  */

	void array2d_nul (ARRAY2D *array2d);

  /*! Initializes each (i,j)-th element of the given ARRAY2D according to the
      following rule: the element is set to one if i == j, and to zero otherwise.

	\remark If the ARRAY2D object is square, this is equivalent to an
              initialization with the identity matrix.
  */

	void array2d_idt (ARRAY2D *array2d);

  /*! Copies the content of the "src" ARRAY2D object into the "dest" ARRAY2D object.
  */

	void array2d_copy (ARRAY2D *src,
	                   ARRAY2D *dest);

  /*! Matrix multiplication: dest = a * b

	\remark The size of the three matrices must be compatible, otherwise a fatal
            error will occur. The compatibility conditions are as follows:

	\arg The number of columns of "a" must equal the number of rows of "b"
	\arg The number of rows of "a" must equal the number of rows of "dest"
	\arg The number of columns of "b" must equal the number of columns of "dest"
  */

	void array2d_mult (ARRAY2D *a,
	                   ARRAY2D *b,
	                   ARRAY2D *dest);

  /*! Matrix addition: dest = a + b

	\remark The size of the three matrices must be the same, otherwis a fatal
            error will occur.
  */

	void array2d_add (ARRAY2D *a,
	                  ARRAY2D *b,
	                  ARRAY2D *dest);

  /*! Matrix substraction: dest = a - b

	\remark The size of the three matrices must be the same, otherwise a fatal
            error will occur.
  */

	void array2d_sub (ARRAY2D *a,
	                  ARRAY2D *b,
	                  ARRAY2D *dest);

  /*! Multiplication of a matrix by a vector : dest = array2d * array1d

	\remark The size of the objects must be compatible, otherwise a fatal
                error will occur.
  */

	void array2d_mult_array1d (ARRAY2D *array2d,
	                           ARRAY1D *array1d,
	                           ARRAY1D *dest);

  /*! Multiplies each element of the src array by a given scalar,
      and stores the result in the "dest" array.
  */

	void array2d_scale (ARRAY2D *src,
	                    float   scalar,
	                    ARRAY2D *dest);


  /*! Computes the pseudo-inverse of a given "src" ARRAY2D.

	\remark The size of the "dest" matrix must be equal to the size of
            the transpose of the "src" matrix, otherwise a fatal error will occur.

	\remark This function is based on Greville's recursive algorithm.
  */

	void array2d_pseudo_inverse (ARRAY2D *src,
	                             ARRAY2D *dest);

  /*! Computes the damped pseudo-inverse of a given "j_mat" ARRAY2D,
      with a given damping factor. The result is stored in the "j_mat" ARRAY2D.
  */

	void array2d_damped_least_square_inverse (ARRAY2D *j_inv,    /*!< output matrix */
	                                          ARRAY2D *j_mat,    /*!< input matrix */
	      	                                  double  damping);

  /*! \warning Obsolete.
  */

void array2d_compute_LS_and_DLS_inverse (ARRAY2D *LS_inv,   /* output LS-inverse  */
                                         ARRAY2D *DLS_inv,  /* output DLS-inverse */
                                         ARRAY2D *j_mat,    /* input matrix */
                                         ARRAY1D *task,
                                         double  damping);

  /*! Writes the content of a given ARARY2D to a file.
  */

	void array2d_write (ARRAY2D *a,
	                    FILE    *file);

  /*! Initializes all elements of a column of an ARRAY2D object with a given value.
  */

	void array2d_set_column (ARRAY2D *array2d,
	                         int     column,
	                         double  value);

  /*! Initializes all elements of a row (line) of an ARRAY2D object with a given value.
  */

	void array2d_set_line (ARRAY2D *array2d,
	                       int     line,
	                       double  value);

  /*! Computes the transpose of a given "src" ARRAY2D object, and stores it
      into the "dest" object.

	\remark The size of "src" object must be equal to the size of the transpose
             of the "dest" object, otherwise a fatal error will occur.
  */

	void array2d_transpose (ARRAY2D *src,
	                        ARRAY2D *dest);

/*@}*/

/*--------------------------------------------------------------------*/
/* ----------------------- polynomial functions --------------------- */

/*@{ @name A set of functions to solve polynomials of 2nd, 3rd and 4th degree.

	\remark The user must store the coefficients of the polynomial in the c [] array.
                The c[i] coefficient is the coefficient related to the i-th power of the unknown variable.
	\remark All functions store the roots in the s [] array and return the # of stored roots.
*/

	int polynomial_solve_quadric (double c[3], double s[2]);

	int polynomial_solve_cubic   (double c[4], double s[3]);

	int polynomial_solve_quartic (double c[5], double s[4]);

/*@}*/

}
#endif
