/***************************************************************************
 *   Copyright (C) 2002 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

/////////////////////////////////////////////////////////////////////
///
///  PROJECT.....: CO-ME
///  RESPONSIBLE.: Anderson Maciel
///
///  AUTHOR......: Fabiana Benedetti
///  DATE........: October/2002
///  DESCRIPTION.: Class definition.
///
////////////////////////////////////////////////////////////////////

#include	<physics/comecollide.h>
#include	<algebra/comemesh.h>
#include	<algebra/comevertex3d.h>
#include	<bio/comebiostructure.h>
#include	<bio/comemoleculescartilage.h>
#include	<bio/comemolecule.h>
#include	<general/comescenario.h>
#include	<RAPID/RAPID.H>
#include	<VCollide.H>

#include	<time.h>
#include	<math.h>

//////////////////////////////////////////////////////////////////////
/// Construction/Destruction
//////////////////////////////////////////////////////////////////////
COME_Collide :: COME_Collide( COME_Scenario* scene ){

	parent = scene;
	collisions1stLevel = ENVELOPE;
	if( COME::flagCollisionTreatment == NEIGHBORS ){
		initializeProximityStructure();
	}
	softhard = SOFT;
	CONTADIST = 0;
}


COME_Collide :: ~COME_Collide(){

	collisions.clear();
	
}

//////////////////////////////////////////////////////////////////////
/// Setting
//////////////////////////////////////////////////////////////////////

void
COME_Collide :: setCollisions1stLevel( int detectionType ) {

	collisions1stLevel = detectionType;
}

//////////////////////////////////////////////////////////////////////
/// Getting
//////////////////////////////////////////////////////////////////////

int		
COME_Collide :: getNumberOfCollisions() const{

	return collisions.size();

}

vector<COME_Collision>	
COME_Collide :: getCollisions() const{

	return collisions;

}

int		
COME_Collide :: getCollisions1stLevel() const{

	return collisions1stLevel;

}


//////////////////////////////////////////////////////////////////////
/// Delete the collisions vector.
//////////////////////////////////////////////////////////////////////
void	
COME_Collide :: resetCollisions(){

	collisions.clear();

}



//////////////////////////////////////////////////////////////////////
/// This method is an alternative to V-Collide/RAPID. It works only for
/// molecules-made objects, but is much faster than the approach of
/// detectContacts method.
//////////////////////////////////////////////////////////////////////
void		
COME_Collide :: treatCollisions() {


	COME_Patient *activePatient = ((COME_Scenario*)parent)->getPatient(((COME_Scenario*)parent)->getPatientList()->size()-1);
	// If there's only one object, doesn't detect collisions
	if( activePatient->getPtOrganList()->size() > 1 ){

		// Take the organs of the first patient (there is only one patientin this phase)
		list<COME_MoleculesCartilage*>::const_iterator iterOrgans1;
		for ( iterOrgans1 =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans1 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans1++  ){
			
			COME_Point3D mins1, maxs1;
			(*iterOrgans1)->getEnvelop( mins1, maxs1 );
			
			list<COME_MoleculesCartilage*>::const_iterator iterOrgans2;
			for ( iterOrgans1++, iterOrgans2 = iterOrgans1, iterOrgans1--; iterOrgans2 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans2++  ){
				
				if( (*iterOrgans1) != (*iterOrgans2) ){
					COME_Point3D mins2, maxs2;
					(*iterOrgans2)->getEnvelop( mins2, maxs2 );

					if( ( ( mins1.getX() > maxs2.getX() ) || ( maxs1.getX() < mins2.getX() ) ) ||
						( ( mins1.getY() > maxs2.getY() ) || ( maxs1.getY() < mins2.getY() ) ) ||
						( ( mins1.getZ() > maxs2.getZ() ) || ( maxs1.getZ() < mins2.getZ() ) ) ){

						// do not collide
					} else {

						// test spheres
						list<COME_Molecule*>::const_iterator iterMolec1;
						for ( iterMolec1 = (*iterOrgans1)->getTissue()->getShape()->begin(); iterMolec1 != (*iterOrgans1)->getTissue()->getShape()->end(); iterMolec1++  ){

							list<COME_Molecule*>::const_iterator iterMolec2;
							for ( iterMolec2 = (*iterOrgans2)->getTissue()->getShape()->begin(); iterMolec2 != (*iterOrgans2)->getTissue()->getShape()->end(); iterMolec2++  ){

								double actualDistance = (*iterMolec1)->getGlobalPosition().vpDistance( (*iterMolec2)->getGlobalPosition() );
								double allowedDistance = ( (*iterMolec1)->getRadius() + (*iterMolec2)->getRadius() );
								if( actualDistance < allowedDistance ){
									// there is collision
									
									double penetrationDistance = allowedDistance - actualDistance;

									double springConstant = ( ( (*iterMolec1)->getMaterial()->getYoungsModulus() + (*iterMolec2)->getMaterial()->getYoungsModulus() ) / 2.0 ) * allowedDistance;

									COME_Vector3D forceDirection; forceDirection.setVector3D( (*iterMolec1)->getGlobalPosition() - (*iterMolec2)->getGlobalPosition() );
									forceDirection.vpNormalize();

									(*iterMolec1)->addCollisionForce( ( forceDirection * penetrationDistance ) * springConstant );
									(*iterMolec2)->addCollisionForce( ( forceDirection * penetrationDistance ) * -springConstant );

								}
							}
						}
					}
				}
			}
		}
	}
}


//////////////////////////////////////////////////////////////////////
/// This method detect the collision between objects
//////////////////////////////////////////////////////////////////////
void		
COME_Collide :: detectContacts() {


	resetCollisions();
	
	COME_Patient *activePatient = ((COME_Scenario*)parent)->getPatient(((COME_Scenario*)parent)->getPatientList()->size()-1);
	
	VCollide	vc;  //V-Collide collision detection library
	int*		objects = new int[((COME_Scenario*) parent)->getPatient(0)->getPtOrganList()->size()];
	int			iPosObject = 0;

	// If there's only one object, doesn't detect collisions
	if( activePatient->getPtOrganList()->size() > 2 ){

		
		if( collisions1stLevel == ENVELOPE ){
			// Test envelope.
			// Take the organs of the first patient (there is only one patientin this phase)
			list<COME_MoleculesCartilage*>::const_iterator iterOrgans1;
			for ( iterOrgans1 =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans1 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans1++  ){
				
				COME_Point3D mins1, maxs1;
				(*iterOrgans1)->getEnvelop( mins1, maxs1 );
				
				list<COME_MoleculesCartilage*>::const_iterator iterOrgans2;
				for ( iterOrgans1++, iterOrgans2 = iterOrgans1, iterOrgans1--; iterOrgans2 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans2++  ){
					
					/// Don't test collisions between two rigit objects
					if( ( (*iterOrgans1)->isFixed() ) && ( (*iterOrgans2)->isFixed() ) ) continue;
					if( (*iterOrgans1) != (*iterOrgans2) ){
						COME_Point3D mins2, maxs2;
						(*iterOrgans2)->getEnvelop( mins2, maxs2 );

						if( ( ( mins1.getX() > maxs2.getX() ) || ( maxs1.getX() < mins2.getX() ) ) ||
							( ( mins1.getY() > maxs2.getY() ) || ( maxs1.getY() < mins2.getY() ) ) ||
							( ( mins1.getZ() > maxs2.getZ() ) || ( maxs1.getZ() < mins2.getZ() ) ) ){

							// do not collide
						} else {

							//call RAPID collision detection routine
							if (COME::flagCollisionTreatment == HYBRID ){
								detectContactsHybrid( (*iterOrgans1), (*iterOrgans2) );
							} else {
								detectContacts( (*iterOrgans1), (*iterOrgans2) );
							}
						}
					}
				}
			}
		} else { // verify collision pairs by V-Collide
		
			// Take the organs of the first patient (there is only one patientin this phase)
			list<COME_BioStructure*>::const_iterator iterOrgans;

			for ( iterOrgans =((list<COME_BioStructure*>*)activePatient->getPtOrganList())->begin(); iterOrgans != ((list<COME_BioStructure*>*)activePatient->getPtOrganList())->end(); iterOrgans++  ){
				
				//(*iterOrgans)->updateSurface(DEFORM); ////
				COME_Mesh *mesh = (*iterOrgans)->getSurface();

				//add the objects to the library				
				vc.NewObject( &( objects[ iPosObject ] ) );
				
				//add triangles
				for ( int face = 0; face < mesh->getNumberFaces(); face++ ){
							
					//read the object of the Mesh Class 
					vector<COME_Vertex3D> vertices;
					vertices = mesh->getVerticesFace( face );

					double v1[ 3 ], v2[ 3 ], v3[ 3 ];
					v1[ 0 ] = vertices[ 0 ].getX();
					v1[ 1 ] = vertices[ 0 ].getY();
					v1[ 2 ] = vertices[ 0 ].getZ();

					v2[ 0 ] = vertices[ 1 ].getX();
					v2[ 1 ] = vertices[ 1 ].getY();
					v2[ 2 ] = vertices[ 1 ].getZ();

					v3[ 0 ] = vertices[ 2 ].getX();
					v3[ 1 ] = vertices[ 2 ].getY();
					v3[ 2 ] = vertices[ 2 ].getZ();

					vc.AddTri( v1, v2, v3 );
				}
      
    			vc.EndObject();           
				iPosObject++;
			}
			
		/*	//transformation matrices.
			double R1[4][4]; 
			R1[0][0] = R1[1][1] = R1[2][2] = R1[3][3] = 1.0;
			R1[0][1] = R1[1][0] = R1[2][0] = R1[3][0] = 0.0;
			R1[0][2] = R1[1][2] = R1[2][1] = R1[3][1] = 0.0;
			R1[0][3] = R1[1][3] = R1[2][3] = R1[3][2] = 0.0;
			vc.UpdateTrans( objects[ 0 ], R1 );    
			vc.UpdateTrans( objects[ 1 ], R1 );    */

  			vc.Collide();  //perform collision test
      
			//report the results.
			int VCReportSize = iPosObject;  //size of report
			VCReportType *vcReport = new VCReportType[ VCReportSize ];
      
			int numberOfCollidingPairs = vc.Report( VCReportSize, vcReport );
      
			if ( numberOfCollidingPairs > VCReportSize ){
				  
				VCReportSize = numberOfCollidingPairs;
				delete vcReport;
				vcReport = new VCReportType[ VCReportSize ];
				numberOfCollidingPairs = vc.Report( VCReportSize, vcReport );
			}
      
			for ( int j = 0; j < numberOfCollidingPairs; j++ ){

				printf( "Detected collision between objects %d and %d \n", vcReport[ j ].id1, vcReport[ j ].id2 );

				//call RAPID collision detection routine
				detectContacts( activePatient->getOrgan( vcReport[ j ].id1 ), activePatient->getOrgan( vcReport[ j ].id2 ) );
			}
		}
	} else if( activePatient->getPtOrganList()->size() == 2 ){

			detectContacts( activePatient->getOrgan( 0 ), activePatient->getOrgan( 1 ) );
			//printf( "collision 2 obj" );
	}

}
	 
//////////////////////////////////////////////////////////////////////
/// This method detect the collision between objects
//////////////////////////////////////////////////////////////////////
void		
COME_Collide :: detectContacts( COME_BioStructure* organ1, COME_BioStructure* organ2 ) {  

	RAPID_model *object1	= new RAPID_model;
	RAPID_model *object2	= new RAPID_model;
	
	double cCollided[] = {1.0, 0.4, 0.4, 1.0};
	double cNOCollided[] = {0.4, 1.0, 1.0, 1.0};
				
	int face;
	
	// Take the organs of the first patient (there is only one patientin this phase)

	COME_Mesh *meshOrgan1 = organ1->getSurface();

	int iId = 0;
		
	//add the first organ to the library				
	object1->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan1->getNumberFaces(); face++ ){
						
		//read each face of the Mesh Class 
		vector<COME_Vertex3D> vertices;
		vertices = meshOrgan1->getVerticesFaceGlobalPosition( face );

		double v1[ 3 ], v2[ 3 ], v3[ 3 ];
		v1[ 0 ] = vertices[ 0 ].getX();
		v1[ 1 ] = vertices[ 0 ].getY();
		v1[ 2 ] = vertices[ 0 ].getZ();

		v2[ 0 ] = vertices[ 1 ].getX();
		v2[ 1 ] = vertices[ 1 ].getY();
		v2[ 2 ] = vertices[ 1 ].getZ();

		v3[ 0 ] = vertices[ 2 ].getX();
		v3[ 1 ] = vertices[ 2 ].getY();
		v3[ 2 ] = vertices[ 2 ].getZ();

		object1->AddTri( v1, v2, v3, iId );
		
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOCollided );
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOCollided );
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOCollided );
		
		iId++;
	}
		
     
	object1->EndModel();
    		
	COME_Mesh *meshOrgan2 = organ2->getSurface();
	iId		= 0;
	
	object2->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan2->getNumberFaces(); face++ ){
						
		//read the object of the Mesh Class 
		vector<COME_Vertex3D> vertices;
		vertices = meshOrgan2->getVerticesFaceGlobalPosition( face );

		double v1[ 3 ], v2[ 3 ], v3[ 3 ];
		v1[ 0 ] = vertices[ 0 ].getX();
		v1[ 1 ] = vertices[ 0 ].getY();
		v1[ 2 ] = vertices[ 0 ].getZ();

		v2[ 0 ] = vertices[ 1 ].getX();
		v2[ 1 ] = vertices[ 1 ].getY();
		v2[ 2 ] = vertices[ 1 ].getZ();

		v3[ 0 ] = vertices[ 2 ].getX();
		v3[ 1 ] = vertices[ 2 ].getY();
		v3[ 2 ] = vertices[ 2 ].getZ();

		object2->AddTri( v1, v2, v3, iId );
		
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOCollided );
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOCollided );
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOCollided );
		
		iId++;
		
	}
      
	object2->EndModel();          
	
	
	//set up the rotation and translation matrices as
	//required by RAPID for the objects.
	double R[3][3], T[3];

	R[0][0] = R[1][1] = R[2][2] = 1.0;
	R[0][1] = R[1][0] = R[2][0] = 0.0;
	R[0][2] = R[1][2] = R[2][1] = 0.0;

	T[0] = 0.0; T[1] = 0.0; T[2] = 0.0;

	//call the RAPID collision detection routine.
	RAPID_Collide( R, T, object1, R, T, object2, RAPID_ALL_CONTACTS );
      		
	//report the results of collision detection.
	//printf ( "Num contact pairs %d\n", RAPID_num_contacts  );
	for ( int i = 0; i < RAPID_num_contacts; i++ ){

		//printf( "\t contact %4d: tri %4d and %4d\n ", i, RAPID_contact[i].id1, RAPID_contact[i].id2 );
		COME_Collision collision = COME_Collision( organ1, RAPID_contact[ i ].id1, organ2, RAPID_contact[ i ].id2 );
	
		collisions.push_back( collision );  
		meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(0)->setColor( cCollided );
		meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(1)->setColor( cCollided );
		meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(2)->setColor( cCollided );
		meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(0)->setColor( cCollided );
		meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(1)->setColor( cCollided );
		meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(2)->setColor( cCollided );

	}

	delete object1;
	delete object2;
}

//////////////////////////////////////////////////////////////////////
/// This method detect the collision between objects
/// It first check the spheres colliding then add the triangles
/// anchored to those spheres to RAPID objects.
//////////////////////////////////////////////////////////////////////
void		
COME_Collide :: detectContactsHybrid( COME_BioStructure* organ1, COME_BioStructure* organ2 ) {  

	RAPID_model *object1	= new RAPID_model;
	RAPID_model *object2	= new RAPID_model;
	
	//int	objects[2];
	int face;
	
	// test spheres (anchors)
	list<COME_Molecule*>::const_iterator iterMolec1;
	for ( iterMolec1 = organ1->getTissue()->getShape()->begin(); iterMolec1 != organ1->getTissue()->getShape()->end(); iterMolec1++  ){

		list<COME_Molecule*>::const_iterator iterMolec2;
		for ( iterMolec2 = organ2->getTissue()->getShape()->begin(); iterMolec2 != organ2->getTissue()->getShape()->end(); iterMolec2++  ){

			double actualDistance = (*iterMolec1)->getGlobalPosition().vpDistance( (*iterMolec2)->getGlobalPosition() );
			double allowedDistance = ( (*iterMolec1)->getRadius() + (*iterMolec2)->getRadius() );
			if( actualDistance < allowedDistance ){
				// there is collision
				(*iterMolec1)->setCollide( true );
				(*iterMolec2)->setCollide( true );
			}
		}
	}
	
	// Take the organs of the first patient (there is only one patientin this phase)

	COME_Mesh *meshOrgan1 = organ1->getSurface();

	int iId = 0;
		
	//add the first organ to the library				
	object1->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan1->getNumberFaces(); face++ ){
						
		//read each face of the Mesh Class 
		vector<COME_Vertex3D> vertices;
		vertices = meshOrgan1->getVerticesFaceGlobalPosition( face );

		if( ( vertices[0].anchorsCollide() ) ||
		    ( vertices[1].anchorsCollide() ) ||
		    ( vertices[2].anchorsCollide() ) ){
			double v1[ 3 ], v2[ 3 ], v3[ 3 ];
			v1[ 0 ] = vertices[ 0 ].getX();
			v1[ 1 ] = vertices[ 0 ].getY();
			v1[ 2 ] = vertices[ 0 ].getZ();
	
			v2[ 0 ] = vertices[ 1 ].getX();
			v2[ 1 ] = vertices[ 1 ].getY();
			v2[ 2 ] = vertices[ 1 ].getZ();
	
			v3[ 0 ] = vertices[ 2 ].getX();
			v3[ 1 ] = vertices[ 2 ].getY();
			v3[ 2 ] = vertices[ 2 ].getZ();
	
			object1->AddTri( v1, v2, v3, iId );
			iId++;
		}
	}
		
     
	object1->EndModel();
    		
	COME_Mesh *meshOrgan2 = organ2->getSurface();
	iId		= 0;
	
	object2->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan2->getNumberFaces(); face++ ){
						
		//read the object of the Mesh Class 
		vector<COME_Vertex3D> vertices;
		vertices = meshOrgan2->getVerticesFaceGlobalPosition( face );

		if( ( vertices[0].anchorsCollide() ) ||
		    ( vertices[1].anchorsCollide() ) ||
		    ( vertices[2].anchorsCollide() ) ){
			double v1[ 3 ], v2[ 3 ], v3[ 3 ];
			v1[ 0 ] = vertices[ 0 ].getX();
			v1[ 1 ] = vertices[ 0 ].getY();
			v1[ 2 ] = vertices[ 0 ].getZ();
	
			v2[ 0 ] = vertices[ 1 ].getX();
			v2[ 1 ] = vertices[ 1 ].getY();
			v2[ 2 ] = vertices[ 1 ].getZ();
	
			v3[ 0 ] = vertices[ 2 ].getX();
			v3[ 1 ] = vertices[ 2 ].getY();
			v3[ 2 ] = vertices[ 2 ].getZ();
	
			object2->AddTri( v1, v2, v3, iId );
			iId++;
		}
		
	}
      
	object2->EndModel();          
	
	
	//set up the rotation and translation matrices as
	//required by RAPID for the objects.
	double R[3][3], T[3];

	R[0][0] = R[1][1] = R[2][2] = 1.0;
	R[0][1] = R[1][0] = R[2][0] = 0.0;
	R[0][2] = R[1][2] = R[2][1] = 0.0;

	T[0] = 0.0; T[1] = 0.0; T[2] = 0.0;

	//call the RAPID collision detection routine.
	RAPID_Collide( R, T, object1, R, T, object2, RAPID_ALL_CONTACTS );
      
	
	//report the results of collision detection.
	//printf ( "Num contact pairs %d\n", RAPID_num_contacts  );
	for ( int i = 0; i < RAPID_num_contacts; i++ ){

		//printf( "\t contact %4d: tri %4d and %4d\n ", i, RAPID_contact[i].id1, RAPID_contact[i].id2 );
		COME_Collision collision = COME_Collision( organ1, RAPID_contact[ i ].id1, organ2, RAPID_contact[ i ].id2 );
	
		collisions.push_back( collision );  

	}
	
	// reset spheres (anchors)
	list<COME_Molecule*>::const_iterator iterMolec;
	for ( iterMolec = organ1->getTissue()->getShape()->begin(); iterMolec != organ1->getTissue()->getShape()->end(); iterMolec++  ){
		(*iterMolec)->setCollide( false );
	}
	for ( iterMolec = organ2->getTissue()->getShape()->begin(); iterMolec != organ2->getTissue()->getShape()->end(); iterMolec++  ){
		(*iterMolec)->setCollide( false );
	}

	delete object1;
	delete object2;
}


//////////////////////////////////////////////////////////////////////
/// This method detect the collision between objects
//////////////////////////////////////////////////////////////////////
bool		
COME_Collide :: detectContactsDisplacement() {

//printf( "Entrou detectContactsDisp01 \n" );

	bool collided = false;
	COME_Patient *activePatient = ((COME_Scenario*)parent)->getPatient(((COME_Scenario*)parent)->getPatientList()->size()-1);
	
	
	// reset Collision Vectors
	list<COME_MoleculesCartilage*>::const_iterator iterOrgansReset;
	for ( iterOrgansReset =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgansReset != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgansReset++  ){
		(*iterOrgansReset)->getSurface()->resetCollisionDisps();
	}
	
	// If there's only one object, doesn't detect collisions
	if( activePatient->getPtOrganList()->size() > 2 ){

		
		//if( collisions1stLevel == ENVELOPE ){
			// Test envelope.
			// Take the organs of the first patient (there is only one patientin this phase)
			list<COME_MoleculesCartilage*>::const_iterator iterOrgans1;
			for ( iterOrgans1 =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans1 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans1++  ){
				
				//COME_Point3D mins1, maxs1;
				//(*iterOrgans1)->getEnvelop( mins1, maxs1 );
				
				list<COME_MoleculesCartilage*>::const_iterator iterOrgans2;
				for ( iterOrgans1++, iterOrgans2 = iterOrgans1, iterOrgans1--; iterOrgans2 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans2++  ){
					
					/// Don't test collisions between two rigit objects
					if( ( (*iterOrgans1)->isFixed() ) && ( (*iterOrgans2)->isFixed() ) ) continue;
					if( (*iterOrgans1) != (*iterOrgans2) ){
						//COME_Point3D mins2, maxs2;
						//(*iterOrgans2)->getEnvelop( mins2, maxs2 );

						/*if( ( ( mins1.getX() > maxs2.getX() ) || ( maxs1.getX() < mins2.getX() ) ) ||
							( ( mins1.getY() > maxs2.getY() ) || ( maxs1.getY() < mins2.getY() ) ) ||
							( ( mins1.getZ() > maxs2.getZ() ) || ( maxs1.getZ() < mins2.getZ() ) ) ){

							// do not collide
						} else {*/
							if(
							( ( (*iterOrgans1)->getDescription() == "labrum" ) || ( (*iterOrgans2)->getDescription() == "labrum" ) )
							||
							( ( ( (*iterOrgans1)->getDescription() == "ischiofemoral" ) && ( (*iterOrgans2)->getDescription() == "labrum" ) )
							|| ( ( (*iterOrgans2)->getDescription() == "ischiofemoral" ) && ( (*iterOrgans1)->getDescription() == "labrum" ) ) )
							||
							( ( ( (*iterOrgans1)->getDescription() == "ischiofemoral" ) && ( (*iterOrgans2)->getDescription() == "pelvisbone" ) )
							|| ( ( (*iterOrgans2)->getDescription() == "ischiofemoral" ) && ( (*iterOrgans1)->getDescription() == "pelvisbone" ) ) )
							){
							// do nothing
							} else {
								if( detectContactsDisplacement( (*iterOrgans1), (*iterOrgans2) ) )
									collided = true;
							}
						//}
					}
				}
			}
		//}
	} else if( activePatient->getPtOrganList()->size() == 2 ){

		if( detectContactsDisplacement( activePatient->getOrgan( 0 ), activePatient->getOrgan( 1 ) ) )
			collided = true;
	}
	
	return collided;

}

//////////////////////////////////////////////////////////////////////
/// This method tests if some point of one organ is inside the other organ
/// Then set for each penetrating vertex a displacement necessary to eliminate the penetration
//////////////////////////////////////////////////////////////////////
bool
COME_Collide :: detectContactsDisplacement( COME_BioStructure* organ1, COME_BioStructure* organ2 ) {  

//printf( "Entrou detectContactsDisp02 \n" );
	
	RAPID_model *object1	= new RAPID_model;
	RAPID_model *object2	= new RAPID_model;
	
	double cCollided[] = {1.0, 0.4, 0.4, 1.0};
	double cCollidedTotal[] = {0.1, 0.1, 1.0, 1.0};
	double cNOCollided[] = {0.4, 1.0, 1.0, 1.0};
	double cNOTenabled[] = {1.0, 1.0, 1.0, 1.0};
	
	//int	objects[2];
	bool collided = false;
	int face;
	
	// Take the organs of the first patient (there is only one patientin this phase)

	COME_Mesh *meshOrgan1 = organ1->getSurface();

	int iId = 0;
		
	//add the first organ to the library				
	object1->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan1->getNumberFaces(); face++ ){
						
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOTenabled );
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOTenabled );
		meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOTenabled );
			
		if( meshOrgan1->getAFacePt(face)->getCollisionDetectable() ){
			//read each face of the Mesh Class 
			vector<COME_Vertex3D> vertices;
			vertices = meshOrgan1->getVerticesFaceGlobalPosition( face );
	
			double v1[ 3 ], v2[ 3 ], v3[ 3 ];
			v1[ 0 ] = vertices[ 0 ].getX();
			v1[ 1 ] = vertices[ 0 ].getY();
			v1[ 2 ] = vertices[ 0 ].getZ();
	
			v2[ 0 ] = vertices[ 1 ].getX();
			v2[ 1 ] = vertices[ 1 ].getY();
			v2[ 2 ] = vertices[ 1 ].getZ();
	
			v3[ 0 ] = vertices[ 2 ].getX();
			v3[ 1 ] = vertices[ 2 ].getY();
			v3[ 2 ] = vertices[ 2 ].getZ();
	
			object1->AddTri( v1, v2, v3, iId );
			
			meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOCollided );
			meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOCollided );
			meshOrgan1->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOCollided );
		}
		iId++;
	}
		
     
	object1->EndModel();
    		
	COME_Mesh *meshOrgan2 = organ2->getSurface();
	iId		= 0;
	
	object2->BeginModel();
			
	//add triangles   
	for ( face = 0; face < meshOrgan2->getNumberFaces(); face++ ){
						
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOTenabled );
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOTenabled );
		meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOTenabled );
		if( meshOrgan2->getAFacePt(face)->getCollisionDetectable() ){
			//read the object of the Mesh Class 
			vector<COME_Vertex3D> vertices;
			vertices = meshOrgan2->getVerticesFaceGlobalPosition( face );
	
			double v1[ 3 ], v2[ 3 ], v3[ 3 ];
			v1[ 0 ] = vertices[ 0 ].getX();
			v1[ 1 ] = vertices[ 0 ].getY();
			v1[ 2 ] = vertices[ 0 ].getZ();
	
			v2[ 0 ] = vertices[ 1 ].getX();
			v2[ 1 ] = vertices[ 1 ].getY();
			v2[ 2 ] = vertices[ 1 ].getZ();
	
			v3[ 0 ] = vertices[ 2 ].getX();
			v3[ 1 ] = vertices[ 2 ].getY();
			v3[ 2 ] = vertices[ 2 ].getZ();
	
			object2->AddTri( v1, v2, v3, iId );
			
			meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(0)->setColor( cNOCollided );
			meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(1)->setColor( cNOCollided );
			meshOrgan2->getAFacePt( face )->getVertexGlobalPositionPt(2)->setColor( cNOCollided );
		}
		iId++;
	}
      
	object2->EndModel();          
	
	
	//set up the rotation and translation matrices as
	//required by RAPID for the objects.
	double R[3][3], T[3];

	R[0][0] = R[1][1] = R[2][2] = 1.0;
	R[0][1] = R[1][0] = R[2][0] = 0.0;
	R[0][2] = R[1][2] = R[2][1] = 0.0;

	T[0] = 0.0; T[1] = 0.0; T[2] = 0.0;

	//call the RAPID collision detection routine.
	RAPID_Collide( R, T, object1, R, T, object2, RAPID_ALL_CONTACTS );
      		
	//report the results of collision detection.
	//printf ( "Num contact pairs %d\n", RAPID_num_contacts  );
	
	vector2ndPass.clear();
	for ( int i = 0; i < RAPID_num_contacts; i++ ){

		// Here we have the two triangles in contact
		COME_Mesh *meshOrgan1 = organ1->getSurface();
		COME_Mesh *meshOrgan2 = organ2->getSurface();
		
		vector<COME_Vertex3D> vertices1;
		vertices1 = meshOrgan1->getVerticesFaceGlobalPosition( RAPID_contact[ i ].id1 );
		vector<COME_Vertex3D> vertices2;
		vertices2 = meshOrgan2->getVerticesFaceGlobalPosition( RAPID_contact[ i ].id2 );
		
		// for each vertex of one face 
		if( !organ1->isFixed() ){
			for( int v1 = 0; v1 < vertices1.size(); v1++ ){
		
				if( !meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(v1)->hasFixedAnchor() )
				if( meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->isInsideGlobalPosition( vertices1[v1] ) ){
					// get distance vertex v1 to face 2
					COME_Point3D pointOnPlane;
					double dist1to2 = meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->distanceVertexPlaneGlobalPosition( vertices1[v1] );

					if( !organ2->isFixed() )
						dist1to2/=2.0;
					
					//set displacemet for vert i1
					meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(v1)->addCollisionDisp( meshOrgan2->getAFacePt(RAPID_contact[ i ].id2)->getNormalGlobalPosition() * dist1to2 );
					
					/// Set collision colors
					meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(v1)->setColor( cCollided );
					
					// Add inner adjacencies for second pass test
					if( flagCollision2ndPass ){
						vector<int> neighborFacesInds = meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->getVertexGlobalPositionPt(v1)->getNeighbourFaces();
						for( int neighborFaces = 0; neighborFaces < neighborFacesInds.size(); neighborFaces++ ){
							for( int vertInd = 0; vertInd < meshOrgan1->getAFacePt( neighborFacesInds[neighborFaces] )->getNumberVertices(); vertInd++ ){
								if( meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->isInsideGlobalPosition( meshOrgan1->getAFacePt( neighborFacesInds[neighborFaces] )->getVertexGlobalPosition( vertInd ) ) ){
									COME_2ndPassItem aCollision;
									aCollision.innerV = meshOrgan1->getAFacePt( neighborFacesInds[neighborFaces] )->getVertexGlobalPositionPt( vertInd );
									aCollision.otherF = meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 );
									vector2ndPass.push_back( aCollision );
								}
							}
						}
					}
				}
			} 
		}
		
		// for each vertex of the other face 
		if( !organ2->isFixed() ){
			for( int v2 = 0; v2 < vertices2.size(); v2++ ){
		
				if( !meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(v2)->hasFixedAnchor() )
				if( meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->isInsideGlobalPosition( vertices2[v2] ) ){
					// get distance vertex v2 to face 1
					double dist2to1 = meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->distanceVertexPlaneGlobalPosition( vertices2[v2] );
					
					if( !organ1->isFixed() )
						dist2to1/=2.0;
					
					//set displacemet for vert i2
					meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(v2)->addCollisionDisp( meshOrgan1->getAFacePt(RAPID_contact[ i ].id1)->getNormalGlobalPosition() * dist2to1 );
					
					/// Set collision colors
					meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(v2)->setColor( cCollided );
					
					// Add inner adjacencies for second pass test
					if( flagCollision2ndPass ){
						vector<int> neighborFacesInds = meshOrgan2->getAFacePt( RAPID_contact[ i ].id2 )->getVertexGlobalPositionPt(v2)->getNeighbourFaces();
						for( int neighborFaces = 0; neighborFaces < neighborFacesInds.size(); neighborFaces++ ){
							for( int vertInd = 0; vertInd < meshOrgan2->getAFacePt( neighborFacesInds[neighborFaces] )->getNumberVertices(); vertInd++ ){
								if( meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 )->isInsideGlobalPosition( meshOrgan2->getAFacePt( neighborFacesInds[neighborFaces] )->getVertexGlobalPosition( vertInd ) ) ){
									COME_2ndPassItem aCollision;
									aCollision.innerV = meshOrgan2->getAFacePt( neighborFacesInds[neighborFaces] )->getVertexGlobalPositionPt( vertInd );
									aCollision.otherF = meshOrgan1->getAFacePt( RAPID_contact[ i ].id1 );
									vector2ndPass.push_back( aCollision );
								}
							}
						}
					}
				}
			} 
		}
		collided = true;
	}

	// Perform 2nd pass
	if( flagCollision2ndPass ){
		for( int ind2ndpass = 0; ind2ndpass < vector2ndPass.size(); ind2ndpass++ ){
		
			if( vector2ndPass[ind2ndpass].innerV->getCollisionDisp().vpModule() == 0 ){
				double dist = vector2ndPass[ind2ndpass].otherF->distanceVertexPlaneGlobalPosition( vector2ndPass[ind2ndpass].innerV );
				
				if( ( !organ1->isFixed() ) && ( !organ2->isFixed() ) )
					dist/=2.0;
				
				//set displacemet for vert
				vector2ndPass[ind2ndpass].innerV->addCollisionDisp( vector2ndPass[ind2ndpass].otherF->getNormalGlobalPosition() * dist );
				vector2ndPass[ind2ndpass].innerV->setColor( cCollidedTotal );
			}
		}
	}
	
	delete object1;
	delete object2;
	
	return collided;
}


//Checks if the integer given as argument is already in the vector group
bool
COME_Collide::isNotAlreadyIn(vector<int> group, int part){

	for (int i = 0; i < group.size(); i++){
		if (group[i] == part) return false;
	}
	return true;
}


//Projects all vertices of the faces given as argument on the the dir Vector
//and returns the one with the biggest value 
double
COME_Collide::findMaxProjection(COME_Vector3D dir, COME_Vertex3D base, COME_Molecule* nearestMol, vector<int> faces, COME_Mesh* mesh){

	double max = -2000000;
	dir = dir.vpNormalize();
	for (int i =0; i < faces.size(); i++){
		if (mesh->getAFacePt(faces[i])->getNearestMolecule() == nearestMol){
			vector<COME_Vertex3D> vertices = mesh->getAFacePt(faces[i])->getVerticesGlobalPosition();
			for (int a = 0; a < vertices.size(); a++){
				COME_Vector3D vect = vertices[a] - base;
				double distance = dir.vpDotProduct(vect);
				if (distance > max) {
				//	cout << " max = " << max << "\n";
					max=distance;
				}
			}
		}
	}
		
	return max;
}


//Projects all vertices of the faces given as argument on the the dir Vector
//and returns the one with the smallest value 
double
COME_Collide::findMinProjection(COME_Vector3D dir, COME_Vertex3D base, COME_Molecule* nearestMol, vector<int> faces, COME_Mesh* mesh){

	double min = 2000000;
	dir = dir.vpNormalize();
	for (int i =0; i < faces.size(); i++){
		if (mesh->getAFacePt(faces[i])->getNearestMolecule() == nearestMol){
			vector<COME_Vertex3D> vertices = mesh->getAFacePt(faces[i])->getVerticesGlobalPosition();
			for (int a = 0; a < vertices.size(); a++){
				COME_Vector3D vect = vertices[a] - base;
				double distance = dir.vpDotProduct(vect);
				if (distance < min) {
					//cout << " min = " << min << "\n";
					min=distance;
				}
			}
		}
	}
		
	return min;
		
}

//Checks if the face given as parameter is inside the acceptable range defined
//by the min and max values. Uses the projection of the vertices of the face on to the
//two direction vectors to compare with min and max values.
bool
COME_Collide::IsInAcceptableRange(COME_Vector3D dir1,COME_Vector3D dir2,COME_Vertex3D base, double maxDir1,double minDir1,double maxDir2,double minDir2, COME_Face* face){

	vector<COME_Vertex3D> vertices = face->getVertices();
	for (int a = 0; a < vertices.size(); a++){
		COME_Vector3D vect = vertices[a] - base;
		double distance1 = dir1.vpDotProduct(vect);
		if ( distance1 > maxDir1 || distance1 < minDir1){
			//cout << "Face is not in acceptable range dir1!!\n";
			return false;			
		}
		double distance2 = dir2.vpDotProduct(vect);
		if (distance2 > maxDir2 || distance2 < minDir2) {
			//cout << "Face is not in acceptable range dir2!!\n";
			return false;			
		}
		
	}
		
	return true;
}

///////////////////////////////////////////////////////////////////////////////// PROXIMITIES FUNCTIONS /////////////////////////////////////////////////////////////////////////
/// 
void
COME_Collide::initializeProximityStructure(){

	clock_t start,end;
	start = clock ();
	proximityStructure.clear();
	COME_Patient *activePatient = ((COME_Scenario*)parent)->getPatient(((COME_Scenario*)parent)->getPatientList()->size()-1);
	// If there's only one object, doesn't detect collisions
	if( activePatient->getPtOrganList()->size() > 1 ){

		// Take the organs of the first patient (there is only one patientin this phase)
		list<COME_MoleculesCartilage*>::const_iterator iterOrgans1;
		for ( iterOrgans1 =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans1 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans1++  ){
			
			list<COME_MoleculesCartilage*>::const_iterator iterOrgans2;
			for ( iterOrgans2 = ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans2 != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans2++  ){
				
				double softhard = SOFT;
				if( (*iterOrgans1)->isFixed() ||  (*iterOrgans2)->isFixed() ){
					softhard = HARD;
				}
				
				if( (*iterOrgans1) != (*iterOrgans2) ){
					if(
					( ( (*iterOrgans1)->getDescription() == "femurcart" ) && ( (*iterOrgans2)->getDescription() == "pelviscart" ) ) || /// TEST WITH TWO CARTILAGES
					( ( (*iterOrgans1)->getDescription() == "femurcart1" ) && ( (*iterOrgans2)->getDescription() == "pelviscart1" ) ) || /// TEST WITH SIMPLE OBECTS
					( ( (*iterOrgans1)->getDescription() == "sphere" ) && ( (*iterOrgans2)->getDescription() == "pelviscart1" ) ) || /// SPHERE WITH BIG FLOOR
					( ( (*iterOrgans1)->getDescription() == "femurbone" ) && ( (*iterOrgans2)->getDescription() == "ischiofemoral" ) ) || /// LIGAMENT and BONE
					( ( (*iterOrgans1)->getDescription() == "femurcart" ) && ( (*iterOrgans2)->getDescription() == "ischiofemoral" ) ) /// LIGAMENT and FEMURCART
					){
						addProximities( (*iterOrgans1)->getSurface(), (*iterOrgans2)->getSurface(), softhard );
					}
				}
			}
		}
	}
	printf( "Proximities initilized. %d pairs. \n", proximityStructure.size() );
	end = clock ();
	double dif = end - start;
	printf ("Init proximities for collision time was %.8f seconds.\n", dif / (double) CLOCKS_PER_SEC );
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
/// Add to the proximity structure of this Collide the proximities between the two meshes given.
/// All vertices of non-facing triangles are marked as non-colliding. Note that this assumption may 
/// not work if  one mesh moves a lot such that the facing triangles change of orientation.
//////////////////////////////////////////////////////////////////////////////////////////////////////
int
COME_Collide::addProximities( COME_Mesh *mA, COME_Mesh * mB, double hardSoft ){

	int iB;
	for( int iA = 0; iA < mA->getVerticesGlobalPt()->size(); iA++ ){
		
		COME_Vertex3D* vertexLocal = &((*(mA->getVerticesPt()))[iA]);
		if( vertexLocal->vpDistance( COME_Point3D() ) >= ((COME_BioStructure*)(mA->getParent()))->getCollisionRadius() ){
				continue;
		}
		COME_Vertex3D* vertexInA = &((*(mA->getVerticesGlobalPt()))[iA]);
		iB = mB->getClosestFaceIndex( COME_Point3D(vertexInA->x, vertexInA->y, vertexInA->z ) );
		int iBV = mB->getClosestVertexIndex( COME_Point3D(vertexInA->x, vertexInA->y, vertexInA->z ) );
		COME_Face* closestFaceInB = mB->getAFacePt(iB);
		COME_Vertex3D* closestInB = &((*(mB->getVerticesGlobalPt()))[iBV]);
		
		COME_Vector3D tempVec = (*closestInB);
		tempVec.vpNormalize();
		if( ( closestFaceInB->getNormalGlobalPosition().vpDotProduct( tempVec ) < 0.0 )
		&& ( vertexInA->getNormalGlobalPosition().vpDotProduct( tempVec ) >= 0.0 ) ) { 
		
			proximityStructure.push_back( COME_Proximity( vertexInA, closestFaceInB, hardSoft ) );
			vertexInA->setCollide( true );
		} else {
			vertexInA->setCollide( false );
		}
		
		/// uncomment to have only one proximity
		//if( proximityStructure.size() > 0 ) return proximityStructure.size();
	}
	
	return proximityStructure.size();
}

void
COME_Collide::updateProximityStructure(){

	for( int iPS = 0; iPS < proximityStructure.size(); iPS++ ){
	
		proximityStructure[iPS].faceB = proximityStructure[iPS].faceB->getNeighborClosestTo( proximityStructure[iPS].objA );
		proximityStructure[iPS].updatePointOnFace();
		double lacor[4]; lacor[0]=1; lacor[1]=1; lacor[2]=1; lacor[3]=1;
		proximityStructure[iPS].faceB->getVertexGlobalPositionPt(0)->setColor( lacor );
		proximityStructure[iPS].faceB->getVertexGlobalPositionPt(1)->setColor( lacor );
		proximityStructure[iPS].faceB->getVertexGlobalPositionPt(2)->setColor( lacor );
	}
}

vector<COME_Proximity>*
COME_Collide::getProximityStructurePt(){

	return &proximityStructure;
}

void
COME_Collide::geometricResponse(){

	// rebuild all Collision Vectors
	for( int iPS = 0; iPS < proximityStructure.size(); iPS++ ){
		proximityStructure[iPS].updateVelocity();
	}
}

void
COME_Collide::resetProximitiesDisps(){

	// rebuild all Collision Vectors
	for( int iPS = 0; iPS < proximityStructure.size(); iPS++ ){
	
		proximityStructure[iPS].resetDisplacements();
	}
}

///////////////////////////////////////////////////////////////////////////////// HASH FUNCTIONS /////////////////////////////////////////////////////////////////////////
/// 


void
COME_Collide::sphericalSlidingResponse(){

	COME_Patient *activePatient = ((COME_Scenario*)parent)->getPatient(((COME_Scenario*)parent)->getPatientList()->size()-1);
	// Reset all collision vectors
	list<COME_MoleculesCartilage*>::const_iterator iterOrgansReset;
	for ( iterOrgansReset =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgansReset != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgansReset++  ){
		//if( (*iterOrgansReset)->internals.size() > 0 ){
			(*iterOrgansReset)->getSurface()->resetCollisionDisps();
		//}
	}
	
	// rebuild all Collision Vectors
	// run all vertices of the internal mesh and construct ephemerous proximities
	
	// first clean ephemerous proximities for test
	proximityStructure.clear();
	
	list<COME_MoleculesCartilage*>::const_iterator iterOrgans;
	for ( iterOrgans =((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->begin(); iterOrgans != ((list<COME_MoleculesCartilage*>*)activePatient->getPtOrganList())->end(); iterOrgans++  ){
	
		for( int iI = 0; iI < (*iterOrgans)->internals.size(); iI++ ){
		
			softhard = SOFT;
			if( (*iterOrgans)->isFixed() ||  (*iterOrgans)->internals[iI]->isFixed() ){
				softhard = HARD;
			}
			COME_Mesh *mA = (*iterOrgans)->internals[iI]->getSurface();
			for( int iA = 0; iA < mA->getVerticesGlobalPt()->size(); iA++ ){
				COME_Vertex3D* vertexInA = &((*(mA->getVerticesGlobalPt()))[iA]);
				if( vertexInA->getCollide() ){
				
					// calculate spherical coord for A
					COME_Vector3D aNormal; aNormal.setVector3D( vertexInA->getVertexAsPoint3D() );
					aNormal.vpNormalize();
					COME_Point2D aSpherical = COME_Point2D( atan2( aNormal.y, aNormal.x ), acos( aNormal.z ) ) - (*iterOrgans)->sSpherical;
					COME_Face *theFace = (*iterOrgans)->getHashElement(aSpherical).face;
					if( theFace ){
						//printf( "hit\n" );
						updateVelocity( theFace, vertexInA );
						proximityStructure.push_back( COME_Proximity( vertexInA, theFace, softhard ) );
					}
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// Calculate velocity update for the face and vertex given when needed
void
COME_Collide::updateVelocity( COME_Face *faceB, COME_Vertex3D *objA ){

	COME_Point3D	pointOnB = faceB->getClosestPointOnFaceToPoint( COME_Point3D( objA->x, objA->y, objA->z ) );

	COME_Vector3D velocDir;
	if( !( signedDistance(faceB, objA->getVertexAsPoint3D(), pointOnB, objA->getNormalGlobalPosition() ) < 0) ){
		velocDir = objA->getVertexAsPoint3D() - pointOnB; velocDir.vpNormalize();
	
		COME_Vector3D velocA = objA->getBlendedVelocity();
		COME_Vector3D velocB = faceB->getBlendedVelocity( pointOnB );
		COME_Vector3D velocRelativeA = velocDir * COME_Vector3D(velocA-velocB).vpDotProduct(velocDir);
		
		COME_Point3D posObjATimePlusOne = objA->getVertexAsPoint3D() + ( velocRelativeA * COME::timestepGlobal * 1.0 );
		
		if( signedDistance( faceB, posObjATimePlusOne, pointOnB, objA->getNormalGlobalPosition() ) < 0.0 ){
			
			COME_Vector3D velocCorrection = velocRelativeA - ( ( pointOnB - objA->getVertexAsPoint3D() ) * ( 1.0 / ( COME::timestepGlobal * 1.0 ) ) );
			
			// update objA
			objA->addVelocityDisp( -velocCorrection/softhard );
			
			// update faceB
			COME_Vertex3D *A =  faceB->getVertexGlobalPositionPt( 0 );
			COME_Vertex3D *B =  faceB->getVertexGlobalPositionPt( 1 );
			COME_Vertex3D *C =  faceB->getVertexGlobalPositionPt( 2 );
			
			A->addVelocityDisp( velocCorrection/softhard );
			B->addVelocityDisp( velocCorrection/softhard );
			C->addVelocityDisp( velocCorrection/softhard );
		}
	} else {
		updatePosition( faceB, objA, pointOnB );
		objA->addVelocityDisp( -objA->getBlendedVelocity() );
		faceB->getVertexGlobalPositionPt( 0 )->addVelocityDisp( -faceB->getVertexGlobalPositionPt( 0 )->getBlendedVelocity() );
		faceB->getVertexGlobalPositionPt( 1 )->addVelocityDisp( -faceB->getVertexGlobalPositionPt( 1 )->getBlendedVelocity() );
		faceB->getVertexGlobalPositionPt( 2 )->addVelocityDisp( -faceB->getVertexGlobalPositionPt( 2 )->getBlendedVelocity() );
	}
	
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// Returns the distance of the point to the face given. Distance is negative if a penetration occurs.
double
COME_Collide::signedDistance( COME_Face *faceB, COME_Point3D p, COME_Point3D pointOnB, COME_Vector3D normal ){
 
	double dist = pointOnB.vpDistance( p );

	if( ( faceB->getNormalGlobalPositionPt()->vpDotProduct( normal ) < 0.0 ) && ( faceB->isInsideGlobalGOOD( p ) ) )
		dist = -dist;
	
	CONTADIST++;
	return dist;
 }

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// Calculate position update for this ephemerous proximity.
void
COME_Collide::updatePosition( COME_Face *faceB, COME_Vertex3D *objA, COME_Point3D pointOnB ){

	COME_Vector3D posCorrection = pointOnB - objA->getVertexAsPoint3D();
	
	// update objA
	objA->addCollisionDisp( posCorrection/softhard );
	
	// update faceB
	COME_Vertex3D *A =  faceB->getVertexGlobalPositionPt( 0 );
	COME_Vertex3D *B =  faceB->getVertexGlobalPositionPt( 1 );
	COME_Vertex3D *C =  faceB->getVertexGlobalPositionPt( 2 );
	
	A->addCollisionDisp( -posCorrection/softhard );
	B->addCollisionDisp( -posCorrection/softhard );
	C->addCollisionDisp( -posCorrection/softhard );

}
