/****************************************************************************

 This file is part of the QGLViewer library.
 Copyright (C) 2002, 2003, 2004, 2005 Gilles Debunne (Gilles.Debunne@imag.fr)
 Version 2.1.1-6, packaged on October 7, 2005.

 http://artis.imag.fr/Members/Gilles.Debunne/QGLViewer

 libQGLViewer is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 libQGLViewer is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with libQGLViewer; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*****************************************************************************/

#ifndef DVONN_GAME_H
#define DVONN_GAME_H

#include "board.h"
#include <qstring.h>
#include <vector>
#include <map>
#include <stdexcept>

namespace dvonn
{

  typedef enum { WhitePlayer=0, BlackPlayer=1 } Player;
  typedef enum { RedPlacementPhase  = 0,
		 PiecePlacementPhase= 1,
		 MovePhase          = 2,
		 GameOverPhase      = 3 } Phase;

  extern Color colorOf(Player p);
  extern Player player(Color c) throw (std::range_error);
  extern const char* nameOf(const dvonn::Player);
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Game
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class Game
  {
  public:
    class Placement;
    class Move;

    Game();
    virtual ~Game();
    void reinit();

    const Board& board() const;

    Player theOnePlaying() const;
    Phase  phase() const;
    bool isOver() const;
    int score(Player) const;

    bool isLegalPlacement(const Placement) const;
    bool isLegalMove(const Move) const;

    bool doPlacement(const Placement);
    bool doMove(const Move);
    const Board::Ghosts* killedBy(const Move) const;
    std::deque<Board::ConstStackHandle> possibleDestinations(const Board::ConstStackHandle& h) const;

    void randomlyFinishPlacement();
    bool getRandomMove(Player p,Move& m) const;

    const char* fileName() const;
    bool save();
    bool saveAs(const char* fileName);
    bool load(const char* fileName);

    void undo();
    void redo();
    bool canUndo() const;
    bool canRedo() const;
  private:
    void switchPlayers(Player p);
    void updateHistory();

    std::string                          fileName_;
    Board                                board_;
    Phase                                phase_;
    Player                               player_;
    std::map<Move,Board::Ghosts>         ghosts_;
    int                                  score_[2];
    unsigned int                         time_;
    unsigned int                         knownTime_;
    std::deque<Board::State>             historyStates_;
    std::deque<Player>                   historyPlayers_;
    std::deque<Phase>                    historyPhases_;
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Game::Placement
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class Game::Placement
  {
  public:
    Placement(Color,Board::Coord);
    Color        color;
    Board::Coord dst;
  };
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Interface of Game::Move
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  class Game::Move
  {
  public:
    Move(Board::Coord = Board::Coord(),Board::Coord = Board::Coord());
    Board::Coord src;
    Board::Coord dst;
    bool operator<(const Move other) const;
  };
}
extern std::ostream& operator<<(std::ostream&,const dvonn::Game::Placement);
extern std::ostream& operator<<(std::ostream&,const dvonn::Game::Move);

#endif // DVONN_GAME_H
