/***************************************************************************
 *   Copyright (C) 2004 by Anderson Maciel                                 *
 *   andi.maciel@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/
#include "comemechatester.h"
#include <qfiledialog.h>
#include <qspinbox.h>
#include <qcursor.h>
#include <qstatusbar.h>
#include <qtable.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <come_mechatestqglwidget.h>
#include <dlgnewmaterial.h>
#include <general/come.h>
#include <general/comesimulator.h>
#include <general/comescenario.h>
#include <bio/comemoleculescartilage.h>
#include <bio/comematerial.h>
#include <algorithm>
#include <qtl.h>

using namespace std;

COME_Mechatester::COME_Mechatester()
 : mechatester(){
 
	scene = new COME_Scenario();
	sim = new COME_Simulator();
	sim->setScene( scene );
	molecFixSelected = molecMobileSelected = NULL;
	//"../biomodel/models/clamp.iv" "/home/amaciel/development/biomodel/models/clamp.iv"
	scene->loadOrganMesh( "../biomodel/models/clamp.iv", CLAMP );
	scene->getPatient(0)->getOrgan(0)->setDescription( "clamp0" );
	scene->getPatient(0)->getOrgan(0)->scale( 0.2, 0.12, 0.01 );
	scene->getPatient(0)->getOrgan(0)->translate( 0.0, 0.0, 0.0 );
	scene->getPatient(0)->getOrgan(0)->updateSkin();
	scene->loadOrganMesh( "../biomodel/models/clamp.iv", CLAMP );
	scene->getPatient(0)->getOrgan(1)->setDescription( "clamp1" );
	scene->getPatient(0)->getOrgan(1)->scale( 0.2, 0.12, 0.01 );
	scene->getPatient(0)->getOrgan(1)->translate( 0.0, 0.0, 0.2 );
	scene->getPatient(0)->getOrgan(1)->updateSkin();
	scene->addMaterialsFromFile( "../biomodel/models/base_materials.xml", sim );
	openglArea->setScene( scene );
	openglArea->setGlobal( (COME*)this );
	cbMesh->setChecked( FALSE );
	updateMaterialsList();
}

 void
 COME_Mechatester::loadFileIfArguments( int argc, char ** argv ){
 
	if( ( argc > 1 ) && ( argc < 3 ) ){
		printf("Usage: mechatester [input_file] [output_file]\n" );
	} else if( argc == 3 ){
		default_output_file = argv[2];
		QString fileName = argv[1];
		if( !fileName.isEmpty() ){
	
			QString folderName = fileName;
			folderName.truncate( folderName.findRev( "/" ) );
			COME::baseFolder = folderName.latin1();
			
			QString msg = "Loading ";
			statusBar()->message( msg + fileName );
			hourglass(true);
			scene->addOrganFromFile( fileName.latin1(), sim );
			updateMaterialsList();
			statusBar()->message( "Loaded.", 2000 );
			hourglass(false);

			// Set scene radius for view
			double sceneRadius = 0.01;
			COME_Point3D mins, maxs;
			list<COME_BioStructure*>::const_iterator iterOrgans;
			for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(); iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
				
				((COME_MoleculesCartilage*)(*iterOrgans))->getEnvelop( mins, maxs );
				sceneRadius = mins.vpDistance( COME_Point3D() ) > sceneRadius ? mins.vpDistance( COME_Point3D() ) : sceneRadius;
				sceneRadius = maxs.vpDistance( COME_Point3D() ) > sceneRadius ? maxs.vpDistance( COME_Point3D() ) : sceneRadius;
			}
			openglArea->setSceneRadius( sceneRadius + sceneRadius*0.1 );
			openglArea->camera()->showEntireScene();

			// Scale clampers (full of magic numbers... it's heuristical!)
			scene->getPatient(0)->getOrgan("clamp0")->getEnvelop( mins, maxs );
			double scaleFactor = ( sceneRadius ) / ( 2.0 * mins.vpDistance( maxs ) );
			double translateZ = sceneRadius / 5.0;
			scene->getPatient(0)->getOrgan("clamp0")->scale(scaleFactor,scaleFactor,scaleFactor);
			scene->getPatient(0)->getOrgan("clamp1")->scale(scaleFactor,scaleFactor,scaleFactor);
			scene->getPatient(0)->getOrgan("clamp1")->translate( 0.0, 0.0, translateZ );

		}

	}

}

COME_Mechatester::~COME_Mechatester(){

	delete scene;
	delete sim;
}

void
COME_Mechatester::fileOpen(){

	QString fileName = QFileDialog::getOpenFileName(
                    ".",
                    "Organ files (*.xml)",
                    this,
                    "open file dialog"
                    "Choose a file" );
	if( !fileName.isEmpty() ){
	
		QString folderName = fileName;
		folderName.truncate( folderName.findRev( "/" ) );
		COME::baseFolder = folderName.latin1();
		
		QString msg = "Loading ";
		statusBar()->message( msg + fileName );
		hourglass(true);
		scene->addOrganFromFile( fileName.latin1(), sim );
		updateMaterialsList();
		statusBar()->message( "Loaded.", 2000 );
		hourglass(false);

		// Set scene radius for view
		double sceneRadius = 0.01;
		COME_Point3D mins, maxs;
		list<COME_BioStructure*>::const_iterator iterOrgans;
		for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(); iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
			
			((COME_MoleculesCartilage*)(*iterOrgans))->getEnvelop( mins, maxs );
			sceneRadius = mins.vpDistance( COME_Point3D() ) > sceneRadius ? mins.vpDistance( COME_Point3D() ) : sceneRadius;
			sceneRadius = maxs.vpDistance( COME_Point3D() ) > sceneRadius ? maxs.vpDistance( COME_Point3D() ) : sceneRadius;
		}
		openglArea->setSceneRadius( sceneRadius + sceneRadius*0.1 );
		openglArea->camera()->showEntireScene();

		// Scale clampers (full of magic numbers... it's heuristical!)
		scene->getPatient(0)->getOrgan("clamp0")->getEnvelop( mins, maxs );
		double scaleFactor = ( sceneRadius ) / ( 2.0 * mins.vpDistance( maxs ) );
		double translateZ = sceneRadius / 5.0;
		scene->getPatient(0)->getOrgan("clamp0")->scale(scaleFactor,scaleFactor,scaleFactor);
		scene->getPatient(0)->getOrgan("clamp1")->scale(scaleFactor,scaleFactor,scaleFactor);
		scene->getPatient(0)->getOrgan("clamp1")->translate( 0.0, 0.0, translateZ );

	}
}

void
COME_Mechatester::fileExportAs(){

	QString fileName;
	QFileDialog* fd = new QFileDialog( this, "save as dialog", TRUE );
	fd->setCaption( "Export molecules file as...");
	fd->setMode( QFileDialog::AnyFile );
	fd->setFilter( "Molecules (*.xml)" );
	if( default_output_file != "" )
		fd->setSelection( default_output_file.c_str() );
	
	if ( fd->exec() == QDialog::Accepted ){
		hourglass(true);
		fileName = fd->selectedFile();
		printf( "Saving %s... ", fileName.latin1() );
		scene->saveFile( fileName.latin1(), sim );
		printf( " saved.\n" );
		hourglass(false);
	}
}

void
COME_Mechatester::updateMaterialsList(){
	
	// load materials onto the list
	listMaterials->clear();
	vector<COME_Material*>* availableMaterials = scene->getAvailableMaterialsList();
	for( int i = 0; i < availableMaterials->size(); i++ ){
		listMaterials->insertItem( QString( (*availableMaterials)[i]->getDescription().c_str() ) );
	}
	// First is selected
	selectedMaterial = (*availableMaterials)[0];
	listMaterials->setSelected( listMaterials->findItem ( (*availableMaterials)[0]->getDescription().c_str() ), true );
	
	// Load material properties onto the table.
	loadMaterialProperties();
}

void
COME_Mechatester::selectMaterial(){

	// Set selected
	selectedMaterial = (*(scene->getAvailableMaterialsList()))[listMaterials->currentItem()];
	// Load material properties onto the table.
	loadMaterialProperties();
}

void
COME_Mechatester::rotateSelected( int rx, int ry, int rz ){

	if( scene->getPatient(0)->getSelected() ){
		double step = sbStepRotate->value() * M_PI / 180.0;
		scene->getPatient(0)->getSelected()->rotate( rx*step, ry*step, rz*step );
		updateClampIntersections();
	}
}

void
COME_Mechatester::translateSelected( int dx, int dy, int dz ){

	if( scene->getPatient(0)->getSelected() ){
		double step = sbStepTranslate->value() * (openglArea->sceneRadius()/(double)openglArea->width());
		scene->getPatient(0)->getSelected()->translate( dx*step, dy*step, dz*step );
		updateClampIntersections();
	}
}

void
COME_Mechatester::scaleSelected( int fx, int fy, int fz ){

	if( scene->getPatient(0)->getSelected() ){
		double step = ( (fx+fy+fz) > 0.0
				? sbStepScale->value() / 100.0
				: 1.0 - (sbStepScale->value() % 100) / 100.0);
		scene->getPatient(0)->getSelected()->scale( fx==0?1.0:step, fy==0?1.0:step, fz==0?1.0:step );
		updateClampIntersections();
	}
}

void
COME_Mechatester::loadMaterialProperties(){
	
	char chrText[20];
	sprintf( chrText, "%1.4f", selectedMaterial->getColor().getX() );
	tMaterials->setText( 0, 0, chrText ); tMaterials->setText( 0, 1, "" ); tMaterials->setText( 0, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getColor().getY() );
	tMaterials->setText( 1, 0, chrText ); tMaterials->setText( 1, 1, "" ); tMaterials->setText( 1, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getColor().getZ() );
	tMaterials->setText( 2, 0, chrText ); tMaterials->setText( 2, 1, "" ); tMaterials->setText( 2, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getDamping() );
	tMaterials->setText( 3, 0, chrText ); tMaterials->setText( 3, 1, "%/100" ); tMaterials->setText( 3, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getDensity() );
	tMaterials->setText( 4, 0, chrText ); tMaterials->setText( 4, 1, "kg/m3" ); tMaterials->setText( 4, 2, "0..~" );
	sprintf( chrText, "%1.10f", selectedMaterial->getPermeability() );
	tMaterials->setText( 5, 0, chrText ); tMaterials->setText( 5, 1, "m3/m2/s?" ); tMaterials->setText( 5, 2, "0..~" );
	sprintf( chrText, "%1.4f", selectedMaterial->getLiquidFraction() );
	tMaterials->setText( 6, 0, chrText ); tMaterials->setText( 6, 1, "%/100" ); tMaterials->setText( 6, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getMediumDensity() );
	tMaterials->setText( 7, 0, chrText ); tMaterials->setText( 7, 1, "kg/m3?" ); tMaterials->setText( 7, 2, "0..~" );
	sprintf( chrText, "%1.4f", selectedMaterial->getYoungsModulus() );
	tMaterials->setText( 8, 0, chrText ); tMaterials->setText( 8, 1, "N/m2" ); tMaterials->setText( 8, 2, "0..~" );
	sprintf( chrText, "%1.4f", selectedMaterial->getAnisotropyVector().getX() );
	tMaterials->setText( 10, 0, chrText ); tMaterials->setText( 10, 1, "%/100" ); tMaterials->setText( 10, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getAnisotropyVector().getY() );
	tMaterials->setText( 11, 0, chrText ); tMaterials->setText( 11, 1, "%/100" ); tMaterials->setText( 11, 2, "0..1" );
	sprintf( chrText, "%1.4f", selectedMaterial->getAnisotropyVector().getZ() );
	tMaterials->setText( 12, 0, chrText ); tMaterials->setText( 12, 1, "%/100" ); tMaterials->setText( 12, 2, "0..1" );
}

void
COME_Mechatester::materialChanged( int l, int c ){

	

	selectedMaterial->setColor( COME_Vector3D( atof( tMaterials->text( 0, 0 ) ), atof( tMaterials->text( 1, 0 ) ), atof( tMaterials->text( 2, 0 ) ) ) );
	selectedMaterial->setDamping( atof( tMaterials->text( 3, 0 ) ) );
	selectedMaterial->setDensity( atof( tMaterials->text( 4, 0 ) ) );
	selectedMaterial->setPermeability( atof( tMaterials->text( 5, 0 ) ) );
	selectedMaterial->setLiquidFraction( atof( tMaterials->text( 6, 0 ) ) );
	selectedMaterial->setMediumDensity( atof( tMaterials->text( 7, 0 ) ) );
	selectedMaterial->setYoungsModulus( atof( tMaterials->text( 8, 0 ) ) );
	selectedMaterial->setAnisotropyVector( COME_Vector3D( atof( tMaterials->text( 10, 0 ) ), atof( tMaterials->text( 11, 0 ) ), atof( tMaterials->text( 12, 0 ) ) ) );
		
	char chrText[40];
	sprintf( chrText, "Material changed at line: %d column: %d", l, c );
	QString msg = chrText;
	statusBar()->message( msg, 1000 );
}

void
COME_Mechatester::clickedNewMaterial(){

	dlgNewMaterial* dlg = new dlgNewMaterial( this );
	
	if ( dlg->exec() == QDialog::Accepted ){
	
		COME_Material *newMaterial = new COME_Material();
		newMaterial->setDescription( dlg->newDescription->text().latin1() );
		scene->getAvailableMaterialsList()->push_back( newMaterial );
		updateMaterialsList();
	}
	delete dlg;
}

void
COME_Mechatester::clickedDelMaterial(){

	vector<COME_Material*>::iterator select_iter;
	for( select_iter = scene->getAvailableMaterialsList()->begin(); select_iter != scene->getAvailableMaterialsList()->end(); select_iter++ ){
		if( (*select_iter)->getDescription() == selectedMaterial->getDescription() )
			break;
	}
	if( select_iter != scene->getAvailableMaterialsList()->end() ){
		scene->getAvailableMaterialsList()->erase( select_iter );
		updateMaterialsList();
	}
}

void
COME_Mechatester::clickedAssociate(){

	if( scene->getPatient(0)->getSelected() ){
		char chrText[100];
		sprintf( chrText, "Material %s is now associated to object %s.", selectedMaterial->getDescription().c_str(), scene->getPatient(0)->getSelected()->getDescription().c_str() );
		QString msg = chrText;
		statusBar()->message( msg, 2000 );
		scene->getPatient(0)->getSelected()->getTissue()->setMoleculesMaterial( selectedMaterial );
	} else {
		statusBar()->message( "No association created because there is no object selected.", 2000 );
	}
}
void
COME_Mechatester::clickedCheck(){

	list<COME_BioStructure*>::const_iterator iterOrgans;
	for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
	
		((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->checkInitialPositions();
		((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->makeAllLocalFrames();
		((COME_MoleculesCartilage*)(*iterOrgans))->updateSkin();
		updateClampIntersections();
	}
}

void
COME_Mechatester::clickedReset(){

	list<COME_BioStructure*>::const_iterator iterOrgans;
	for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
	
		((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->resetInitialPositions();
		((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->makeAllLocalFrames();
		((COME_MoleculesCartilage*)(*iterOrgans))->updateSkin();
		updateClampIntersections();
	}
}

void
COME_Mechatester::clickedStartSim(){

	char chrText[20];
	
	if( cbConfigure->isChecked() ){

		// initialize molec config TEMPORARY
		if( scene->getPatient(0)->getOrgan(2)->getGlobalHooke() == 0.0 ){
			scene->getPatient(0)->getOrgan(2)->setGlobalHooke( scene->getPatient(0)->getOrgan(2)->getTissue()->getShape()->front()->getConnectionList()->front()->getHookeConst() );
		}
	}
	
	if( tabSimulation->currentPageIndex() == 0 ){
	
		// Simulate by forces
		sim->setDuration( atof(leFadeInForce->text()) + atof(leKeepForce->text()) + atof(leFadeOutForce->text()) );
		pressureArea = 0.0;
		
		// Set forces
		list<COME_BioStructure*>::const_iterator iterOrgans;
		for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
		
			list<COME_Molecule*>* molecules = ((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->getShape();
			list<COME_Molecule*>::iterator iterMolecules;
			double totalMolecForce = 0.0;
			for ( iterMolecules = molecules->begin(); iterMolecules != molecules->end(); iterMolecules++){
				if( (*iterMolecules)->intersectClampMobile() ){
					totalMolecForce+=1.0;
					pressureArea+=(*iterMolecules)->getPressureArea();
					// just choose one
					molecMobileSelected = (*iterMolecules);
				}
				(*iterMolecules)->clearExternalForces();
			}
			double xSplitForce = atof(leForceX->text())/totalMolecForce;
			double ySplitForce = atof(leForceY->text())/totalMolecForce;
			double zSplitForce = atof(leForceZ->text())/totalMolecForce;
			double currDistance = INFINITE;
			for ( iterMolecules = molecules->begin(); iterMolecules != molecules->end(); iterMolecules++){
			
				(*iterMolecules)->setFixed( false );
				if( (*iterMolecules)->intersectClampMobile() ){
					COME_TimeForce *forceIn = new COME_TimeForce( xSplitForce, ySplitForce, zSplitForce, atof(leFadeInForce->text()) );
					COME_TimeForce *forceKeep = new COME_TimeForce( xSplitForce, ySplitForce, zSplitForce, atof(leFadeInForce->text()) + atof(leKeepForce->text()) );
					COME_TimeForce *forceOut = new COME_TimeForce( 0.0, 0.0, 0.0, atof(leFadeInForce->text()) + atof(leKeepForce->text()) + atof(leFadeOutForce->text()) );
					
					(*iterMolecules)->addExternalForce( new COME_TimeForce( 0.0, 0.0, 0.0, 0.0 ) );
					(*iterMolecules)->addExternalForce( forceIn );
					(*iterMolecules)->addExternalForce( forceKeep );
					(*iterMolecules)->addExternalForce( forceOut );
				} else if( (*iterMolecules)->intersectClampFix() ) {
					(*iterMolecules)->setFixed( true );
					// take the closest as fixed selected
					if( molecMobileSelected )
					if( (*iterMolecules)->getPosition().vpDistance( molecMobileSelected->getPosition() ) < currDistance ){
						molecFixSelected = (*iterMolecules);
						currDistance = (*iterMolecules)->getPosition().vpDistance( molecMobileSelected->getPosition() );
					}
				}
			}
		}
		if( molecFixSelected && molecMobileSelected )
			previousLenght = initialLength = molecFixSelected->getPosition().vpDistance( molecMobileSelected->getPosition() );
	} else {
		// Simulate by displacement
		sim->setDuration( atof(leFadeInDisp->text()) + atof(leKeepDisp->text()) + atof(leFadeOutDisp->text()) );
	
	}
	
	// Simulate physics here to perform the test
	sim->setFPS( sbFPS->value() );
	sim->start();

}

void
COME_Mechatester::updateClampIntersections(){

	if( !sim->isRunning() ){
	
		// test intersections
		COME_Mesh* clamp0 = scene->getPatient(0)->getOrgan(0)->getSurface();
		COME_Mesh* clamp1 = scene->getPatient(0)->getOrgan(1)->getSurface();
		list<COME_BioStructure*>::const_iterator iterOrgans;
		for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
		
			list<COME_Molecule*>* molecules = ((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->getShape();
			list<COME_Molecule*>::iterator iterMolecules;
			for ( iterMolecules = molecules->begin(); iterMolecules != molecules->end(); iterMolecules++){
			
				if( clamp0->isInside( (*iterMolecules)->getPosition() ) ){
					(*iterMolecules)->setIntersectClamp( CLAMPFIX );
				} else if ( clamp1->isInside( (*iterMolecules)->getPosition() ) ){
					(*iterMolecules)->setIntersectClamp( CLAMPMOBILE );
				} else {
					(*iterMolecules)->setIntersectClamp( FREE );
				}
			}
		}
		
	}
}

void
COME_Mechatester::updateEquation(){

	char chrText[20];
	double momentaryElasticity;
	
	if( molecFixSelected && molecMobileSelected ){
		COME_Vector3D force(atof(leForceX->text()), atof(leForceY->text()), atof(leForceZ->text()));
		sprintf( chrText, "%3.4f x %2.4f", force.vpModule(), initialLength );
		tlNumerator->setText( chrText );
		
		double currentLength = molecFixSelected->getPosition().vpDistance( molecMobileSelected->getPosition() );
		sprintf( chrText, "%1.5f x %2.5f", currentLength - initialLength, pressureArea );
		tlDenominator->setText( chrText );
		momentaryElasticity = ( force.vpModule() * initialLength ) / ( fabs(currentLength - initialLength) * pressureArea );
		sprintf( chrText, "%6.1f", momentaryElasticity );
		tlElasticity->setText( chrText );
	
		if( cbConfigure->isChecked() ){
	
			double increment = 1.0;
			/*// recalculate hooks
			double targetDistance = initialLength + ( ( force.vpModule() * initialLength ) / ( selectedMaterial->getYoungsModulus() * pressureArea ) );
			if( targetDistance > currentLength ){
				if(( previousLenght >= currentLength ) ){
					increment = 1.0 - fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					printf("inc: %f ", increment );
				}
			} else if( targetDistance < currentLength ){
				if( ( previousLenght >= currentLength ) ){
					increment = 1.0 + fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					printf("inc: %f ", increment );
				}
			}*/
			
			double targetDistance = initialLength + ( ( force.vpModule() * initialLength ) / ( selectedMaterial->getYoungsModulus() * pressureArea ) );
			if( fabs( targetDistance - previousLenght ) < fabs( targetDistance - currentLength ) ){ // if NOT approaching
			
				COME_Vector3D direction( molecMobileSelected->getPosition() - molecFixSelected->getPosition() );
				if( ( force.vpDotProduct( direction ) / ( force.vpModule() * direction.vpModule() ) ) >= 0.0 ){ // if force is tensioning
				
					if( targetDistance < currentLength ){ // bigger
						increment = 1.0 + fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					} else { // smaller
						increment = 1.0 - fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					}
				
				} else { // if force is compressing
				
					if( targetDistance < currentLength ){ // bigger
						increment = 1.0 - fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					} else { // smaller
						increment = 1.0 + fabs( fabs(targetDistance) - fabs(currentLength))*atof( lePrecision->text() );
					}
				
				}
			}
			
			previousLenght = currentLength;
	
			list<COME_BioStructure*>::const_iterator iterOrgans;
			for( iterOrgans = ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->begin(), iterOrgans++, iterOrgans++; iterOrgans != ((list<COME_BioStructure *>*)scene->getPatient(0)->getPtOrganList())->end(); iterOrgans++  ){
			
				double newGlobalHooke = ((COME_MoleculesCartilage*)(*iterOrgans))->getGlobalHooke() * increment;
				((COME_MoleculesCartilage*)(*iterOrgans))->setGlobalHooke( newGlobalHooke );
				COME_Vector3D aniso = selectedMaterial->getAnisotropyVector();
			
				list<COME_Molecule*>* molecules = ((COME_MoleculesCartilage*)(*iterOrgans))->getTissue()->getShape();
				list<COME_Molecule*>::iterator iterMolecules;
				for ( iterMolecules = molecules->begin(); iterMolecules != molecules->end(); iterMolecules++){
	
					list<COME_MoleculeLink*>::iterator iterLink;
					for (iterLink = (*iterMolecules)->getConnectionList()->begin(); iterLink != (*iterMolecules)->getConnectionList()->end(); iterLink++ ){
						COME_Vector3D kVec = (*iterLink)->getElement(0)->getPosition() - (*iterLink)->getElement(1)->getPosition();
						kVec.vpNormalize();
						double kThisLink = ( ( fabs(kVec.getX()) * aniso.getX() ) + ( fabs(kVec.getY()) * aniso.getY() ) + ( fabs(kVec.getZ()) * aniso.getZ() ) )* newGlobalHooke;
						(*iterLink)->setHookeConst( kThisLink );
					}
				}
			}	
		}
	}
}

void
COME_Mechatester::changedNumIntegration(){

	char chrType[60];
	switch( cbNumIntegration->currentItem() ){
		case 0: COME::flagNumIntegration = EULER; strcpy(chrType,"Numerical integration method is now Euler.");break;
		case 1: COME::flagNumIntegration = RUNGE_KUTTA4; strcpy(chrType,"Numerical integration method is now Runge-Kutta 4."); break;
	}
	QString msg = chrType;
	statusBar()->message( msg, 2000 );
}

void
COME_Mechatester::hourglass( bool yesno ){

	if( yesno ){
		setCursor( Qt::WaitCursor );
		openglArea->setCursor( Qt::WaitCursor );
	} else {
		setCursor( Qt::ArrowCursor );
		openglArea->setCursor( Qt::ArrowCursor );
	}
}
